/**
 * @file
 * Ethernet Interface Skeleton
 *
 */

/*
 * Copyright (c) 2001-2004 Swedish Institute of Computer Science.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * This file is part of the lwIP TCP/IP stack.
 *
 * Author: Adam Dunkels <adam@sics.se>
 *
 */

/*
 * This file is a skeleton for developing Ethernet network interface
 * drivers for lwIP. Add code to the low_level functions and do a
 * search-and-replace for the word "ethernetif" to replace it with
 * something that better describes your network interface.
 */

#include "lwip/opt.h"

#if 1 /* ready */

#include "lwip/def.h"
#include "lwip/mem.h"
#include "lwip/pbuf.h"
#include "lwip/sys.h"
#include <lwip/stats.h>
#include <lwip/snmp.h>
#include "netif/etharp.h"
#include "netif/ppp_oe.h"

/* Kernel includes. */
#include "FreeRTOS.h"
#include "semphr.h"
#include "task.h"

/* Standard library includes. */
#include <string.h>

#include "macqueue.h"

#include "TM_FlashManager.h"
#include "AT_Config.h"

extern unsigned char cw1200_mac_template[ETH_ALEN];
extern enum cw1200_running_status g_RunningState;

/* MAC address configuration. */
#define configMAC_ADDR0	0x00
#define configMAC_ADDR1	0xCF
#define configMAC_ADDR2	0x52
#define configMAC_ADDR3	0x35
#define configMAC_ADDR4	0x00
#define configMAC_ADDR5	0x01

/* Define those to better describe your network interface. */
#define IFNAME0 'p'
#define IFNAME1 '2'

/* PHY hardware specifics. */
#define PHY_STATUS								( 0x1F )
#define PHY_DUPLEX_STATUS							( 4<<2 )
#define PHY_SPEED_STATUS							( 1<<2 )

/* Delay to wait for a DMA buffer to become available if one is not already
available. */
#define netifBUFFER_WAIT_ATTEMPTS					        10
#define netifBUFFER_WAIT_DELAY						        (10 / portTICK_RATE_MS)

/* Delay between polling the PHY to see if a link has been established. */
#define netifLINK_DELAY								( 500 / portTICK_RATE_MS )

/* Delay between looking for incoming packets.  In ideal world this would be
infinite. */
#define netifBLOCK_TIME_WAITING_FOR_INPUT			                netifLINK_DELAY

/* Very short delay to use when waiting for the Tx to finish with a buffer if
we run out of Rx buffers. */
#define enetMINIMAL_DELAY							( 2 / portTICK_RATE_MS )

/* The DMA buffers.  These are char arrays to allow them to be alligned correctly. */
static unsigned portBASE_TYPE uxNextRxBuffer = 0, uxNextTxBuffer = 0;

/* Semaphores used by the ENET interrupt handler to wake the handler task. */
static xSemaphoreHandle xRxENETSemaphore;
static xSemaphoreHandle xTxENETSemaphore;

extern int periph_clk_khz;

xTaskHandle xEthIntTask = NULL;

/**
 * Helper struct to hold private data used to operate your ethernet interface.
 * Keeping the ethernet address of the MAC in this struct is not necessary
 * as it is already kept in the struct netif.
 * But this is only an example, anyway...
 */
struct ethernetif {
  struct eth_addr *ethaddr;
  /* Add whatever per-interface state that is needed here. */
};

/* Standard lwIP netif handlers. */
static void prvInitialiseENETBuffers( void );
static void low_level_init( struct netif *netif );
static err_t low_level_output(struct netif *netif, struct pbuf *p);
static struct pbuf *low_level_input(struct netif *netif);

/* Forward declarations. */
static void  ethernetif_input(/*FSL:struct netif *netif*/void *pParams);

/**
 * In this function, the hardware should be initialized.
 * Called from ethernetif_init().
 *
 * @param netif the already initialized lwip network interface structure
 *        for this ethernetif
 */
static void
low_level_init(struct netif *netif)
{
/*unsigned portLONG*/ int usData;
const unsigned portCHAR ucMACAddress[6] =
{
  configMAC_ADDR0, configMAC_ADDR1,configMAC_ADDR2,configMAC_ADDR3,configMAC_ADDR4,configMAC_ADDR5
};

//FSL:struct ethernetif *ethernetif = netif->state;

  /* set MAC hardware address length */
  netif->hwaddr_len = ETHARP_HWADDR_LEN;

  /* set MAC hardware address */
#if 0 //Tom
  netif->hwaddr[0] = configMAC_ADDR0;
  netif->hwaddr[1] = configMAC_ADDR1;
  netif->hwaddr[2] = configMAC_ADDR2;
  netif->hwaddr[3] = configMAC_ADDR3;
  netif->hwaddr[4] = configMAC_ADDR4;
  netif->hwaddr[5] = configMAC_ADDR5;
#endif

  memcpy( &netif->hwaddr[0], &cw1200_mac_template[0], ETH_ALEN );
  /* maximum transfer unit */
  netif->mtu = configENET_MAX_PACKET_SIZE-20;

  /* device capabilities */
  /* don't set NETIF_FLAG_ETHARP if this device is not an ethernet one */
  netif->flags = NETIF_FLAG_BROADCAST | NETIF_FLAG_ETHARP | NETIF_FLAG_LINK_UP|NETIF_FLAG_IGMP;


  /* Do whatever else is needed to initialize interface. */

//wxg need talk with mac80211
  /* Can we talk to the PHY? */
  do
  {
    vTaskDelay( netifLINK_DELAY );
    //mii_read( 0, configPHY_ADDRESS, PHY_PHYIDR1, &usData );

  } while( g_RunningState != CW1200_RUNNING_STATUS_CONNECTTED );

  /* Start auto negotiate. */
  //mii_write( 0, configPHY_ADDRESS, PHY_BMCR, ( PHY_BMCR_AN_RESTART | PHY_BMCR_AN_ENABLE ) );


  /* When we get here we have a link - find out what has been negotiated. */
  usData = 0;
 // mii_read( 0, configPHY_ADDRESS, PHY_STATUS, &usData );



  /* Create the task that handles the MAC ENET. */
  if(xEthIntTask == NULL)
  	xTaskCreate( ethernetif_input, "ETH_INT     ", configETHERNET_INPUT_TASK_STACK_SIZE, (void *)netif, configETHERNET_INPUT_TASK_PRIORITY, &xEthIntTask );

}

/**
 * This function should do the actual transmission of the packet. The packet is
 * contained in the pbuf that is passed to the function. This pbuf
 * might be chained.
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @param p the MAC packet to send (e.g. IP packet including MAC addresses and type)
 * @return ERR_OK if the packet could be sent
 *         an err_t value if the packet couldn't be sent
 *
 * @note Returning ERR_MEM here if a DMA queue of your MAC is full can lead to
 *       strange results. You might consider waiting for space in the DMA queue
 *       to become availale since the stack doesn't retry to send a packet
 *       dropped because of memory failure (except for the TCP timers).
 */
extern netdev_tx_t ieee80211_subif_start_xmit(void *buff, unsigned short length, unsigned int flag,int mark);
#define LWIP_OUTPUT_BUF_SIZE (1536)
unsigned char lwip_output_buf[LWIP_OUTPUT_BUF_SIZE];

static err_t
low_level_output(struct netif *netif, struct pbuf *p)
{
 struct ethernetif *ethernetif = netif->state;
  struct pbuf *q;
    unsigned int pos = 0;
  //initiate transfer();

#if ETH_PAD_SIZE
  pbuf_header(p, -ETH_PAD_SIZE); /* drop the padding word */
#endif

  for(q = p; q != NULL; q = q->next) {
    /* Send the data from the pbuf to the interface, one pbuf at a
       time. The size of the data in each pbuf is kept in the ->len
       variable. */
    //send data from(q->payload, q->len);
        if ( (pos + q->len) < LWIP_OUTPUT_BUF_SIZE )
        {
            memcpy(&lwip_output_buf[pos], q->payload, q->len);
            pos += q->len;
        }
        else
        {
            /* the lwip output buff is too small*/
			printu("the lwip output buff is too small");
            break;
        }
  }
  //printu("low_level_output pos = %d\r\n",pos);
  if(pos > 1514)
  {
  	printu("low_level_output pos = %d\r\n",pos);
	while(1);
  }
  ieee80211_subif_start_xmit(lwip_output_buf, pos,IEEE80211_STYPE_QOS_DATA,1, netif->if_id);//IEEE80211_STYPE_QOS_DATA IEEE80211_STYPE_DATA

  //signal that packet should be sent();

#if ETH_PAD_SIZE
  pbuf_header(p, ETH_PAD_SIZE); /* reclaim the padding word */
#endif

  //LINK_STATS_INC(link.xmit);

  return ERR_OK;
}

//extern struct mac_input_data MacReceiveBuff;

static struct pbuf *
low_level_input(struct netif *netif)
{
  struct ethernetif *ethernetif = netif->state;
  struct pbuf *p = NULL, *q;
  u16_t len = 0;
  u16_t cpylen = 0;
  /* Obtain the size of the packet and put it into the "len"
     variable. */
  //len = ;
  len = macrxQ_reclen();
  if (len==0) {
      return NULL;
  }

#if ETH_PAD_SIZE
  len += ETH_PAD_SIZE; /* allow room for Ethernet padding */
#endif

  /* We allocate a pbuf chain of pbufs from the pool. */
  p = pbuf_alloc(PBUF_RAW, len, PBUF_POOL);

  if (p != NULL) {

#if ETH_PAD_SIZE
    pbuf_header(p, -ETH_PAD_SIZE); /* drop the padding word */
#endif

    /* We iterate over the pbuf chain until we have read the entire
     * packet into the pbuf. */

    /* Read enough bytes to fill this pbuf in the chain. The
       * available data in the pbuf is given by the q->len
       * variable.
       * This does not necessarily have to be a memcpy, you can also preallocate
       * pbufs for a DMA-enabled MAC and after receiving truncate it to the
       * actually received size. In this case, ensure the tot_len member of the
       * pbuf is the sum of the chained pbuf len members.
       */

        for (q = p; q != NULL; q = q->next)
        {
            /* Read enough bytes to fill this pbuf in the chain. The
            * available data in the pbuf is given by the q->len
            * variable. */
            cpylen = (q->len >= len) ? len : q->len;
            macrxQ_getrec(q->payload, cpylen);
            len -= cpylen;
        }  // read data into(q->payload, q->len);

    //acknowledge that packet has been read();

#if ETH_PAD_SIZE
    pbuf_header(p, ETH_PAD_SIZE); /* reclaim the padding word */
#endif

    LINK_STATS_INC(link.recv);
  } else {
    //drop packet();
    printb("%s lwip no mem\r\n",__func__);
    LINK_STATS_INC(link.memerr);
    LINK_STATS_INC(link.drop);

#ifdef DEBUG_RESET_ON_NO_MEM
    ResetModule();
#endif
  }
  //xSemaphoreGive(xMAC_recv_lock);

  return p;
}
/**
 * This function should be called when a packet is ready to be read
 * from the interface. It uses the function low_level_input() that
 * should handle the actual reception of bytes from the network
 * interface. Then the type of the received packet is determined and
 * the appropriate input function is called.
 *
 * @param netif the lwip network interface structure for this ethernetif
 */
static void
ethernetif_input(void *pParams)
{
//FSL:  struct ethernetif *ethernetif;
  struct netif *netif;
  struct eth_hdr *ethhdr;
  struct pbuf *p;

//FSL:  ethernetif = netif->state;
  netif = (struct netif*) pParams;

  for( ;; )
  {
    do
    {
      /* move received packet into a new pbuf */
      p = low_level_input(netif);
      /* no packet could be read, silently ignore this */
      if (p == NULL)
      {
	/* No packet could be read.  Wait a for an interrupt to tell us
	   there is more data available. */
	//xSemaphoreTake( xRxENETSemaphore, /*netifBLOCK_TIME_WAITING_FOR_INPUT*/portMAX_DELAY );
            vTaskDelay(1);
      }
    }while( p == NULL );
    /* points to packet payload, which starts with an Ethernet header */
    ethhdr = p->payload;
	//printu("ethernetif_input type 0x%x\n\r\n\r",htons(ethhdr->type));
    switch (htons(ethhdr->type))
    {
      /* IP or ARP packet? */
      case ETHTYPE_IP:
      case ETHTYPE_ARP:
    #if PPPOE_SUPPORT
      /* PPPoE packet? */
      case ETHTYPE_PPPOEDISC:
      case ETHTYPE_PPPOE:
    #endif /* PPPOE_SUPPORT */
        /* full packet send to tcpip_thread to process */
        if (netif->input(p, netif)!=ERR_OK)
         { LWIP_DEBUGF(NETIF_DEBUG, ("ethernetif_input: IP input error\n"));
           pbuf_free(p);
           p = NULL;
         }
        break;

      default:
        pbuf_free(p);
        p = NULL;
        break;
    }
  }
}

/**
 * Should be called at the beginning of the program to set up the
 * network interface. It calls the function low_level_init() to do the
 * actual setup of the hardware.
 *
 * This function should be passed as a parameter to netif_add().
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @return ERR_OK if the loopif is initialized
 *         ERR_MEM if private data couldn't be allocated
 *         any other err_t on error
 */
err_t ethernetif_init(struct netif *netif);

err_t
ethernetif_init(struct netif *netif)
{
  struct ethernetif *ethernetif;

  LWIP_ASSERT("netif != NULL", (netif != NULL));

  ethernetif = mem_malloc(sizeof(struct ethernetif));

  if (ethernetif == NULL) {
    LWIP_DEBUGF(NETIF_DEBUG, ("ethernetif_init: out of memory\n"));
    return ERR_MEM;
  }

#if LWIP_NETIF_HOSTNAME
  /* Initialize interface hostname */
  //netif->hostname = "lwip";
  netif->hostname = os_malloc(16);
  strncpy(netif->hostname, TM_GetFlashReadPtr(VAR_ID_DeviceName), 16);
  netif->hostname[15] = 0;

#endif /* LWIP_NETIF_HOSTNAME */

  /*
   * Initialize the snmp variables and counters inside the struct netif.
   * The last argument should be replaced with your link speed, in units
   * of bits per second.
   */
  NETIF_INIT_SNMP(netif, snmp_ifType_ethernet_csmacd, LINK_SPEED_OF_YOUR_NETIF_IN_BPS);

  netif->state = ethernetif;
  netif->name[0] = IFNAME0;
  netif->name[1] = IFNAME1;
  /* We directly use etharp_output() here to save a function call.
   * You can instead declare your own function an call etharp_output()
   * from it if you have to do some checks before sending (e.g. if link
   * is available...) */
  netif->output = etharp_output;
  netif->linkoutput = low_level_output;

  ethernetif->ethaddr = (struct eth_addr *)&(netif->hwaddr[0]);

  /* initialize the hardware */
  low_level_init(netif);

  return ERR_OK;
}

#endif /* 0 */
