/*********************************************************************************************************
*                                        CODEC WM8904 driver
*                             COPYRIGHT(c) 2017 Amped RF Technology (ART)
*                                        All Rights Reserved
*
* File		: wm8904.h
* Version	: V1.0.0
* Brief:    : This file contains all the functions prototypes for the WM8904 driver.  
* ---------------
**********************************************************************************************************/
#ifndef _WM8904_H
#define _WM8904_H

#include <stdint.h>
#include "wm8904_conf.h"

#ifdef __cplusplus
    extern "C" {
#endif

//------------------------------------------------------------------
// Writing to this register will reset all registers to their default value.
// Reading from this register will get the device ID 0x8904.
//------------------------------------------------------------------
#define WM8904_SWRST_ID_REG_ADDR        (0x00)
//Software reset and ID register structure type definition.
typedef union _wm8904SwRstIdReg_t{
    uint16_t regVal;
    uint16_t SW_RST_DEV_ID;
    struct {
        uint16_t regVal_L : 8;
        uint16_t regVal_H : 8;
    } bits;
} wm8904SwRstIdReg_t;

//------------------------------------------------------------------
// Bits[3:2]:Master bias control.
// Bits[0]  :Enables the normal bias current generator.
//------------------------------------------------------------------
#define WM8904_BIAS_CTRL0_REG_ADDR      (0x04)
//Master bias control options.
typedef enum {
    LOW_PWR_BIAS = 0,
    HIGH_PERFORM_BIAS = 2
} WM8904_BIAS_ISEL_t;
//Bias control0 register structure type definition.
typedef union _wm8904BiasCtrl0Reg_t{
    uint16_t regVal;
    struct {
        FunctionalState BIAS_ENA : 1;//Enables the normal bias current generator.
        uint16_t reserved0       : 1;
        WM8904_BIAS_ISEL_t ISEL  : 2;//Master bias control. Default value is 2.
        uint16_t reserved1       : 12;
    } bits;
} wm8904BiasCtrl0Reg_t;

//------------------------------------------------------------------
// Bits[6]  :Enables VMID buffer to unused inputs/outputs.
// Bits[2:1]:VMID divider enable and select.
// Bits[0]  :Enables VMID buffer.
//------------------------------------------------------------------
#define WM8904_VMID_CTRL0_REG_ADDR      (0x05)
//VMID divider options.
typedef enum {
    VMID_DISABLE = 0,       //for OFF mode.
    VMID_DUAL50K_DIV = 1,   //for normal operation.
    VMID_DUAL250K_DIV = 2,  //for low power standby.
    VMID_DUAL5K_DIV = 3     //for fast start up.
} VMID_DIV_SEL_t;
//VMID control0 register structure type definition.
typedef union _wm8904VmidCtrl0Reg_t {
    uint16_t regVal;
    struct {
        FunctionalState VMID_ENA     : 1;//Enable VMID buffer.
        VMID_DIV_SEL_t  VMID_RES     : 2;//VMID divider enable and select.
        uint16_t reserved0           : 3;
        FunctionalState VMID_BUF_ENA : 1;//Enables VMID buffer to unused inputs/outputs.
        uint16_t reserved1           : 9;
    } bits;
} wm8904VmidCtrl0Reg_t;

//------------------------------------------------------------------
// Bits[6:4]:Mic bias current detect threshold(AVdd = 1.8V).
// Bits[3:2]:Mic bias short circuit threshold(AVdd = 1.8V).
// Bits[1]  :Mic bias current and short circuit detect enable.
// Bits[0]  :Mic bias enable.
//------------------------------------------------------------------
#define WM8904_MICBIAS_CTRL0_REG_ADDR   (0x06)
//Mic bias current detect threshold(AVdd = 1.8V) options.
typedef enum {
    THR_0_07MA = 0,
    THR_0_26MA = 1,
    THR_0_45MA = 2,
    THR_0_64MA = 3,
    THR_0_83MA = 4,
    THR_1_02MA = 5,
    THR_1_21MA = 6,
    THR_1_40MA = 7
} WM8904_MICDETTHR_t;
//Mic bias short circuit threshold(AVdd = 1.8V) options.
typedef enum {
    THR_0_52MA = 0,
    THR_0_88MA = 1,
    THR_1_24MA = 2,
    THR_1_60MA = 3
} WM8904_MICSHORTTHR_t;
//Mic bias voltage control0 register structure type definition.
typedef union _wm8904MicBiasCtrl0Reg_t {
    uint16_t regVal;
    struct {
        FunctionalState MICBIAS_ENA       : 1;//Mic bias enable.
        FunctionalState MICDET_ENA        : 1;//Mic bias current and short circuit detect enable.
        WM8904_MICSHORTTHR_t MICSHORT_THR : 2;//Mic bias short circuit threshold(AVdd = 1.8V).
        WM8904_MICDETTHR_t   MICDET_THR   : 3;//Mic bias current detect threshold(AVdd = 1.8V).
        uint16_t reserved                 : 9;
    } bits;
} wm8904MicBiasCtrl0Reg_t;

//------------------------------------------------------------------
// Bits[2:0]:Selects Mic bias voltage.
//------------------------------------------------------------------
#define WM8904_MICBIAS_CTRL1_REG_ADDR   (0x07)
//Mic bias voltage options
typedef enum {
    AVDD_1_6V = 0,
    AVDD_2_0V = 1,
    AVDD_2_1V = 2,
    AVDD_2_4V = 3,
    AVDD_2_7V = 4
} WM8904_MICBIAS_t;
//Mic bias voltage control1 register structure type definition.
typedef union _wm8904MicBiasCtrl1Reg_t {
    uint16_t regVal;
    struct {
        WM8904_MICBIAS_t MICBIAS_SEL : 3;//Selects Mic bias voltage.
        uint16_t reserved            : 13;
    } bits;
} wm8904MicBiasCtrl1Reg_t;

//------------------------------------------------------------------
// Bits[0]:Set ADC over sampling ratio
//------------------------------------------------------------------
#define WM8904_ADC0_REG_ADDR            (0x0A)
//ADC over sampling ratio options.
typedef enum {
    LOW_PWR = 0,//64 x fs.
    HIGH_PERFORM = 1//128 x fs.
} WM8904_ADCOSR_t;
//Analogue ADC 0 register structure type definition.
typedef union _wm8904Adc0Reg_t {
    uint16_t regVal;
    struct {
        WM8904_ADCOSR_t ADC_OSR128 : 1;//Set ADC over sampling ratio.
        uint16_t reserved          : 15;
    } bits;
} wm8904AdcReg_t;

//------------------------------------------------------------------
// Bits[1]:Left input PGA enable.
// Bits[0]:Right input PGA enable.
//------------------------------------------------------------------
#define WM8904_PWRMANAGE0_REG_ADDR       (0x0C)
//Power management0 register structure type definition.
typedef union _wm8904PwrManage0Reg_t {
    uint16_t regVal;
    struct {
        FunctionalState INR_ENA : 1;//Right input PGA enable.
        FunctionalState INL_ENA : 1;//Left input PAG enable.
        uint16_t reserved       : 14;
    } bits;
} wm8904PwrManage0Reg_t;

//------------------------------------------------------------------
// Bits[1]:Left headphone output enable.
// Bits[0]:Right headphone output enable.
//------------------------------------------------------------------
#define WM8904_PWRMANAGE2_REG_ADDR      (0x0E)
//Power management2 register structure type definition.
typedef union _wm8904PwrManage2Reg_t {
    uint16_t regVal;
    struct {
        FunctionalState HPR_PGA_ENA : 1;//Right headphone output enable.
        FunctionalState HPL_PGA_ENA : 1;//Left headphone output enable.
        uint16_t reserved           : 14;
    } bits;
} wm8904PwrManage2Reg_t;

//------------------------------------------------------------------
// Bits[1]:Left line output enable.
// Bits[0]:Right line output enable.
//------------------------------------------------------------------
#define WM8904_PWRMANAGE3_REG_ADDR      (0x0F)
//Power management3 register structure type definition.
typedef union _wm8904PwrManage3Reg_t {
    uint16_t regVal;
    struct {
        FunctionalState LINEOUTR_PGA_ENA : 1;//Right line output enable.
        FunctionalState LINEOUTL_PGA_ENA : 1;//Left line output enable.
        uint16_t reserved                : 14;
    } bits;
} wm8904PwrManage3Reg_t;

//------------------------------------------------------------------
// Bits[3]:Left DAC enable.
// Bits[2]:Right DAC enable.
// Bits[1]:Left ADC enable.
// Bits[0]:Right ADC enable.
//------------------------------------------------------------------
#define WM8904_PWRMANAGE6_REG_ADDR      (0x12)
//Power management6 register structure type definition.
typedef union _wm8904PwrManage6Reg_t {
    uint16_t regVal;
    struct {
        FunctionalState ADCR_ENA : 1;//Right ADC enable.
        FunctionalState ADCL_ENA : 1;//Left ADC enable.
        FunctionalState DACR_ENA : 1;//Right DAC enable.
        FunctionalState DACL_ENA : 1;//Left DAC enable.
        uint16_t reserved        : 12;
    } bits;
} wm8904PwrManage6Reg_t;

//------------------------------------------------------------------
// Bits[14]:TOCLK rate divider.
// Bits[13]:TOCLK rate multiplier.
// Bits[0] :Enable divide by 2 on MCLK.
//------------------------------------------------------------------
#define WM8904_CLKRATE0_REG_ADDR        (0x14)
//TOCLK rate divider options(/16) options.
typedef enum {
    TOCLK16_DIV1 = 0,//f / 1
    TOCLK16_DIV16 = 1//f / 16
} WM8904_TOCLKDIV16_t;
//TOCLK rate multiplier opotions.
typedef enum {
    TOCLK_Fx1 = 0, //f x 1
    TOCLK_Fx4 = 1, //f x 4
} WM8904_TOCLKMULTI_t;
//MCLK divide by 2 options.
typedef enum {
    MCLK_DIV1 = 0, //SYSCLK = MCLK
    MCLK_DIV2 = 1  //SYSCLK = MCLK / 2
} WM8904_MCLKDIV_t;
//Clock rates0 register structure type definition.
typedef union _wm8904ClkRate0Reg_t {
    uint16_t regVal;
    struct {
        WM8904_MCLKDIV_t MCLK_DIV            : 1; //Enable divide by 2 on MCLK.
        uint16_t reserved0                   : 12;
        WM8904_TOCLKMULTI_t TOCLK_RATE_X4    : 1; //TOCLK rate multiplier.
        WM8904_TOCLKDIV16_t TOCLK_RATE_DIV16 : 1; //TOCLK rate divider(/16).
        uint16_t reserved1                   : 1;
    } bits;
} wm8904ClkRate0Reg_t;

//------------------------------------------------------------------
// Bits[13:10]:Selects the SYSCLK / fs ratio.
// Bits[2:0]  :Selects the sample rate(fs).
//------------------------------------------------------------------
#define WM8904_CLKRATE1_REG_ADDR        (0x15)
//SYSCLK / fs ratio options.
typedef enum {
    SYSCLKRATE64   = 0,
    SYSCLKRATE128  = 1,
    SYSCLKRATE192  = 2,
    SYSCLKRATE256  = 3,
    SYSCLKRATE384  = 4,
    SYSCLKRATE512  = 5,
    SYSCLKRATE768  = 6,
    SYSCLKRATE1024 = 7,
    SYSCLKRATE1408 = 8,
    SYSCLKRATE1536 = 9
} WM8904_SYSCLKRATE_t;
//Sample rate(fs) options.
typedef enum {
    SAMPLERATE_8KHZ  = 0,
    SAMPLERATE_12KHZ = 1,
    SAMPLERATE_16KHZ = 2,
    SAMPLERATE_24KHZ = 3,
    SAMPLERATE_32KHZ = 4,
    SAMPLERATE_48KHZ = 5
} WM8904_SAMPLERATE_t;
//Clock rates1 register structure type definition.
typedef union _wm8904ClkRate1Reg_t {
    uint16_t regVal;
    struct {
        WM8904_SAMPLERATE_t SAMPLE_RATE  : 3;//Selects the sample rate(fs).
        uint16_t reserved0               : 7;
        WM8904_SYSCLKRATE_t CLK_SYS_RATE : 4;//Selects the SYSCLK / fs ratio.
        uint16_t reserved1               : 2;
    } bits;
} wm8904ClkRate1Reg_t;

//------------------------------------------------------------------
// Bits[15]:MCLK invert.
// Bits[14]:SYSCLK source select.
// Bits[12]:TOCLK rate divider(/2).
// Bits[3]:GPIO clock output enable.
// Bits[2]:System clock enable.
// Bits[1]:DSP clock enable.
// Bits[0]:Zero cross timeout enable.
//------------------------------------------------------------------
#define WM8904_CLKRATE2_REG_ADDR        (0x16)
//MCLK invert options.
typedef enum {
    MCLK_NOTINVERTED = 0,
    MCLK_INVERTED = 1
} WM8904_MCLKINV_t;
//System clock source select options.
typedef enum {
    SYSCLK_MCLK = 0,
    SYSCLK_FLL = 1
} WM8904_SYSCLKSRC_t;
//TOCLK rate devider options.
typedef enum {
    TOCLK2_DIV2 = 0,
    TOCLK2_DIV1 = 1
} WM8904_TOCLKDIV2_t;
//Clock rates2 register structure type definition.
typedef union _wm8904ClkRate2Reg_t {
    uint16_t regVal;
    struct {
        FunctionalState TOCLK_ENA     : 1;//Zero cross timeout enable.
        FunctionalState CLK_DSP_ENA   : 1;//DSP clock enable.
        FunctionalState CLK_SYS_ENA   : 1;//System clock enable.
        FunctionalState OPCLK_ENA     : 1;//GPIO clock output enable.
        uint16_t reserved0            : 8;
        WM8904_TOCLKDIV2_t TOCLK_RATE : 1;//TOCLK rate divider(/2)
        uint16_t reserved1            : 1;
        WM8904_SYSCLKSRC_t SYSCLK_SRC : 1;//SYSCLK source select.
        WM8904_MCLKINV_t MCLK_INV     : 1;//MCLK invert.
    } bits;
} wm8904ClkRate2Reg_t;

//------------------------------------------------------------------
// Bits[12]  :Left DAC invert.
// Bits[11]  :Right DAC invert.
// Bits[10:9]:DAC digital input volume boost.
// Bits[8]   :Digital loopback function.
// Bits[7]   :Left digital audio channel source.
// Bits[6]   :Right digital audio channel source.
// Bits[5]   :Left DAC data source select.
// Bits[4]   :Right DAC data source select.
// Bits[3]   :ADC companding enable.
// Bits[2]   :ADC companding type.
// Bits[1]   :DAC companding enable.
// Bits[0]   :DAC companding type.
//------------------------------------------------------------------
#define WM8904_AUDIOINTERFACE0_REG_ADDR (0x18)
//DAC invert options.
typedef enum {
    DACOUT_NOTINVERT = 0,
    DACOUT_INVERT = 1
} WM8904_DACINV_t;
//DAC digital input volume boost options.
typedef enum {
    DAC_BOOST_0DB = 0,
    DAC_BOOST_6DB = 1, //Input data must not exceed -6dBFS.
    DAC_BOOST_12DB = 2,//Input data must not exceed -12dBFS.
    DAC_BOOST_18DB = 3 //Input data must not exceed -18dBFS.
} WM8904_DACINBOOST_t;
//Digital audio channel source options.
typedef enum {
    LT_ADCDATA = 0,
    RT_ADCDATA = 1
} WM8904_AUDIOSRC_t;
//DAC data source select options.
typedef enum {
    LT_CHANNEL = 0,
    RT_CHANNEL = 1
} WM8904_DACOUTSRC_t;
//ADC/DAC companding type options.
typedef enum {
    U_LAW = 0,
    A_LAW = 1
} WM8904_COMPANDINGTYPE_t;
//Audio interface0 register structure type definition.
typedef union _wm8904AudioInterface0Reg_t {
    uint16_t regVal;
    struct {
        WM8904_COMPANDINGTYPE_t DAC_COMPMODE : 1;//DAC companding type.
        FunctionalState DAC_COMP             : 1;//DAC companding enable.
        WM8904_COMPANDINGTYPE_t ADC_COMPMODE : 1;//ADC companding type.
        FunctionalState ADC_COMP             : 1;//ADC companding enable.
        WM8904_DACOUTSRC_t AIFDACR_SRC       : 1;//Right DAC data source select.
        WM8904_DACOUTSRC_t AIFDACL_SRC       : 1;//Left DAC data source select.
        WM8904_AUDIOSRC_t AIFADCR_SRC        : 1;//Right digital audio channel source.
        WM8904_AUDIOSRC_t AIFADCL_SRC        : 1;//Left digital audio channel source.
        FunctionalState LOOPBACK             : 1;//Digital loopback function enable.
        WM8904_DACINBOOST_t DAC_BOOST        : 2;//DAC digital input volume boost.
        WM8904_DACINV_t DACR_DATINV          : 1;//Right DAC invert.
        WM8904_DACINV_t DACL_DATINV          : 1;//Left DAC invert.
        uint16_t reseved                     : 3;
    } bits;
} wm8904AudioInterface0Reg_t;

//------------------------------------------------------------------
// Bits[13] :DAC TDM enable.
// Bits[12] :DACDAT TDM channel select.
// Bits[11] :ADC TDM enable.
// Bits[10] :ADCDAT TDM channel select.
// Bits[8]  :Audio interface tristate.
// Bits[7]  :BCLK invert.
// Bits[6]  :Audio interface BCLK direction.
// Bits[4]  :LRC Polarity / DSP Mode A-B select.
//           LRC Polarity:Right, left and I2S modes invert.
//           DSP Mode:Mode A-B select.
// Bits[3:2]:Digital audio interface word length.
// Bits[1:0]:Digital audio interface format.
//------------------------------------------------------------------
#define WM8904_AUDIOINTERFACE1_REG_ADDR (0x19)
//TDM enable options.
typedef enum {
    DAT_NORMAL = 0,//Normal DAT operation.
    DAT_TDM = 1    //TDM enabled.
} WM8904_TDM_t;
//TDM channel select options.
typedef enum {
    DAT_ONSLOT0 = 0,
    DAT_ONSLOT1 = 1
} WM8904_TDMCHANNEL_t;
//Audio interface tristate options.
typedef enum {
    PINS_NORMAL = 0, //Audio interface pins operate normally.
    PINS_TRISTATE = 1//Tristate all audio interface pins.
} WM8904_TRISTATE_t;
//BCLK invert options.
typedef enum {
    BCLK_NOTINVERT = 0,
    BCLK_INVERT = 1
} WM8904_BCLKINV_t;
//Audio interface BCLK direction options.
typedef enum {
    BCLK_INPUT = 0,
    BCLK_OUTPUT = 1
} WM8904_BCLKDIR_t;
//LRC polarity options.
typedef enum {
    LRC_NOTINVERT = 0,
    LRC_INVERT = 1
} WM8904_LRCLKINV_t;
//DSP mode A-B select options.
typedef enum {
    DSP_MODE_A = 0,//MSB is available on 2nd BCLK rising edge after LRC rising edge.
    DSP_MODE_B = 1 //MSB is available on 1st BCLK rising edge after LRC rising edge.
} WM8904_DSPMODE_t;
//Digital audio interface word length options.
typedef enum {
    WL_16BITS = 0,
    WL_20BITS = 1,
    WL_24BITS = 2,
    WL_32BITS = 3
} WM8904_AUDIOWORDLENGTH_t;
//Digital audio interface format options.
typedef enum {
    RIGHT_JUSTIFIED = 0,
    LEFT_JUSTIFIED = 1,
    FORMAT_I2S = 2,
    FORMAT_DSP = 3
} WM8904_AUDIOFORMAT_t;
//Audio interface1 register structure type definition.
typedef union _wm8904AudioInterface1Reg_t {
    uint16_t regVal;
    struct {
        uint16_t reserved                   : 4;
        WM8904_LRCLKINV_t LRCLK_INV         : 1;//LRCLK invert.
    } AIF_LRCLK_INV;
    struct {
        uint16_t reserved                   : 4;
        WM8904_DSPMODE_t DSP_MODE           : 1;//DSP mode.
    } DSP_MODE_SET;
    struct {
        WM8904_AUDIOFORMAT_t AIF_FMT        : 2;//Digital audio interface format.
        WM8904_AUDIOWORDLENGTH_t AIF_WL     : 2;//Digital audio interface word length.
        uint16_t reserved0                  : 2;
        WM8904_BCLKDIR_t BCLK_DIR           : 1;//Audio interface BCLK direction.
        WM8904_BCLKINV_t AIF_BCLK_INV       : 1;//BCLK invert.
        WM8904_TRISTATE_t AIF_TRIS          : 1;//Audio interface tristate.
        uint16_t reserved1                  : 1;
        WM8904_TDMCHANNEL_t AIFADC_TDM_CHAN : 1;//ADCDAT TDM channel select.
        WM8904_TDM_t AIFADC_TDM             : 1;//ADC TDM enable.
        WM8904_TDMCHANNEL_t AIFDAC_TDM_CHAN : 1;//DACDAT TDM channel select.
        WM8904_TDM_t AIFDAC_TDM             : 1;//DAC TDM enable.
        uint16_t reserved2                  : 2;
    } bits;
} wm8904AudioInterface1Reg_t;

//------------------------------------------------------------------
// Bits[11:8]:GPIO output clock divider.
// Bits[4:0] :BCLK frequency(Master mode).
//------------------------------------------------------------------
#define WM8904_AUDIOINTERFACE2_REG_ADDR (0x1A)
//GPIO output clock divider options.
typedef enum {
    GPIOCLK_DIV1 = 0,
    GPIOCLK_DIV2 = 1,
    GPIOCLK_DIV3 = 2,
    GPIOCLK_DIV4 = 3,
    GPIOCLK_DIV5_5 = 4,
    GPIOCLK_DIV6 = 5,
    GPIOCLK_DIV8 = 6,
    GPIOCLK_DIV12 = 7,
    GPIOCLK_DIV16 = 8
} WM8904_GPIOOUT_CLKDIV_t;
//BCLK frequency(Master mode) options.
typedef enum {
    BCLK_DIV1 = 0,
    BCLK_DIV1_5 = 1,
    BCLK_DIV2 = 2,
    BCLK_DIV3 = 3,
    BCLK_DIV4 = 4,
    BCLK_DIV5 = 5,
    BCLK_DIV5_5 = 6,
    BCLK_DIV6 = 7,
    BCLK_DIV8 = 8,
    BCLK_DIV10 = 9,
    BCLK_DIV11 = 10,
    BCLK_DIV12 = 11,
    BCLK_DIV16 = 12,
    BCLK_DIV20 = 13,
    BCLK_DIV22 = 14,
    BCLK_DIV24 = 15,
    BCLK_DIV25 = 16,
    BCLK_DIV30 = 17,
    BCLK_DIV32 = 18,
    BCLK_DIV44 = 19,
    BCLK_DIV48 = 20
} WM8904_BCLKFREQ_t;
//Audio interface2 register structure type definition.
typedef union _wm8904AudioInterface2Reg_t {
    uint16_t regVal;
    struct {
        WM8904_BCLKFREQ_t BCLK_DIV        : 5;//BCLK frequency(Master mode).
        uint16_t reserved0                : 3;
        WM8904_GPIOOUT_CLKDIV_t OPCLK_DIV : 4;//GPIO output clock divider.
        uint16_t reserved1                : 4;
    } bits;
} wm8904AudioInterface2Reg_t;

//------------------------------------------------------------------
// Bits[11]  :Audio interface LRC direction.
// Bits[10:0]:LRC rate(Master mode).
//------------------------------------------------------------------
#define WM8904_AUDIOINTERFACE3_REG_ADDR (0x1B)
//Audio interface LRC direction options.
typedef enum {
    LRC_INPUT = 0,
    LRC_OUTPUT = 1
} WM8904_LRCDIR_t;
//Audio interface3 register structure type definition.
typedef union _wm8904AudioInterface3Reg_t {
    uint16_t regVal;
    struct {
        uint16_t LRCLK_DIV        : 11;//LRC rate(Master mode).
                                       //LRC clock output = BLCK / LRCLK_RATE.
                                       //Integer(LSB = 1).
                                       //Valid range: 8 to 2047.
        WM8904_LRCDIR_t LRCLK_DIR : 1; //Audio interface LRC direction.
        uint16_t reserved         : 4;
    } bits;
} wm8904AudioInterface3Reg_t;

//------------------------------------------------------------------
// Bits[8]  :DAC volume update.
// Bits[7:0]:DAC digital volume.
//------------------------------------------------------------------
#define WM8904_DACDIGVOL_LEFT_REG_ADDR  (0x1E)
#define WM8904_DACDIGVOL_RIGHT_REG_ADDR (0x1F)
#define WM8904_ADCDIGVOL_LEFT_REG_ADDR  (0x24)
#define WM8904_ADCDIGVOL_RIGHT_REG_ADDR (0x25)
//Digital volume register structure type definition.
typedef union _wm8904DigVolReg_t{
    uint16_t regVal;
    struct {
        uint16_t VOL       : 8;//Digital volume.
                               //0x00 = Mute.
                               //0x01 = -71.625dB.
                               //0x02 = -71.250dB.
                               //...(0.375dB steps)
                               //0xC0 ~ 0xFF = 0dB
        FlagStatus VU      : 1;//Volume update.
                               //Writing a 1 to this bit causes left and right
                               //Volume to be updated simultaneously.
        uint16_t reserved  : 7;
    } bits;
    struct {
        uint16_t vol       : 8;//DAC digital volume.
                               //0x00 = Mute.
                               //0x01 = -71.625dB.
                               //0x02 = -71.250dB.
                               //...(0.375dB steps)
                               //0xC0 ~ 0xFF = 0dB
        uint16_t reserved  : 8;
    } DAC_VOL;
    struct {
        uint16_t vol       : 8;//ADC digital volume.
                               //0x00 = Mute.
                               //0x01 = -71.625dB.
                               //0x02 = -71.250dB.
                               //...(0.375dB steps)
                               //0xC0 = 0dB.
                               //...(0.375dB steps)
                               //0xEF ~ 0xFF = +17.625dB.
        uint16_t reserved  : 8;
    } ADC_VOL;
    struct {
        uint16_t reserved0 : 8;
        FlagStatus vu      : 1;//DAC volume update.
                               //Writing a 1 to this bit causes left and right
                               //DAC volume to be updated simultaneously.
        uint16_t reserved1 : 7;
    } DAC_VU;
    struct {
        uint16_t reserved0 : 8;
        FlagStatus vu      : 1;//ADC volume update.
                               //Writing a 1 to this bit causes left and right
                               //ADC volume to be updated simultaneously.
        uint16_t reserved1 : 7;
    } ADC_VU;
} wm8904DigVolReg_t;

//------------------------------------------------------------------
// Bits[11:8]:Left digital sidetone volume.
// Bits[7:4] :Right digital sidetone volume.
// Bits[3:2] :Left DAC digital sidetone source.
// Bits[1:0] :Right DAC digital sidetone source.
//------------------------------------------------------------------
#define WM8904_DACDIG0_REG_ADDR         (0x20)
//Digital sidetone volume options.
typedef enum {
    NO_SIDETONE = 0,
    SIDETONE_LEFT_ADC = 1,
    SIDETONE_RIGHT_ADC = 2
} WM8904_SIDETONESRC_t;
//DAC digital sidetone volume register structure type definition.
typedef union _wm8904SidetoneVolReg_t {
    uint16_t regVal;
    struct {
        WM8904_SIDETONESRC_t ADC_TO_DACR : 2;//Right DAC digital sidetone source.
        WM8904_SIDETONESRC_t ADC_TO_DACL : 2;//Left DAC digital sidetone source.
        uint16_t ADCR_DAC_SVOL           : 4;//Right digital sidetone volume.
                                             //0x0000 = -36dB.
                                             //0x0001 = -33dB.
                                             //(...3dB steps)
                                             //0x1011 = -3dB.
                                             //0x11XX = 0dB.
        uint16_t ADCL_DAC_SVOL           : 4;//Left digital sidetone volume.
                                             //0x0000 = -36dB.
                                             //0x0001 = -33dB.
                                             //(...3dB steps)
                                             //0x1011 = -3dB.
                                             //0x11XX = 0dB.
        uint16_t reserved                : 4;
    } bits;
} wm8904SidetoneVolReg_t;

//------------------------------------------------------------------
// Bits[12] :DAC mono mix.
// Bits[11] :Selects DAC filter characteristics.
// Bits[10] :DAC soft mute ramp rate.
// Bits[9]  :DAC soft mute mode.
// Bits[6]  :DAC oversample rate select.
// Bits[3]  :DAC soft mute control.
// Bits[2:1]:DAC de-emphasis control.
//------------------------------------------------------------------
#define WM8904_DACDIG1_REG_ADDR         (0x21)
//DAC soft mute ramp rate options.
typedef enum {
    FAST_RAMP = 0,//fs/2, maximum ramp time is 10.7ms at fs = 48k.
    SLOW_RAMP = 1 //fs/32, maximum ramp time is 171ms at fs = 48k.
} WM8904_DACMUTERATE_t;
//DAC soft mute mode options.
typedef enum {
    CHANGE_IMMEDIATELY = 0,//Disabling soft-mute(DAC_MUTE = 0) will cause the
                           //DAC volume to change immediately to DACL_VOL and
                           //DACR_VOL settings.
    RAMPUP_GRADUALLY = 1   //Disabling soft-mute(DAC_MUTE = 0) will cause the
                           //DAC volume to ramp up gradually to the DACL_VOL
                           //and DACR_VOL settings.
} WM8904_DACUNMUTERAMP_t;
//DAC oversample rate select options.
typedef enum {
    LOW_PWR_MODE = 0,        //Normal OSR.
    HIGH_PERFORMANCE_MODE = 1//Double OSR.
} WM8904_DACOSR_t;
//DAC de-emphasis control options.
typedef enum {
    NO_DEEMPHASIS = 0,
    DEEMPHASIS_32KHZ = 1,
    DEEMPHASIS_44_1KHZ = 2,
    DEEMPHASIS_48KHZ = 3
} WM8904_DEEMPH_t;
//DAC digital1 register structure type definition.
typedef union _wm8904DacDig1Reg_t {
    uint16_t regVal;
    struct {
        uint16_t reserved0                     : 1;
        WM8904_DEEMPH_t DEEMPH                 : 2;//DAC de-emphasis control.
        FunctionalState DAC_MUTE               : 1;//DAC soft mute control.
        uint16_t reserved1                     : 2;
        WM8904_DACOSR_t DAC_OSR128             : 1;//DAC oversample rate select.
        uint16_t reserved2                     : 2;
        WM8904_DACUNMUTERAMP_t DAC_UNMUTE_RAMP : 1;//DAC soft mute mode.
        WM8904_DACMUTERATE_t DAC_MUTERATE      : 1;//DAC soft mute ramp rate.
        FunctionalState DAC_SB_FILT            : 1;//Selects DAC filter characteristics.
        FunctionalState DAC_MONO               : 1;//DAC mono mix.
        uint16_t reserved3                     : 3;
    } bits;
} wm8904DacDig1Reg_t;

//------------------------------------------------------------------
// Bits[6:5]:ADC digital high pass filter cut-off frequency(fc).
// Bits[4]  :ADC digital high pass filter enable.
// Bits[1]  :Left ADC invert.
// Bits[0]  :Right ADC invert.
//------------------------------------------------------------------
#define WM8904_ADCDIG0_REG_ADDR         (0x26)
//ADC digital high pass filter cut-off frequency options.
typedef enum {
    HIFI_MODE = 0,  //fc = 4Hz at fs = 48kHz.
    VOICE_MODE1 = 1,//fc = 127Hz at fs = 16kHz.
    VOICE_MODE2 = 2,//fc = 130Hz at fs = 8kHz.
    VOICE_MODE3 = 3 //fc = 267Hz at fs = 8kHz
} WM8904_ADCHPF_t;
//Left ADC invert options.
typedef enum {
    ADCDAT_NOTINVERT = 0,
    ADCDAT_INVERT = 1
} WM8904_ADCINVERT_t;
//ADC digital0 register structure type definition.
typedef union _wm8904AdcDig0Reg_t {
    uint16_t regVal;
    struct {
        WM8904_ADCINVERT_t ADCR_DATINV : 1;//Right ADC invert.
        WM8904_ADCINVERT_t ADCL_DATINV : 1;//Left ADC invert.
        uint16_t reserved0             : 2;
        FunctionalState ADC_HPF        : 1;//ADC digital high pass filter enable.
        WM8904_ADCHPF_t ADC_HPF_CUT    : 2;//ADC digital high pass filter cut-off frequency.
        uint16_t reserved1             : 9;
    } bits;
} wm8904AdcDig0Reg_t;

//------------------------------------------------------------------
// Bits[12]:Enable digital microphone mode.
//          When DMIC_ENA = 0, the digital microphone clock is held low.
// Bits[11]:Selects digital microphone data input pin.
//------------------------------------------------------------------
#define WM8904_MICPHONE0_REG_ADDR       (0x27)
//Enable digital microphone mode options.
typedef enum {
    AUDIO_FROM_ADC = 0,
    AUDIO_FROM_MIC = 1
} WM8904_MICMODE_t;
//Selects digital microphone data input pin options.
typedef enum {
    IN1L_DMICDAT1 = 0,
    IN1R_DMICDAT2 = 1
} WM8904_MICINPUTPIN_t;
//Digital microphone0 register structure type definition.
typedef union _wm8904Mic0Reg_t {
    uint16_t regVal;
    struct {
        uint16_t reserved0            : 11;
        WM8904_MICINPUTPIN_t DMIC_SRC : 1;//Selects digital microphone data input pin.
        WM8904_MICMODE_t DMIC_ENA     : 1;//Enables digital microphone mode.
        uint16_t reserved1            : 3;
    } bits;
} wm8904Mic0Reg_t;

//------------------------------------------------------------------
// Bits[15]   :DRC enable.
// Bits[14]   :DRC path select.
// Bits[12:11]:Gain smoothing hysteresis threshold.
// Bits[10:6] :Initial gain at DRC startup.
// Bits[5]    :Feed-forward delay for anti-clip feature.
// Bits[3]    :Gain smoothing enable.
// Bits[2]    :Quick release enable.
// Bits[1]    :Anti-clip enable.
// Bits[0]    :Gain smoothing hysteresis enable.
//------------------------------------------------------------------
#define WM8904_DRC0_REG_ADDR        (0x28)
//DRC path options.
typedef enum {
    ADC_PATH = 0,
    DAC_PATH = 1
} WM8904_DRCPATH_t;
//Gain smoothing hysteresis threshold options.
typedef enum {
    GSHYST_LOW = 0,
    GSHYST_MID = 1,
    GSHYST_HIGH = 2
} WM8904_GSHYST_THD_t;
//Initial gain at DRC startup options.
typedef enum {
    GAIN_n3DB = 0,
    GAIN_n2_5DB = 1,
    GAIN_n2DB = 2,
    GAIN_n1_5DB = 3,
    GAIN_n1DB = 4,
    GAIN_n0_5DB = 5,
    GAIN_0DB = 6,
    GAIN_0_5DB = 7,
    GAIN_1DB = 8,
    GAIN_1_5DB = 9,
    GAIN_2DB = 10,
    GAIN_2_5DB = 11,
    GAIN_3DB = 12,
    GAIN_3_5DB = 13,
    GAIN_4DB = 14,
    GAIN_4_5DB = 15,
    GAIN_5DB = 16,
    GAIN_5_5DB = 17,
    GAIN_6DB = 18
} WM8904_STARTUP_GAIN_t;
//Feed-forward delay for anti-clip feature options.
typedef enum {
    FFDELAY_5SAMPLES = 0,
    FFDELAY_9SAMPLES = 1
} WM8904_FFDELAY_t;
//DRC0 register structure type definition.
typedef union _wm8904Drc0Reg_t {
    uint16_t regVal;
    struct {
        FunctionalState DRC_GS_HYST            : 1;//Gain smoothing hysteresis enable.
        FunctionalState DRC_ANTICLIP           : 1;//Anti-clip enable.
        FunctionalState DRC_QR                 : 1;//Quick release enable.
        FunctionalState DRC_GS_ENA             : 1;//Gain smoothing enable.
        uint16_t reserved0                     : 1;
        WM8904_FFDELAY_t DRC_FF_DELAY          : 1;//Feed-forward delay for anti-clip feature.
        WM8904_STARTUP_GAIN_t DRC_STARTUP_GAIN : 5;//Initial gain at DRC startup.
        WM8904_GSHYST_THD_t DRC_GS_HYST_LVL    : 2;//Gain smoothing hysteresis threshold.
        uint16_t reserved1                     : 1;
        WM8904_DRCPATH_t DRC_DAC_PATH          : 1;//DRC path select.
        FunctionalState DRC_ENA                : 1;//DRC enable.
    } bits;
} wm8904Drc0Reg_t;

//------------------------------------------------------------------
// Bits[15:12]:Gain attack rate(seconds/6dB).
// Bits[11:8] :Gain decay rate(seconds/6dB).
// Bits[7:6]  :Quick release crest factor threshold.
// Bits[5:4]  :Quick release decay rate(seconds/6dB).
// Bits[3:2]  :Minimum gain the DRC can use to attenuate audio signals.
// Bits[1:0]  :Maximum gain the DRC can use to boost audio signals.
//------------------------------------------------------------------
#define WM8904_DRC1_REG_ADDR             (0x29)
//Gain attack rate options.
typedef enum {
    ATK_182US = 1,
    ATK_363US = 2,
    ATK_726US = 3,
    ATK_1_45MS = 4,
    ATK_2_9MS = 5,
    ATK_5_8MS = 6,
    ATK_11_6MS = 7,
    ATK_23_2MS = 8,
    ATK_46_4MS = 9,
    ATK_92_8MS = 10
} WM8904_DRCATK_t;
//Gain decay rate options.
typedef enum {
    DCY_186MS = 0,
    DCY_372MS = 1,
    DCY_743MS = 2,
    DCY_1_49S = 3,
    DCY_2_97S = 4,
    DCY_5_94S = 5,
    DCY_11_89S = 6,
    DCY_23_78S = 7,
    DCY_47_56S = 8
} WM8904_DRCDCY_t;
//Quick release crest factor threshold options.
typedef enum {
    QRTHD_12DB = 0,
    QRTHD_18DB = 1,
    QRTHD_24DB = 2,
    QRTHD_30DB = 3
} WM8904_DRCQRTHD_t;
//Quick release decay rate(seconds/6dB) options.
typedef enum {
    QRDCY_0_725MS = 0,
    QRDCY_1_45MS = 1,
    QRDCY_5_8MS = 2
} WM8904_QRDCY_t;
//Minimum gain the DRC can use to attenuate audio signals options.
typedef enum {
    MINGAIN_0DB = 0,
    MINGAIN_n6DB = 1,
    MINGAIN_n12DB = 2,
    MINGAIN_n18DB = 3
} WM8904_MINGAIN_t;
//Maximum gain the DRC can use to boost audio signals options.
typedef enum {
    MAXGAIN_12DB = 0,
    MAXGAIN_18DB = 1,
    MAXGAIN_24DB = 2,
    MAXGAIN_36DB = 3
} WM8904_MAXGAIN_t;
//DRC1 register structure type definition.
typedef union _wm8904Drc1Reg_t {
    uint16_t regVal;
    struct {
        WM8904_MAXGAIN_t DRC_MAXGAIN : 2;//Maximum gain the DRC can use to boost audio signals
        WM8904_MINGAIN_t DRC_MINGAIN : 2;//Minimum gain the DRC can use to attenuate audio signals options.
        WM8904_QRDCY_t DRC_QR_DCY    : 2;//Quick release decay rate(seconds/6dB).
        WM8904_DRCQRTHD_t DRC_QR_THD : 2;//Quick release crest factor threshold
        WM8904_DRCDCY_t DRC_DCY      : 4;//Gain decay rate.
        WM8904_DRCATK_t DRC_ATK      : 4;//Gain attack rate
    } bits;
} wm8904Drc1Reg_t;

//------------------------------------------------------------------
// Bits[5:3]:Compressor slope(upper region).
// Bits[2:0]:Compressor slope(lower region).
//------------------------------------------------------------------
#define WM8904_DRC2_REG_ADDR             (0x2A)
//Compressor slope(upper region) options.
typedef enum {
    HI_COMP_DIV1 = 0,
    HI_COMP_DIV2 = 1,
    HI_COMP_DIV4 = 2,
    HI_COMP_DIV8 = 3,
    HI_COMP_DIV16 = 4,
    HI_COMP_DIV0 = 5
} WM8904_DRCHICOMP_t;
//Compressor slope(lower region) options.
typedef enum {
    LO_COMP_DIV1 = 0,
    LO_COMP_DIV2 = 1,
    LO_COMP_DIV4 = 2,
    LO_COMP_DIV8 = 3,
    LO_COMP_DIV0 = 4
} WM8904_DRCLOCOMP_t;
//DRC2 register structure type definition.
typedef union _wm8904Drc2Reg_t {
    uint16_t regVal;
    struct {
        WM8904_DRCLOCOMP_t DRC_LO_COMP : 3;//Compressor slope(lower region).
        WM8904_DRCHICOMP_t DRC_HI_COMP : 3;//Compressor slope(upper region).
        uint16_t reserved              : 10;
    } bits;
} wm8904Drc2Reg_t;

//------------------------------------------------------------------
// Bits[10:5]:Input signal at the compressor 'knee'.
// Bits[4:0] :Output signal at the compressor 'knee'.
//------------------------------------------------------------------
#define WM8904_DRC3_REG_ADDR             (0x2B)
//DRC3 register structure type definition.
typedef union _wm8904Drc3Reg_t {
    uint16_t regVal;
    struct {
        uint16_t DRC_KNEE_OP : 5;//Output signal at the compressor 'knee'.
                                 //0x00 = 0dB.
                                 //0x01 = -0.75dB.
                                 //0x02 = -1.5dB.
                                 //...(-0.75dB steps)
                                 //0x3C = -45dB.
                                 //0x3D ~ 0x3F = Reserved.
        uint16_t DRC_KNEE_IP : 5;//Input signal at the compressor 'knee'.
                                 //0x00 = 0dB.
                                 //0x01 = -0.75dB.
                                 //0x02 = -1.5dB.
                                 //...(-0.75dB steps)
                                 //0x1E = -22.5dB.
                                 //0x1F = Reserved.
        uint16_t reserved    : 6;
    } bits;
} wm8904Drc3Reg_t;

//------------------------------------------------------------------
// Bits[7]  :Input PGA mute.
// Bits[4:0]:Input PGA volume.
//------------------------------------------------------------------
#define WM8904_ANA_LTINPUT0_REG_ADDR   (0x2C)
#define WM8904_ANA_RTINPUT0_REG_ADDR   (0x2D)
//Input PGA mute options.
typedef enum {
    INPUT_NOT_MUTE = 0,
    INPUT_MUTE = 1
} WM8904_INPUT_MUTE_t;
//Input PGA volume in mode = 0(Single ended) or mode = 1(Differential line) options.
typedef enum {
    INVOL_n1_5DB = 0,
    INVOL_n1_3DB = 1,
    INVOL_n1DB = 2,
    INVOL_n0_7DB = 3,
    INVOL_n0_3DB = 4,
    INVOL_0DB = 5,
    INVOL_0_3DB = 6,
    INVOL_0_7DB = 7,
    INVOL_1_0DB = 8,
    INVOL_1_4DB = 9,
    INVOL_1_8DB = 10,
    INVOL_2_3DB = 11,
    INVOL_2_7DB = 12,
    INVOL_3_2DB = 13,
    INVOL_3_7DB = 14,
    INVOL_4_2DB = 15,
    INVOL_4_8DB = 16,
    INVOL_5_4DB = 17,
    INVOL_6_0DB = 18,
    INVOL_6_7DB = 19,
    INVOL_7_5DB = 20,
    INVOL_8_3DB = 21,
    INVOL_9_2DB = 22,
    INVOL_10_2DB = 23,
    INVOL_11_4DB = 24,
    INVOL_12_7DB = 25,
    INVOL_14_3DB = 26,
    INVOL_16_2DB = 27,
    INVOL_19_2DB = 28,
    INVOL_22_3DB = 29,
    INVOL_25_2DB = 30,
    INVOL_28_3DB = 31
} WM8904_INVOL_MOD0_1_t;
//Input PGA volume in mode = 2(Differential MIC).
typedef enum {
    INVOL_12DB = 1,
    INVOL_15DB = 2,
    INVOL_18DB = 3,
    INVOL_21DB = 4,
    INVOL_24DB = 5,
    INVOL_27DB = 6,
    INVOL_30DB = 7
} WM8904_INVOL_MOD2_t;
//Analogue input0 register structure type definition.
typedef union _wm8904AnaInput0Reg_t {
    uint16_t regVal;
    struct {
        WM8904_INVOL_MOD0_1_t inVol : 5;//Input PGA volume if mode = 0 or mode = 1.
        uint16_t reserved           : 11;
    } INVOL_MOD0_1;
    struct {
        WM8904_INVOL_MOD2_t inVol   : 5;//Input PGA volume if mode = 2.
        uint16_t reserved           : 11;
    } INVOL_MOD2;
    struct {
        uint16_t reserved0          : 7;
        WM8904_INPUT_MUTE_t INMUTE  : 1;//Input PGA mute.
        uint16_t reserved1          : 8;
    } bits;
} wm8904AnaInput0Reg_t;

//------------------------------------------------------------------
// Bits[6]  :Input PGA common mode rejection enable.
// Bits[5:4]:In Single-Ended or Differential Line modes, this field
//           selects the input pin for the inverting side of the
//           input path.
//           In Differential Mic mode, this field selects the input
//           pin for the non-inverting side of the input path.
// Bits[3:2]:In Single-Ended or Differential Line modes, this field
//           selects the input pin for the non-inverting side of the
//           input path.
//           In Differential Mic mode, this field selects the input
//           pin for the inverting side of the input path.
// Bits[1:0]:Sets the mode for the analogue input.
//------------------------------------------------------------------
#define WM8904_ANA_LTINPUT1_REG_ADDR   (0x2E)
#define WM8904_ANA_RTINPUT1_REG_ADDR   (0x2F)
//Input pin for the inverting or non-inverting side of the input path options.
typedef enum {
    IN1 = 0,
    IN2 = 1,
    IN3 = 2
} WM8904_INSEL_t;
//Sets the mode for the analogue input options.
typedef enum {
    SINGLE_ENDED = 0,
    DIFFER_LINE = 1,
    DIFFER_MIC = 2
} WM8904_INMODE_t;
//Analogue input1 register structure type definition.
typedef union _wm8904AnaInput1Reg_t {
    uint16_t regVal;
    struct {
        WM8904_INMODE_t IN_MODE   : 2;//Sets the mode for the analogue input.
        WM8904_INSEL_t IN_SEL_P   : 2;//Sets the inverting or non-inverting side of the input path.
        WM8904_INSEL_t IN_SEL_N   : 2;//Sets the inverting or non-inverting side of the input path.
        FunctionalState IN_CM_ENA : 1;//Input PGA common mode rejection enable.
        uint16_t reserved         : 9;
    } bits;
} wm8904AnaInput1Reg_t;

//------------------------------------------------------------------
// Bits[8]  :Headphone or line output mute.
// Bits[7]  :Headphone or line output volume update.
//           Writing a 1 to this bit will update HPOUTL/HPOUTR or
//           LINEOUTL/LINEOUTR volumes simultaneously.
// Bits[6]  :Headphone or line output zero cross enable.
// Bits[5:0]:Headphone or line output volume.
//------------------------------------------------------------------
#define WM8904_ANA_HPLTOUTPUT_REG_ADDR     (0x39)
#define WM8904_ANA_HPRTOUTPUT_REG_ADDR     (0x3A)
#define WM8904_ANA_LINELTOUTPUT_REG_ADDR   (0x3B)
#define WM8904_ANA_LINERTOUTPUT_REG_ADDR   (0x3C)
//Output mute options.
typedef enum {
    OUTPUT_NOT_MUTE = 0,
    OUTPUT_MUTE = 1
} WM8904_HP_LINE_MUTE_t;
//Analogue output register structure type definition.
typedef union _wm8904AnaOutputReg_t {
    uint16_t regVal;
    struct {
        uint16_t OUT_VOL               : 6;//Output volume.
                                           //0x00 = -57dB.
                                           //0x01 = -56dB.
                                           //...(1dB steps).
                                           //0x39 = 0dB.
                                           //...(1dB steps).
                                           //0x3E = +5dB.
                                           //0x3F = +6dB.
        FunctionalState OUT_ZC         : 1;//Output zero cross enable.
        FlagStatus OUT_VU              : 1;//Output volume update.
        WM8904_HP_LINE_MUTE_t OUT_MUTE : 1;//Output mute.
        uint16_t reserved              : 7;
    } bits;
} wm8904AnaOutputReg_t;

//------------------------------------------------------------------
// Bits[3]:Selects input for left headphone output MUX.
// Bits[2]:Selects input for right headphone output MUX.
// Bits[1]:Select input for left line output MUX.
// Bits[0]:Select input for right line output MUX.
//------------------------------------------------------------------
#define WM8904_ANA_OUT12ZC_REG_ADDR  (0x3D)
//Input for output MUX options.
typedef enum {
    INPUT_DAC = 0,
    INPUT_PGA = 1
} WM8904_INSRC_TO_OUTMUX_t;
//Analogue OUT12 ZC register structure type definition.
typedef union _wm8904AnaOut12ZcReg_t {
    uint16_t regVal;
    struct {
        WM8904_INSRC_TO_OUTMUX_t LINEOUTR_BYP_ENA : 1;//Select input for right line output MUX.
        WM8904_INSRC_TO_OUTMUX_t LINEOUTL_BYP_ENA : 1;//Select input for left line output MUX.
        WM8904_INSRC_TO_OUTMUX_t HPR_BYP_ENA      : 1;//Selects input for right headphone output MUX.
        WM8904_INSRC_TO_OUTMUX_t HPL_BYP_ENA      : 1;//Selects input for left headphone output MUX.
        uint16_t reserved                         : 12;
    } bits;
} wm8904AnaOut12ZcReg_t;

//------------------------------------------------------------------
// Bits[3]:DC Servo enable for LINEOUTR.
// Bits[2]:DC Servo enable for LINEOUTL.
// Bits[1]:DC Servo enable for HPOUTR.
// Bits[0]:DC Servo enable for HPOUTL.
//------------------------------------------------------------------
#define WM8904_DC_SERVO0_REG_ADDR       (0x43)
//DC Servo0 register structure type definition.
typedef union _wm8904DcServo0Reg_t {
    uint16_t regVal;
    struct {
        FunctionalState DCS_ENA_CHAN_0 : 1;//DC servo enable for HPOUTL.
        FunctionalState DCS_ENA_CHAN_1 : 1;//DC servo enable for HPOUTR.
        FunctionalState DCS_ENA_CHAN_2 : 1;//DC servo enable for LINEOUTL.
        FunctionalState DCS_ENA_CHAN_3 : 1;//DC servo enable for LINEOUTR.
        uint16_t reserved              : 12;
    } bits;
} wm8904DcServo0Reg_t;

//------------------------------------------------------------------
// Bits[15]:Writing 1 to this bit selects a single DC offset 
//          correction for LINEOUTR.
//          In readback, a value of 1 indicates that the DC Servo
//          single correction is in progress.
// Bits[14]:Writing 1 to this bit selects a single DC offset 
//          correction for LINEOUTL.
//          In readback, a value of 1 indicates that the DC Servo
//          single correction is in progress.
// Bits[13]:Writing 1 to this bit selects a single DC offset 
//          correction for HPOUTR.
//          In readback, a value of 1 indicates that the DC Servo
//          single correction is in progress.
// Bits[12]:Writing 1 to this bit selects a single DC offset 
//          correction for HPOUTL.
//          In readback, a value of 1 indicates that the DC Servo
//          single correction is in progress.
// Bits[11]:Writing 1 to this bit selects a series of DC offset
//          corrections for LINEOUTR.
//          In readback, a value of 1 indicates that the DC Servo
//          DAC Write correction is in progress.
// Bits[10]:Writing 1 to this bit selects a series of DC offset
//          corrections for LINEOUTL.
//          In readback, a value of 1 indicates that the DC Servo
//          DAC Write correction is in progress.
// Bits[9] :Writing 1 to this bit selects a series of DC offset
//          corrections for HPOUTR.
//          In readback, a value of 1 indicates that the DC Servo
//          DAC Write correction is in progress.
// Bits[8] :Writing 1 to this bit selects a series of DC offset
//          corrections for HPOUTL.
//          In readback, a value of 1 indicates that the DC Servo
//          DAC Write correction is in progress.
// Bits[7] :Writing 1 to this bit selects Start-Up DC Servo mode
//          for LINEOUTR.
//          In readback, a value of 1 indicates that the DC Servo
//          Start-Up correction is in progress.
// Bits[6] :Writing 1 to this bit selects Start-Up DC Servo mode
//          for LINEOUTL.
//          In readback, a value of 1 indicates that the DC Servo
//          Start-Up correction is in progress.
// Bits[5] :Writing 1 to this bit selects Start-Up DC Servo mode
//          for HPOUTR.
//          In readback, a value of 1 indicates that the DC Servo
//          Start-Up correction is in progress.
// Bits[4] :Writing 1 to this bit selects Start-Up DC Servo mode
//          for HPOUTL.
//          In readback, a value of 1 indicates that the DC Servo
//          Start-Up correction is in progress.
// Bits[3] :Writing 1 to this bit selects DAC Write DC Servo mode
//          for LINEOUTR.
//          In readback, a value of 1 indicates that the DC Servo
//          DAC Write correction is in progress.
// Bits[2] :Writing 1 to this bit selects DAC Write DC Servo mode
//          for LINEOUTL.
//          In readback, a value of 1 indicates that the DC Servo
//          DAC Write correction is in progress.
// Bits[1] :Writing 1 to this bit selects DAC Write DC Servo mode
//          for HPOUTR.
//          In readback, a value of 1 indicates that the DC Servo
//          DAC Write correction is in progress.
// Bits[0] :Writing 1 to this bit selects DAC Write DC Servo mode
//          for HPOUTL.
//          In readback, a value of 1 indicates that the DC Servo
//          DAC Write correction is in progress.
//------------------------------------------------------------------
#define WM8904_DC_SERVO1_REG_ADDR       (0x44)
//DC Servo1 register structure type definition.
typedef union _wm8904DcServo1Reg_t {
    uint16_t regVal;
    struct {
        FlagStatus DCS_TRIG_DAC_WR_0  : 1;
        FlagStatus DCS_TRIG_DAC_WR_1  : 1;
        FlagStatus DCS_TRIG_DAC_WR_2  : 1;
        FlagStatus DCS_TRIG_DAC_WR_3  : 1;
        FlagStatus DCS_TRIG_STARTUP_0 : 1;
        FlagStatus DCS_TRIG_STARTUP_1 : 1;
        FlagStatus DCS_TRIG_STARTUP_2 : 1;
        FlagStatus DCS_TRIG_STARTUP_3 : 1;
        FlagStatus DCS_TRIG_SERIES_0  : 1;
        FlagStatus DCS_TRIG_SERIES_1  : 1;
        FlagStatus DCS_TRIG_SERIES_2  : 1;
        FlagStatus DCS_TRIG_SERIES_3  : 1;
        FlagStatus DCS_TRIG_SINGLE_0  : 1;
        FlagStatus DCS_TRIG_SINGLE_1  : 1;
        FlagStatus DCS_TRIG_SINGLE_2  : 1;
        FlagStatus DCS_TRIG_SINGLE_3  : 1;
    } bits;
} wm8904DcServo1Reg_t;

//------------------------------------------------------------------
// Bits[11:8]:Time between periodic updates for LINEOUTL/LINEOUTR.
//            Time is calculated as 0.256s x (2^PERIOD).
// Bits[3:0] :Time between periodic updates for HPOUTL/HPOUTR.
//            Time is calculated as 0.256s x (2^PERIOD).
//------------------------------------------------------------------
#define WM8904_DC_SERVO2_REG_ADDR       (0x45)
//Time between periodic updates options.
typedef enum {
    TIMPERIOD_OFF = 0,
    TIMPERIOD_0_52S = 1,
    TIMPERIOD_266S = 10,
    TIMPERIOD_8519S = 15
} WM8904_DCS_TIMPERIOD_t;
//DC Servo2 register structure type definition.
typedef union _wm8904DcServo2Reg_t {
    uint16_t regVal;
    struct {
        WM8904_DCS_TIMPERIOD_t DCS_TIMER_PERIOD_01 : 4;//Time between periodic updates for HPOUTL/HPOUTR.
        uint16_t reserved0                         : 4;
        WM8904_DCS_TIMPERIOD_t DCS_TIMER_PERIOD_23 : 4;//Time between periodic updates for LINEOUTL/LINEOUTR.
        uint16_t reserved1                         : 4;
    } bits;
} wm8904DcServo2Reg_t;

//------------------------------------------------------------------
// Bits[6:0]:Number of DC Servo updates to perform in a series event.
//------------------------------------------------------------------
#define WM8904_DC_SERVO4_REG_ADDR       (0x47)
#define WM8904_DC_SERVO5_REG_ADDR       (0x48)
//DC Servo4/5 Series Number register structure type definition.
typedef union _wm8904DcServoSnReg_t {
    uint16_t regVal;
    struct {
        uint16_t DCS_SERIES_NO : 7;//Number of DC Servo updates to perform in a series event.
                                   //0x00 = 1 updates.
                                   //0x01 = 2 updates.
                                   //...
                                   //0x7F = 128 updates.
        uint16_t reserved      : 9;
    } bits;
} wm8904DcServoSnReg_t;

//------------------------------------------------------------------
// Bits[7:0]:DC Offset value in DAC Write DC Servo mode in two's
//           complement format.
//           In readback, the current DC Offset value is returned 
//           in two's complement format.
//           Two's complement format:
//           LSB is 0.25mV.
//           Range is +/-32mV.
//------------------------------------------------------------------
#define WM8904_DC_SERVO6_REG_ADDR       (0x49)
#define WM8904_DC_SERVO7_REG_ADDR       (0x4A)
#define WM8904_DC_SERVO8_REG_ADDR       (0x4B)
#define WM8904_DC_SERVO9_REG_ADDR       (0x4C)
//DC Servo6/7/8/9 DAC Write Value register structure type definition.
typedef union _wm8904DcServoDacWrValReg_t {
    uint16_t regVal;
    struct {
        int16_t DCS_DAC_WR_VAL  : 8;
        uint16_t reserved       : 8;
    } bits;
} wm8904DcServoDacWrValReg_t;

//------------------------------------------------------------------
// Bits[11:8]:DC Servo complete status.
// Bits[7:4] :DC Servo DAC Write status.
// Bits[3:0] :DC Servo Start-Up status.
//------------------------------------------------------------------
#define WM8904_DCSERVO_RDBACK_REG_ADDR (0x4D)
//DC Servo readback register structure type definition.
typedef union _wm8904DcServoRdBackReg_t {
    uint16_t regVal;
    struct {
        FlagStatus DCS_STARTUP_DONE_HPOUTL   : 1;
        FlagStatus DCS_STARTUP_DONE_HPOUTR   : 1;
        FlagStatus DCS_STARTUP_DONE_LINEOUTL : 1;
        FlagStatus DCS_STARTUP_DONE_LINEOUTR : 1;
        FlagStatus DCS_DAC_WR_DONE_HPOUTL    : 1;
        FlagStatus DCS_DAC_WR_DONE_HPOUTR    : 1;
        FlagStatus DCS_DAC_WR_DONE_LINEOUTL  : 1;
        FlagStatus DCS_DAC_WR_DONE_LINEOUTR  : 1;
        FlagStatus DCS_CAL_DONE_HPOUTL       : 1;
        FlagStatus DCS_CAL_DONE_HPOUTR       : 1;
        FlagStatus DCS_CAL_DONE_LINEOUTL     : 1;
        FlagStatus DCS_CAL_DONE_LINEOUTR     : 1;
        uint16_t reserved                    : 4;
    } bits;
} wm8904DcServoRdBackReg_t;

//------------------------------------------------------------------
// Bits[7]:Output left stereo short enable.
//         For normal operation, this bit should be set as the final
//         step of the left stereo Enable sequence.
// Bits[6]:Enables left stereo output stage.
//         For normal operation, this bit should be set to 1 after
//         the DC offset cancellation has been scheduled.
// Bits[5]:Enables left stereo output intermediate stage.
//         For normal operation, this bit should be set to 1 after
//         the output signal path has been configured, and before
//         DC offset cancellation is scheduled.
//         This bit should be set with at least 20us delay after
//         left stereo enable.
// Bits[4]:Enables left stereo input stage.
//         For normal operation, this bit should be set as the 
//         first setp of the left stereo Enable sequence.
// Bits[3]:Output right stereo short enable.
//         For normal operation, this bit should be set as the final
//         step of the right stereo Enable sequence.
// Bits[2]:Enables right stereo output stage.
//         For normal operation, this bit should be set to 1 after
//         the DC offset cancellation has been scheduled.
// Bits[1]:Enables right stereo output intermediate stage.
//         For normal operation, this bit should be set to 1 after
//         the output signal path has been configured, and before
//         DC offset cancellation is scheduled.
//         This bit should be set with at least 20us delay after
//         right stereo enable.
// Bits[0]:Enables right stereo input stage.
//         For normal operation, this bit should be set as the 
//         first setp of the right stereo Enable sequence.
//------------------------------------------------------------------
#define WM8904_ANA_HP0_REG_ADDR      (0x5A)
#define WM8904_ANA_LINEOUT0_REG_ADDR (0x5E)
//Removes short options.
typedef enum {
    SHORT_EN = 0,
    SHORT_RM = 1
} WM8904_RMSHORT_t;
//DC analogue HP or Lineout register structure type definition.
typedef union _wm8904AnaHpLineReg_t {
    uint16_t regVal;
    struct {
        WM8904_RMSHORT_t LT_RM_SHORT : 1;
        FunctionalState LT_ENA_OUTP  : 1;
        FunctionalState LT_ENA_DLY   : 1;
        FunctionalState LT_ENA       : 1;
        WM8904_RMSHORT_t RT_RM_SHORT : 1;
        FunctionalState RT_ENA_OUTP  : 1;
        FunctionalState RT_ENA_DLY   : 1;
        FunctionalState RT_ENA       : 1;
        uint16_t reserved            : 8;
    } bits;
} wm8904AnaHpLineReg_t;

//------------------------------------------------------------------
// Bits[0]:Enable charge-pump digits.
//------------------------------------------------------------------
#define WM8904_CHARGE_PUMP0_REG_ADDR    (0x62)
//Charge Pump0 register structure type definition.
typedef union _wm8904ChargePump0Reg_t {
    uint16_t regVal;
    struct {
        FunctionalState CP_ENA : 1;//Enable charge-pump digits.
        uint16_t reserved      : 15;
    } bits;
} wm8904ChargePump0Reg_t;

//------------------------------------------------------------------
// Bits[0]:Enable dynamic charge pump power control.
//------------------------------------------------------------------
#define WM8904_CLASS_W_REG_ADDR         (0x68)
//Dynamic charge pump power control options.
typedef enum {
    PWRCTRL_VOLREG_CLASS_G = 0,
    PWRCTRL_RTAUDIOLV_CLASS_W = 1
} WM8904_DYNPWRCTRL_t;
//Class W register structure type definition.
typedef union _wm8904ClassWReg_t {
    uint16_t regVal;
    struct {
        WM8904_DYNPWRCTRL_t CP_DYN_PWR : 1;//Dynamic charge pump power control.
        uint16_t reserved              : 15;
    } bits;
} wm8904ClassWReg_t;

//------------------------------------------------------------------
// Bits[8]:  Write Sequencer Enable.
// Bits[4:0]:Sequence Write Index.
//           This is the memory location to which any updates to 
//           R109 and R110 will be copied.
//           0 to 31 = RAM addresses.
//------------------------------------------------------------------
#define WM8904_WRSEQ0_ADDR              (0x6C)
//Write sequencer0 register structure type definition.
typedef union _wm8904WrSeq0Reg_t {
    uint16_t regVal;
    struct {
        uint16_t WSEQ_WRITE_INDEX : 5;
        uint16_t reserved0        : 3;
        FunctionalState WSEQ_ENA  : 1;
        uint16_t reserved1        : 7;
    } bits;
} wm8904WrSeq0Reg_t;

//------------------------------------------------------------------
// Bits[14:12]:Width of the data block written in this sequence step.
// Bits[11:8] :Bit position of the LSB of the data block written in
//             this sequence step.
// Bits[7:0]  :Control Register Address to be written to in this
//             sequence step.
//------------------------------------------------------------------
#define WM8904_WRSEQ1_ADDR              (0x6D)
//Width of the data block options.
typedef enum {
    WRSEQ_WIDTH_1BIT  = 0,
    WRSEQ_WIDTH_2BITS = 1,
    WRSEQ_WIDTH_3BITS = 2,
    WRSEQ_WIDTH_4BITS = 3,
    WRSEQ_WIDTH_5BITS = 4,
    WRSEQ_WIDTH_6BITS = 5,
    WRSEQ_WIDTH_7BITS = 6,
    WRSEQ_WIDTH_8BITS = 7
} WM8904_WRSEQ_WIDTH_t;
//Bit position of the LSB options.
typedef enum {
    WRSEQ_POS_BIT0 = 0,
    WRSEQ_POS_BIT1 = 1,
    WRSEQ_POS_BIT2 = 2,
    WRSEQ_POS_BIT3 = 3,
    WRSEQ_POS_BIT4 = 4,
    WRSEQ_POS_BIT5 = 5,
    WRSEQ_POS_BIT6 = 6,
    WRSEQ_POS_BIT7 = 7,
    WRSEQ_POS_BIT8 = 8,
    WRSEQ_POS_BIT9 = 9,
    WRSEQ_POS_BIT10 = 10,
    WRSEQ_POS_BIT11 = 11,
    WRSEQ_POS_BIT12 = 12,
    WRSEQ_POS_BIT13 = 13,
    WRSEQ_POS_BIT14 = 14,
    WRSEQ_POS_BIT15 = 15,
} WM8904_WRSEQ_BITPOS_t;
//Write sequencer1 register structure type definition.
typedef union _wm8904WrSeq1Reg_t {
    uint16_t regVal;
    struct {
        uint16_t WSEQ_ADDR                    : 8;//Control Register Address to be written.
        WM8904_WRSEQ_BITPOS_t WSEQ_DATA_START : 4;//Bit position of the LSB.
        WM8904_WRSEQ_WIDTH_t WSEQ_DATA_WIDTH  : 3;//Width of the data block written.
        uint16_t reserved                     : 1;
    } bits;
} wm8904WrSeq1Reg_t;

//------------------------------------------------------------------
// Bits[14]  :End of Sequence flag. This bit indicates whether the
//            Control Write Sequencer should stop after executing
//            this step.
// Bits[11:8]:Time delay after executing this step.
//            Total delay time per step(including execution)
//            = 62.5us x (2^WSEQ_DELAY + 8).
// Bits[7:0] :Data to be written in this sequence step.
//            When the data width is less than 8 bits, then one or
//            more of the MSBs of WSEQ_DATA are ignored.
//            It is recommended that unused bits be set to 0.
//------------------------------------------------------------------
#define WM8904_WRSEQ2_ADDR              (0x6E)
//End of Sequence flag options.
typedef enum {
    SEQ_NOT_END = 0,
    SEQ_END = 1
} WM8904_SEQENDFLG_t;
//Write sequencer2 register structure type definition.
typedef union _wm8904WrSeq2Reg_t {
    uint16_t regVal;
    struct {
        uint16_t WSEQ_DATA            : 8;//Data to be written.
        uint16_t WSEQ_DELAY           : 4;//Time delay after executing this step.
        uint16_t reserved0            : 2;
        WM8904_SEQENDFLG_t WSEQ_EOS   : 1;//End of Sequence flag.
        uint16_t reserved1            : 1;
    } bits;
} wm8904WrSeq2Reg_t;

//------------------------------------------------------------------
// Bits[9]  :Writing a 1 to this bit aborts the current sequence and
//           returns control of the device back to the serial control
//           interface.
// Bits[8]  :Writing a 1 to this bit starts the write sequencer at
//           the memory location indicated by the WSEQ_START_INDEX
//           field.
//           The sequence continues until it reaches an
//           "End of sequence" flag.
//           At the end of the sequence, this bit will be reset by
//           the Write Sequencer.
// Bits[5:0]:Sequence Start Index. This is the memory location of
//           the fist command in the selected sequence.
//           0  to 31 = RAM address.
//           32 to 48 = ROM address.
//------------------------------------------------------------------
#define WM8904_WRSEQ3_ADDR              (0x6F)
//Write sequencer3 register structure type definition.
typedef union _wm8904WrSeq3Reg_t {
    uint16_t regVal;
    struct {
        uint16_t WSEQ_START_INDEX : 6;//Sequence start index.
        uint16_t reserved0        : 2;
        FlagStatus WSEQ_START     : 1;//Writing 1 to this bit starts the write sequencer.
        FlagStatus WSEQ_ABORT     : 1;//Writing 1 to this bit returns control of the device.
        uint16_t reserved1        : 6;
    } bits;
} wm8904WrSeq3Reg_t;

//------------------------------------------------------------------
// Bits[9:4]:Sequence Current Index(read only).
//           This is the location of the most recently accessed
//           command in the write sequencer memory.
// Bits[0]  :Sequencer Busy flag(read only).
//------------------------------------------------------------------
#define WM8904_WRSEQ4_ADDR              (0x70)
//Sequencer Busy flag options.
typedef enum {
    SEQ_IDLE = 0,
    SEQ_BUSY = 1
} WM8904_SEQBSY_t;
//Write sequencer4 register structure type definition.
typedef union _wm8904WrSeq4Reg_t {
    uint16_t regVal;
    struct {
        WM8904_SEQBSY_t WSEQ_BUSY   : 1;//Sequencer Busy flag.
        uint16_t reserved0          : 3;
        uint16_t WSEQ_CURRENT_INDEX : 6;//Sequence current index.
        uint16_t reserved1          : 6;
    } bits;
} wm8904WrSeq4Reg_t;

//------------------------------------------------------------------
// Bits[2]:FLL Fractional enable.
//         Fractional mode(FLL_FRACN_ENA = 1) is recommended in all cases.
// Bits[1]:FLL Oscillator enable.
//         FLL_OSC_ENA must be enabled before enabling FLL_ENA.
//         Note that this field is required for free-running FLL modes only.
// Bits[0]:FLL enable.
//         FLL_OSC_ENA must be enabled before enabling FLL_ENA.
//------------------------------------------------------------------
#define WM8904_FLL_CTRL1_ADDR           (0x74)
//FLL Fractional enable options.
typedef enum {
    INTEGER_MODE = 0,
    FRACTIONAL_MODE  = 1
} WM8904_FLL_FRACN_t;
//FLL control1 register structure type definition.
typedef union _wm8904FllCtrl1Reg_t {
    uint16_t regVal;
    struct {
        FunctionalState FLL_ENA          : 1;//FLL enable.
        FunctionalState FLL_OSC_ENA      : 1;//FLL Oscillator enable.
        WM8904_FLL_FRACN_t FLL_FRACN_ENA : 1;//FLL Fractional enable.
        uint16_t reserved                : 13;
    } bits;
} wm8904FllCtrl1Reg_t;

//------------------------------------------------------------------
// Bits[13:8]:FLL FOUT clock divider.
//            FOUT = FVCO / FLL_OUTDIV.
//            Range: 0x03 = 4 ~ 0x3F = 64.
// Bits[6:4] :Frequency of the FLL control block.
//            Recommended that these are not changed from default.
// Bits[2:0] :Fvco clock divider.
//            000 recommended for Fref > 1MHz.
//            100 recommended for Fref < 64kHz.
//------------------------------------------------------------------
#define WM8904_FLL_CTRL2_ADDR           (0x75)
//Frequency of the FLL control block options.
typedef enum {
    FVCO_DIV1 = 0,
    FVCO_DIV2 = 1,
    FVCO_DIV3 = 2,
    FVCO_DIV4 = 3,
    FVCO_DIV5 = 4,
    FVCO_DIV6 = 5,
    FVCO_DIV7 = 6,
    FVCO_DIV8 = 7
} WM8904_FLLCTRLFREQ_t;
//Fvco clock divider options.
typedef enum {
    FVCO_CLK_DIV1  = 0,
    FVCO_CLK_DIV2  = 1,
    FVCO_CLK_DIV4  = 2,
    FVCO_CLK_DIV8  = 3,
    FVCO_CLK_DIV16 = 4
} WM8904_FLLFRATIO_t;
//FLL control2 register structure type definition.
typedef union _wm8904FllCtrl2Reg_t {
    uint16_t regVal;
    struct {
        WM8904_FLLFRATIO_t FLL_FRATIO      : 3;//Fvco clock divider.
        uint16_t reserved0                 : 1;
        WM8904_FLLCTRLFREQ_t FLL_CTRL_RATE : 3;//Frequency of the FLL control block.
        uint16_t reserved1                 : 1;
        uint16_t FLL_OUTDIV                : 6;//FLL FOUT clock divider.
        uint16_t reserved2                 : 2;
    } bits;
} wm8904FllCtrl2Reg_t;

//------------------------------------------------------------------
// Bits[15:0]:Fractional multiply for Fref.(MSB = 0.5)
//------------------------------------------------------------------
#define WM8904_FLL_CTRL3_ADDR           (0x76)
//FLL control3 register structure type definition.
typedef union _wm8904FllCtrl3Reg_t {
    uint16_t regVal;
    struct {
        uint16_t FLL_K;//Fractional multiply for Fref.
    } bits;
} wm8904FllCtrl3Reg_t;

//------------------------------------------------------------------
// Bits[14:5]:Integer multiply for Fref.(LSB = 1)
// Bits[3:0] :FLL Gain applied to error.
//------------------------------------------------------------------
#define WM8904_FLL_CTRL4_ADDR           (0x77)
//FLL Gain applied to error options.
typedef enum {
    FLL_GAIN_x1 = 0,
    FLL_GAIN_x2 = 1,
    FLL_GAIN_x4 = 2,
    FLL_GAIN_x8 = 3,
    FLL_GAIN_x16 = 4,
    FLL_GAIN_x32 = 5,
    FLL_GAIN_x64 = 6,
    FLL_GAIN_x128 = 7,
    FLL_GAIN_x256 = 8
} WM8904_FLLGAIN_t;
//FLL control4 register structure type definition.
typedef union _wm8904FllCtrl4Reg_t {
    uint16_t regVal;
    struct {
        WM8904_FLLGAIN_t FLL_GAIN : 4;//FLL Gain applied to error.
        uint16_t reserved0        : 1;
        uint16_t FLL_N            : 10;//Integer multiply for Fref.
        uint16_t reserved1        : 1;
    } bits;
} wm8904FllCtrl4Reg_t;

//------------------------------------------------------------------
// Bits[4:3]:FLL Clock Reference Divider.
//           MCLK(or other input reference) must be divided down
//           to <= 13.5MHz.
//           For lower power operation, the reference clock can
//           be divided down further is desired.
// Bits[1:0]:FLL Clock source.
//------------------------------------------------------------------
#define WM8904_FLL_CTRL5_ADDR           (0x78)
//FLL Clock Reference Divider options.
typedef enum {
    FLL_MCLK_DIV1 = 0,
    FLL_MCLK_DIV2 = 1,
    FLL_MCLK_DIV4 = 2,
    FLL_MCLK_DIV8 = 3
} WM8904_FLLCLKREFDIV_t;
//FLL Clock source options.
typedef enum {
    FLL_CLKSRC_MCLK = 0,
    FLL_CLKSRC_BCLK = 1,
    FLL_CLKSRC_LRCLK = 2
} WM8904_FLLCLKSRC_t;
//FLL control5 register structure type definition.
typedef union _wm8904FllCtrl5Reg_t {
    uint16_t regVal;
    struct {
        WM8904_FLLCLKSRC_t FLL_CLK_REF_SRC    : 2;//FLL Clock source.
        uint16_t reserved0                    : 1;
        WM8904_FLLCLKREFDIV_t FLL_CLK_REF_DIV : 2;//FLL Clock Reference Divider.
        uint16_t reserved1                    : 11;
    } bits;
} wm8904FllCtrl5Reg_t;

//------------------------------------------------------------------
// Bits[5]  :GPIO pull-up resistor enable.
// Bits[4]  :GPIO pull-down resistor enable.
// Bits[3:0]:GPIO Function Select.
//------------------------------------------------------------------
#define WM8904_GPIO_CTRL1_ADDR           (0x79)
#define WM8904_GPIO_CTRL2_ADDR           (0x7A)
#define WM8904_GPIO_CTRL3_ADDR           (0x7B)
//GPIO Function options.
typedef enum {
    GPIO_INPUTPIN = 0, //GPIO2, GPIO3, GPIO4 = Default.
    GPIO_CLKOUTPUT = 1,//f = SYSCLK / OPCLKDIV.
    GPIO_LOGIC_0 = 2,
    GPIO_LOGIC_1 = 3,
    GPIO_IRQ = 4,      //GPIO1 = Default.
    GPIO_FLL_LOCK = 5,
    GPIO_MIC_DETECT = 6,
    GPIO_MIC_SHORT = 7,
    GPIO_DMIC_CLKOUT = 8,
    GPIO_FLL_CLKOUT = 9
} WM8904_GPIOFUNC_t;
//GPIO Control register structure type definition.
typedef union _wm8904GpioCtrlReg_t {
    uint16_t regVal;
    struct {
        WM8904_GPIOFUNC_t GPIO_SEL : 4;//GPIO function select.
        FunctionalState GPIO_PD    : 1;//GPIO pull-down resistor enable.
        FunctionalState GPIO_PU    : 1;//GPIO pull-up resistor enable.
        uint16_t reserved          : 10;
    } bits;
} wm8904GpioCtrlReg_t;

//------------------------------------------------------------------
// Bits[9]  :GPI7 input enable.
// Bits[8]  :GPI8 input enable.
// Bits[7]  :Selects BCLK/GPIO4 pin function.
// Bits[3:0]:GPIO_BCLK function select.
//------------------------------------------------------------------
#define WM8904_GPIO_CTRL4_ADDR           (0x7C)
//BCLK/GPIO4 pin function options.
typedef enum {
    PIN_BCLK = 0,
    PIN_GPIO4 = 1
} WM8904_GPIO4FUNC_t;
//GPIO Control4 register structure type definition.
typedef union _wm8904GpioCtrl4Reg_t {
    uint16_t regVal;
    struct {
        WM8904_GPIOFUNC_t GPIO_BCLK_SEL       : 4;//GPIO_BCLK function select.
        uint16_t reserved0                    : 3;
        WM8904_GPIO4FUNC_t GPIO_BCLK_MODE_ENA : 1;//Selects BCLK/GPIO4 pin function.
        FunctionalState GPI8_ENA              : 1;//GPI8 input enable.
        FunctionalState GPI7_ENA              : 1;//GPI7 input enable.
    } bits;
} wm8904GpioCtrl4Reg_t;

//------------------------------------------------------------------
// Bits[7]:MCLK pull-up resistor enable.
// Bits[6]:MCLK pull-down resistor enable.
// Bits[5]:DACDAT pull-up resistor enable.
// Bits[4]:DACDAT pull-down resistor enable.
// Bits[3]:LRCLK pull-up resistor enable.
// Bits[2]:LRCLK pull-down resistor enable.
// Bits[1]:BCLK pull-up resistor enable.
// Bits[0]:BCLK pull-down resistor enable.
//------------------------------------------------------------------
#define WM8904_DIG_PULL_ADDR         (0x7E)
//Digital Pulls register structure type definition.
typedef union _wm8904DigPullsReg_t {
    uint16_t regVal;
    struct {
        FunctionalState BCLK_PD   : 1;//BCLK pull-down resistor enable.
        FunctionalState BCLK_PU   : 1;//BCLK pull-up resistor enable.
        FunctionalState LRCLK_PD  : 1;//LRCLK pull-down resistor enable.
        FunctionalState LRCLK_PU  : 1;//LRCLK pull-up resistor enable.
        FunctionalState DACDAT_PD : 1;//DACDAT pull-down resistor enable.
        FunctionalState DACDAT_PU : 1;//DACDAT pull-up resistor enable.
        FunctionalState MCLK_PD   : 1;//MCLK pull-down resistor enable.
        FunctionalState MCLK_PU   : 1;//MCLK pull-up resistor enable.
        uint16_t reserved         : 8;
    } bits;
} wm8904DigPullsReg_t;

//------------------------------------------------------------------
// Bits[10]:Logical OR of all other interrupt flags.
// Bits[9] :GPIO4 interrupt. Cleared when a '1' is written.
// Bits[8] :Write Sequence interrupt. Cleared when a '1' is written.
// Bits[7] :GPIO3 interrupt. Cleared when a '1' is written.
// Bits[6] :GPIO2 interrupt. Cleared when a '1' is written.
// Bits[5] :GPIO1 interrupt. Cleared when a '1' is written.
// Bits[4] :GPI8 interrupt. Cleared when a '1' is written.
// Bits[3] :GPI7 interrupt. Cleared when a '1' is written.
// Bits[2] :FLL Lock interrupt. Cleared when a '1' is written.
// Bits[1] :MICBIAS short circuit interrupt. Cleared when a '1' is written.
// Bits[0] :MICBIAS current detect interrupt. Cleared when a '1' is written.
//------------------------------------------------------------------
#define WM8904_INTR_STATUS_ADDR         (0x7F)
//Interrupt status register structure type definition.
typedef union _wm8904IntrStatusReg_t {
    uint16_t regVal;
    struct {
        FlagStatus MIC_DET_EINT   : 1;//MICBIAS current detect interrupt.
        FlagStatus MIC_SHORT_EINT : 1;//MICBIAS short circuit interrupt.
        FlagStatus FLL_LOCK_EINT  : 1;//FLL Lock interrupt.
        FlagStatus GPI7_EINT      : 1;//GPI7 interrupt.
        FlagStatus GPI8_EINT      : 1;//GPI8 interrupt.
        FlagStatus GPIO1_EINT     : 1;//GPIO1 interrupt.
        FlagStatus GPIO2_EINT     : 1;//GPIO2 interrupt.
        FlagStatus GPIO3_EINT     : 1;//GPIO3 interrupt.
        FlagStatus WSEQ_EINT      : 1;//Write Sequence interrupt.
        FlagStatus GPIO_BCLK_EINT : 1;//GPIO4 interrupt.
        FlagStatus IRQ_OR         : 1;//Logical OR of all other interrupt flags.
        uint16_t reserved         : 7;
    } bits;
} wm8904IntrStatusReg_t;

//------------------------------------------------------------------
// Bits[9]:GPIO4 interrupt mask.
// Bits[8]:Write Sequence interrupt mask.
// Bits[7]:GPIO3 interrupt mask.
// Bits[6]:GPIO2 interrupt mask.
// Bits[5]:GPIO1 interrupt mask.
// Bits[4]:GPI8 interrupt mask.
// Bits[3]:GPI7 interrupt mask.
// Bits[2]:FLL Lock interrupt mask.
// Bits[1]:MICBIAS short circuit interrupt mask.
// Bits[0]:MICBIAS current detect interrupt mask.
//------------------------------------------------------------------
#define WM8904_INTR_MASK_ADDR         (0x80)
//Interrupt mask register structure type definition.
typedef union _wm8904IntrMaskReg_t {
    uint16_t regVal;
    struct {
        FlagStatus IM_MIC_DET_EINT   : 1;//MICBIAS current detect interrupt mask.
        FlagStatus IM_MIC_SHORT_EINT : 1;//MICBIAS short circuit interrupt mask.
        FlagStatus IM_FLL_LOCK_EINT  : 1;//FLL Lock interrupt mask.
        FlagStatus IM_GPI7_EINT      : 1;//GPI7 interrupt mask.
        FlagStatus IM_GPI8_EINT      : 1;//GPI8 interrupt mask.
        FlagStatus IM_GPIO1_EINT     : 1;//GPIO1 interrupt mask.
        FlagStatus IM_GPIO2_EINT     : 1;//GPIO2 interrupt mask.
        FlagStatus IM_GPIO3_EINT     : 1;//GPIO3 interrupt mask.
        FlagStatus IM_WSEQ_EINT      : 1;//Write Sequence interrupt mask.
        FlagStatus IM_GPIO_BCLK_EINT : 1;//GPIO4 interrupt mask.
        uint16_t reserved            : 6;
    } bits;
} wm8904IntrMaskReg_t;

//------------------------------------------------------------------
// Bits[9]:GPIO4 interrupt polarity.
// Bits[8]:Write Sequence interrupt polarity.
// Bits[7]:GPIO3 interrupt polarity.
// Bits[6]:GPIO2 interrupt polarity.
// Bits[5]:GPIO1 interrupt polarity.
// Bits[4]:GPI8 interrupt polarity.
// Bits[3]:GPI7 interrupt polarity.
// Bits[2]:FLL Lock interrupt polarity.
// Bits[1]:MICBIAS short circuit interrupt polarity.
// Bits[0]:MICBIAS current detect interrupt polarity.
//------------------------------------------------------------------
#define WM8904_INTR_POLARITY_ADDR      (0x81)
//Interrupt polarity options.
typedef enum {
    INTR_ACT_HIGH = 0,
    INTR_ACT_LOW = 1
} WM8904_INTRPOLARITY_t;
//Interrupt polarity register structure type definition.
typedef union _wm8904IntrPolarityReg_t {
    uint16_t regVal;
    struct {
        WM8904_INTRPOLARITY_t MIC_DET_EINT_POL   : 1;//MICBIAS current detect interrupt polarity.
        WM8904_INTRPOLARITY_t MIC_SHRT_EINT_POL  : 1;//MICBIAS short circuit interrupt polarity.
        WM8904_INTRPOLARITY_t FLL_LOCK_EINT_POL  : 1;//FLL Lock interrupt polarity.
        WM8904_INTRPOLARITY_t GPI7_EINT_POL      : 1;//GPI7 interrupt polarity.
        WM8904_INTRPOLARITY_t GPI8_EINT_POL      : 1;//GPI8 interrupt polarity.
        WM8904_INTRPOLARITY_t GPIO1_EINT_POL     : 1;//GPIO1 interrupt polarity.
        WM8904_INTRPOLARITY_t GPIO2_EINT_POL     : 1;//GPIO2 interrupt polarity.
        WM8904_INTRPOLARITY_t GPIO3_EINT_POL     : 1;//GPIO3 interrupt polarity.
        WM8904_INTRPOLARITY_t WSEQ_EINT_POL      : 1;//Write Sequence interrupt polarity.
        WM8904_INTRPOLARITY_t GPIO_BCLK_EINT_POL : 1;//GPIO4 interrupt polarity.
        uint16_t reserved                        : 6;
    } bits;
} wm8904IntrPolarityReg_t;

//------------------------------------------------------------------
// Bits[9]:GPIO4 interrupt debounce.
// Bits[8]:Write Sequence interrupt debounce.
// Bits[7]:GPIO3 interrupt debounce.
// Bits[6]:GPIO2 interrupt debounce.
// Bits[5]:GPIO1 interrupt debounce.
// Bits[4]:GPI8 interrupt debounce.
// Bits[3]:GPI7 interrupt debounce.
// Bits[2]:FLL Lock interrupt debounce.
// Bits[1]:MICBIAS short circuit interrupt debounce.
// Bits[0]:MICBIAS current detect interrupt debounce.
//------------------------------------------------------------------
#define WM8904_INTR_DEBOUNCE_ADDR      (0x82)
//Interrupt debounce register structure type definition.
typedef union _wm8904IntrDebounceReg_t {
    uint16_t regVal;
    struct {
        FunctionalState MIC_DET_EINT_DB   : 1;//MICBIAS current detect interrupt debounce.
        FunctionalState MIC_SHRT_EINT_DB  : 1;//MICBIAS short circuit interrupt debounce.
        FunctionalState FLL_LOCK_EINT_DB  : 1;//FLL Lock interrupt debounce.
        FunctionalState GPI7_EINT_DB      : 1;//GPI7 interrupt debounce.
        FunctionalState GPI8_EINT_DB      : 1;//GPI8 interrupt debounce.
        FunctionalState GPIO1_EINT_DB     : 1;//GPIO1 interrupt debounce.
        FunctionalState GPIO2_EINT_DB     : 1;//GPIO2 interrupt debounce.
        FunctionalState GPIO3_EINT_DB     : 1;//GPIO3 interrupt debounce.
        FunctionalState WSEQ_EINT_DB      : 1;//Write Sequence interrupt debounce.
        FunctionalState GPIO_BCLK_EINT_DB : 1;//GPIO4 interrupt debounce.
        uint16_t reserved                 : 6;
    } bits;
} wm8904IntrDebounceReg_t;

//------------------------------------------------------------------
// Bits[0]:EQ enable.
//------------------------------------------------------------------
#define WM8904_EQ_CTRL_ADDR            (0x86)
//EQ enable register structure type definition.
typedef union _wm8904EqCtrlReg_t {
    uint16_t regVal;
    struct {
        FunctionalState EQ_ENA : 1;//EQ enable.
        uint16_t reserved      : 15;
    } bits;
} wm8904EqCtrlReg_t;

//------------------------------------------------------------------
// Bits[4:0]:Gain for specific EQ band.
//------------------------------------------------------------------
#define WM8904_EQBAND1_GAIN_ADDR       (0x87)
#define WM8904_EQBAND2_GAIN_ADDR       (0x88)
#define WM8904_EQBAND3_GAIN_ADDR       (0x89)
#define WM8904_EQBAND4_GAIN_ADDR       (0x8A)
#define WM8904_EQBAND5_GAIN_ADDR       (0x8B)
//EQ band gain register structure type definition.
typedef union _wm8904EqBandGainReg_t {
    uint16_t regVal;
    struct {
        uint16_t EQ_BAND_GAIN : 5;//Gain for specific EQ band.
                                  //0x00 = -12dB
                                  //0x01 = -11dB
                                  //...(1dB steps)
                                  //0x0C = 0dB
                                  //...(1dB steps)
                                  //0x30 = +12dB
                                  //0x31 ~ 0x3F = reserved.
        uint16_t reserved     : 11;
    } bits;
} wm8904EqBandGainReg_t;

//------------------------------------------------------------------
// Bits[15:0]:EQ band coefficient.
//------------------------------------------------------------------
#define WM8904_EQBAND1_A_ADDR          (0x8C)
#define WM8904_EQBAND1_B_ADDR          (0x8D)
#define WM8904_EQBAND1_PG_ADDR         (0x8E)
#define WM8904_EQBAND2_A_ADDR          (0x8F)
#define WM8904_EQBAND2_B_ADDR          (0x90)
#define WM8904_EQBAND2_C_ADDR          (0x91)
#define WM8904_EQBAND2_PG_ADDR         (0x92)
#define WM8904_EQBAND3_A_ADDR          (0x93)
#define WM8904_EQBAND3_B_ADDR          (0x94)
#define WM8904_EQBAND3_C_ADDR          (0x95)
#define WM8904_EQBAND3_PG_ADDR         (0x96)
#define WM8904_EQBAND4_A_ADDR          (0x97)
#define WM8904_EQBAND4_B_ADDR          (0x98)
#define WM8904_EQBAND4_C_ADDR          (0x99)
#define WM8904_EQBAND4_PG_ADDR         (0x9A)
#define WM8904_EQBAND5_A_ADDR          (0x9B)
#define WM8904_EQBAND5_B_ADDR          (0x9C)
#define WM8904_EQBAND5_PG_ADDR         (0x9D)
//EQ band coefficient register structure type definition.
typedef union _wm8904EqBandCffcntReg_t {
    uint16_t regVal;
    struct {
        uint16_t EQ_BAND_CFFCNT;//EQ band coefficient.
    } bits;
} wm8904EqBandCffcntReg_t;

//------------------------------------------------------------------
// Bits[2]:ADC Bias control(1)
//         Set this bit to 1 in ADC 64fs mode(ADC_OSR128 = 0).
//         Set this bit to 0 in ADC 128fs mode(ADC_OSR128 = 1).
// Bits[0]:ADC Bias control(2)
//         Set this bit to 1 in ADC 64fs mode(ADC_OSR128 = 0).
//         Set this bit to 0 in ADC 128fs mode(ADC_OSR128 = 1).
//------------------------------------------------------------------
#define WM8904_ADC_TEST0_REG_ADDR      (0xC6)
//ADC test0 register structure type definition.
typedef union _wm8904AdcTest0Reg_t {
    uint16_t regVal;
    struct {
        FlagStatus ADC_BIASX1P5         : 1;
        uint16_t reserved0              : 1;
        FlagStatus ADC_128_OSR_TST_MODE : 1;
        uint16_t reserved1              : 13;
    } bits;
} wm8904AdcTest0Reg_t;

//------------------------------------------------------------------
// Bits[0]:FLL forced control select.
//------------------------------------------------------------------
#define WM8904_FLL_NCO_TEST0_REG_ADDR  (0xF7)
//FLL forced control options.
typedef enum {
    FRC_NORMAL = 0,
    FRC_NCOVAL = 1
} WM8904_FRC_t;
//FLL NCO test0 register structure type definition.
typedef union _wm8904FllNcoTest0Reg_t {
    uint16_t regVal;
    struct {
        WM8904_FRC_t FLL_FRC_NCO : 1;//FLL forced control select.
        uint16_t reserved        : 15;
    } bits;
} wm8904FllNcoTest0Reg_t;

//------------------------------------------------------------------
// Bits[5:0]:FLL forced oscillator value.
//           Range is 0x00 to 0x3F.
//           0x19 = 12MHz approx.
//------------------------------------------------------------------
#define WM8904_FLL_NCO_TEST1_REG_ADDR  (0xF8)
//FLL NCO test1 register structure type definition.
typedef union _wm8904FllNcoTest1Reg_t {
    uint16_t regVal;
    struct {
        uint16_t FLL_FRC_NCO_VAL : 6;//FLL forced oscillator value.
        uint16_t reserved        : 10;
    } bits;
} wm8904FllNcoTest1Reg_t;


//------------------------------------------------------------------
// Registers attributes and table.
//------------------------------------------------------------------
//Register attribute structure.
typedef struct {
    uint8_t addr; //The register address.
    uint8_t depth;//The register depth. All register is 16bits. 
                  //So this value always =2.
} wm8904RegAttr_t;
//Register attribute table.
static const wm8904RegAttr_t wm8904RegAttrTbl[] = {
    {WM8904_SWRST_ID_REG_ADDR,        sizeof(wm8904SwRstIdReg_t)},
    {WM8904_BIAS_CTRL0_REG_ADDR,      sizeof(wm8904BiasCtrl0Reg_t)},
    {WM8904_VMID_CTRL0_REG_ADDR,      sizeof(wm8904VmidCtrl0Reg_t)},
    {WM8904_MICBIAS_CTRL0_REG_ADDR,   sizeof(wm8904MicBiasCtrl0Reg_t)},
    {WM8904_MICBIAS_CTRL1_REG_ADDR,   sizeof(wm8904MicBiasCtrl1Reg_t)},
    {WM8904_ADC0_REG_ADDR,            sizeof(wm8904AdcReg_t)},
    {WM8904_PWRMANAGE0_REG_ADDR,      sizeof(wm8904PwrManage0Reg_t)},
    {WM8904_PWRMANAGE2_REG_ADDR,      sizeof(wm8904PwrManage2Reg_t)},
    {WM8904_PWRMANAGE3_REG_ADDR,      sizeof(wm8904PwrManage3Reg_t)},
    {WM8904_PWRMANAGE6_REG_ADDR,      sizeof(wm8904PwrManage6Reg_t)},
    {WM8904_CLKRATE0_REG_ADDR,        sizeof(wm8904ClkRate0Reg_t)},
    {WM8904_CLKRATE1_REG_ADDR,        sizeof(wm8904ClkRate1Reg_t)},
    {WM8904_CLKRATE2_REG_ADDR,        sizeof(wm8904ClkRate2Reg_t)},
    {WM8904_AUDIOINTERFACE0_REG_ADDR, sizeof(wm8904AudioInterface0Reg_t)},
    {WM8904_AUDIOINTERFACE1_REG_ADDR, sizeof(wm8904AudioInterface1Reg_t)},
    {WM8904_AUDIOINTERFACE2_REG_ADDR, sizeof(wm8904AudioInterface2Reg_t)},
    {WM8904_AUDIOINTERFACE3_REG_ADDR, sizeof(wm8904AudioInterface3Reg_t)},
    {WM8904_DACDIGVOL_LEFT_REG_ADDR,  sizeof(wm8904DigVolReg_t)},
    {WM8904_DACDIGVOL_RIGHT_REG_ADDR, sizeof(wm8904DigVolReg_t)},
    {WM8904_ADCDIGVOL_LEFT_REG_ADDR,  sizeof(wm8904DigVolReg_t)},
    {WM8904_ADCDIGVOL_RIGHT_REG_ADDR, sizeof(wm8904DigVolReg_t)},
    {WM8904_DACDIG0_REG_ADDR,         sizeof(wm8904SidetoneVolReg_t)},
    {WM8904_DACDIG1_REG_ADDR,         sizeof(wm8904DacDig1Reg_t)},
    {WM8904_ADCDIG0_REG_ADDR,         sizeof(wm8904AdcDig0Reg_t)},
    {WM8904_MICPHONE0_REG_ADDR,       sizeof(wm8904Mic0Reg_t)},
    {WM8904_DRC0_REG_ADDR,            sizeof(wm8904Drc0Reg_t)},
    {WM8904_DRC1_REG_ADDR,            sizeof(wm8904Drc1Reg_t)},
    {WM8904_DRC2_REG_ADDR,            sizeof(wm8904Drc2Reg_t)},
    {WM8904_DRC3_REG_ADDR,            sizeof(wm8904Drc3Reg_t)},
    {WM8904_ANA_LTINPUT0_REG_ADDR,    sizeof(wm8904AnaInput0Reg_t)},
    {WM8904_ANA_RTINPUT0_REG_ADDR,    sizeof(wm8904AnaInput0Reg_t)},
    {WM8904_ANA_LTINPUT1_REG_ADDR,    sizeof(wm8904AnaInput1Reg_t)},
    {WM8904_ANA_RTINPUT1_REG_ADDR,    sizeof(wm8904AnaInput1Reg_t)},
    {WM8904_ANA_HPLTOUTPUT_REG_ADDR,  sizeof(wm8904AnaOutputReg_t)},
    {WM8904_ANA_HPRTOUTPUT_REG_ADDR,  sizeof(wm8904AnaOutputReg_t)},
    {WM8904_ANA_LINELTOUTPUT_REG_ADDR,sizeof(wm8904AnaOutputReg_t)},
    {WM8904_ANA_LINERTOUTPUT_REG_ADDR,sizeof(wm8904AnaOutputReg_t)},
    {WM8904_ANA_OUT12ZC_REG_ADDR,     sizeof(wm8904AnaOut12ZcReg_t)},
    {WM8904_DC_SERVO0_REG_ADDR,       sizeof(wm8904DcServo0Reg_t)},
    {WM8904_DC_SERVO1_REG_ADDR,       sizeof(wm8904DcServo1Reg_t)},
    {WM8904_DC_SERVO2_REG_ADDR,       sizeof(wm8904DcServo2Reg_t)},
    {WM8904_DC_SERVO4_REG_ADDR,       sizeof(wm8904DcServoSnReg_t)},
    {WM8904_DC_SERVO5_REG_ADDR,       sizeof(wm8904DcServoSnReg_t)},
    {WM8904_DC_SERVO6_REG_ADDR,       sizeof(wm8904DcServoDacWrValReg_t)},
    {WM8904_DC_SERVO7_REG_ADDR,       sizeof(wm8904DcServoDacWrValReg_t)},
    {WM8904_DC_SERVO8_REG_ADDR,       sizeof(wm8904DcServoDacWrValReg_t)},
    {WM8904_DC_SERVO9_REG_ADDR,       sizeof(wm8904DcServoDacWrValReg_t)},
    {WM8904_DCSERVO_RDBACK_REG_ADDR,  sizeof(wm8904DcServoRdBackReg_t)},
    {WM8904_ANA_HP0_REG_ADDR,         sizeof(wm8904AnaHpLineReg_t)},
    {WM8904_ANA_LINEOUT0_REG_ADDR,    sizeof(wm8904AnaHpLineReg_t)},
    {WM8904_CHARGE_PUMP0_REG_ADDR,    sizeof(wm8904ChargePump0Reg_t)},
    {WM8904_CLASS_W_REG_ADDR,         sizeof(wm8904ClassWReg_t)},
    {WM8904_WRSEQ0_ADDR,              sizeof(wm8904WrSeq0Reg_t)},
    {WM8904_WRSEQ1_ADDR,              sizeof(wm8904WrSeq1Reg_t)},
    {WM8904_WRSEQ2_ADDR,              sizeof(wm8904WrSeq2Reg_t)},
    {WM8904_WRSEQ3_ADDR,              sizeof(wm8904WrSeq3Reg_t)},
    {WM8904_WRSEQ4_ADDR,              sizeof(wm8904WrSeq4Reg_t)},
    {WM8904_FLL_CTRL1_ADDR,           sizeof(wm8904FllCtrl1Reg_t)},
    {WM8904_FLL_CTRL2_ADDR,           sizeof(wm8904FllCtrl2Reg_t)},
    {WM8904_FLL_CTRL3_ADDR,           sizeof(wm8904FllCtrl3Reg_t)},
    {WM8904_FLL_CTRL4_ADDR,           sizeof(wm8904FllCtrl4Reg_t)},
    {WM8904_FLL_CTRL5_ADDR,           sizeof(wm8904FllCtrl5Reg_t)},
    {WM8904_GPIO_CTRL1_ADDR,          sizeof(wm8904GpioCtrlReg_t)},
    {WM8904_GPIO_CTRL2_ADDR,          sizeof(wm8904GpioCtrlReg_t)},
    {WM8904_GPIO_CTRL3_ADDR,          sizeof(wm8904GpioCtrlReg_t)},
    {WM8904_GPIO_CTRL4_ADDR,          sizeof(wm8904GpioCtrl4Reg_t)},
    {WM8904_DIG_PULL_ADDR,            sizeof(wm8904DigPullsReg_t)},
    {WM8904_INTR_STATUS_ADDR,         sizeof(wm8904IntrStatusReg_t)},
    {WM8904_INTR_MASK_ADDR,           sizeof(wm8904IntrMaskReg_t)},
    {WM8904_INTR_POLARITY_ADDR,       sizeof(wm8904IntrPolarityReg_t)},
    {WM8904_INTR_DEBOUNCE_ADDR,       sizeof(wm8904IntrDebounceReg_t)},
    {WM8904_EQ_CTRL_ADDR,             sizeof(wm8904EqCtrlReg_t)},
    {WM8904_EQBAND1_GAIN_ADDR,        sizeof(wm8904EqBandGainReg_t)},
    {WM8904_EQBAND2_GAIN_ADDR,        sizeof(wm8904EqBandGainReg_t)},
    {WM8904_EQBAND3_GAIN_ADDR,        sizeof(wm8904EqBandGainReg_t)},
    {WM8904_EQBAND4_GAIN_ADDR,        sizeof(wm8904EqBandGainReg_t)},
    {WM8904_EQBAND5_GAIN_ADDR,        sizeof(wm8904EqBandGainReg_t)},        
    {WM8904_EQBAND1_A_ADDR,           sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND1_B_ADDR,           sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND1_PG_ADDR,          sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND2_A_ADDR,           sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND2_B_ADDR,           sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND2_C_ADDR,           sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND2_PG_ADDR,          sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND3_A_ADDR,           sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND3_B_ADDR,           sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND3_C_ADDR,           sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND3_PG_ADDR,          sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND4_A_ADDR,           sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND4_B_ADDR,           sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND4_C_ADDR,           sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND4_PG_ADDR,          sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND5_A_ADDR,           sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND5_B_ADDR,           sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_EQBAND5_PG_ADDR,          sizeof(wm8904EqBandCffcntReg_t)},
    {WM8904_ADC_TEST0_REG_ADDR,       sizeof(wm8904AdcTest0Reg_t)},
    {WM8904_FLL_NCO_TEST0_REG_ADDR,   sizeof(wm8904FllNcoTest0Reg_t)},
    {WM8904_FLL_NCO_TEST1_REG_ADDR,   sizeof(wm8904FllNcoTest1Reg_t)}
};


//------------------------------------------------------------------
// Low power playback mode structure and sequence.
//------------------------------------------------------------------
//Low power playback mode structure.
typedef struct _wm8904LowPwrCfgStruct_t {
    uint8_t  addr;  //The register address.
    uint16_t regVal;//The register value.
} wm8904LowPwrCfgStruct_t;
//Low power playback mode enable sequence.
static const wm8904LowPwrCfgStruct_t wm8904LowPwrEnSeq[] = {
    {0x04, 0x0011},
    {0x08, 0x0019},
    {0xCC, 0x0030},
    {0x5B, 0x0002},
    {0x63, 0x2425},
    {0x64, 0x2B23},
    {0xA1, 0x0002},
    {0x65, 0x00C0},
    {0xA1, 0x0000}
};
//Low power playback mode disable sequence.
static const wm8904LowPwrCfgStruct_t wm8904LowPwrDisSeq[] = {
    {0x04, 0x0019},
    {0x08, 0x0001},
    {0xCC, 0x0000},
    {0x5B, 0x0000},
    {0x63, 0x1F25},
    {0x64, 0x2B19},
    {0xA1, 0x0002},
    {0x65, 0x01C0},
    {0xA1, 0x0000}
};


//Configuration const.
#define FVCO_MIN            ((float)(90000000))  //The minimum value of Fvco is 90MHz.
#define FVCO_MAX            ((float)(100000000)) //The maximum value of Fvco is 100MHz.
#define K_MULTI             (0x10000)            //The K register value is 0.K * 2^16.
#define ADDA_VOLUME_STEP    (0.375)              //The ADC and DAC volume step is 0.375dB.
#define ADDA_VOLUME_MIN     (-71.625)            //The minimum ADC and DAC volume.
#define ADDA_VOLUME_MAX     ((float)(0))         //The maximum ADC and DAC volume.
#define OUTPUT_VOLUME_STEP  (1)                  //The output volume step is 1dB.
#define OUTPUT_VOLUME_MIN   (-57)                //The minimum output volume.
#define OUTPUT_VOLUME_MAX   (6)                  //The maximum output volume.
#define DAC_WR_VAL_LSB      (0.25)               //The DC offset value LSB.

//WM8904 master or slave mode options.
typedef enum {
    WM8904_SLVMODE = 0,
    WM8904_MSTMODE = 1
} WM8904_MST_SLV_t;

//Output port options.
typedef enum {
    OUTPUT_HP   = 0,
    OUTPUT_LINE = 1
} WM8904_OUTPORT_t;

//Stereo options.
typedef enum {
    LEFT_STEREO  = 0,
    RIGHT_STEREO = 1
} WM8904_STEREO_t;

//Input/Output signal options.
typedef enum {
    WM8904_DA = 0,
    WM8904_AD = 1
} WM8904_DA_AD_t;

//Analogue interface options.
typedef enum {
    LEFT_PGA_IN = 0,
    RIGHT_PGA_IN = 1,
    LEFT_HP_OUT = 2,
    RIGHT_HP_OUT = 3,
    LEFT_LINE_OUT = 4,
    RIGHT_LINE_OUT = 5,
    LEFT_DAC = 6,
    RIGHT_DAC = 7,
    LEFT_ADC = 8,
    RIGHT_ADC = 9
} WM8904_ANACHN_t;

//Digital port options.
typedef enum {
    WM8904_BCLK   = 0,
    WM8904_LRCLK  = 1,
    WM8904_DACDAT = 2,
    WM8904_MCLK   = 3
} WM8904_DIGPORT_t;

//Digital port state options.
typedef enum {
    WM8904_OD = 0,
    WM8904_PU = 1,
    WM8904_PD = 2
} WM8904_DIGPORTSTATE_t;

//Interrupt source options.
typedef enum {
    INTR_MIC_DET  = 0,
    INTR_MIC_SHRT = 1,
    INTR_FLL_LOCK = 2,
    INTR_GPI7     = 3,
    INTR_GPI8     = 4,
    INTR_GPIO1    = 5,
    INTR_GPIO2    = 6,
    INTR_GPIO3    = 7,
    INTR_WSEQ     = 8,
    INTR_GPIO4    = 9,
    INTR_IRQ      = 10
} WM8904_INTRSRC_t;

//Bias control parameter structure.
typedef struct _wm8904BiasCtrlParamStruct_t {
    WM8904_BIAS_ISEL_t isel;        //Master bias control.
                                    //@arg LOW_PWR_BIAS.
                                    //@arg HIGH_PERFORM_BIAS.
    FunctionalState ena;            //Enables the normal bias current generator.
                                    //@arg ENABLE.
                                    //@arg DISABLE.
} wm8904BiasCtrlParamStruct_t;

//VMID parameter structure.
typedef struct _wm8904VmidParamStruct_t {
    FunctionalState buf_ena;        //Enable VMID buffer to unused inputs/outputs.
                                    //@arg ENABLE.
                                    //@arg DISABLE.
    VMID_DIV_SEL_t res;             //VMID divider enable and select.
                                    //@arg VMID_DISABLE.
                                    //@arg VMID_DUAL50K_DIV.
                                    //@arg VMID_DUAL250K_DIV.
                                    //@arg VMID_DUAL5K_DIV.
    FunctionalState ena;            //Enable VMID buffer.
                                    //@arg ENABLE.
                                    //@arg DISABLE.
} wm8904VmidParamStruct_t;

//Micphone bias parameter structure.
typedef struct _wm8904MicBiasParamStruct_t {
    FunctionalState mic_bias_ena;       //Mic bias enable.
                                        //@arg ENABLE.
                                        //@arg DISABLE.
    FunctionalState mic_det_ena;        //Mic bias current and short circuit detect enable.
                                        //@arg ENABLE.
                                        //@arg DISABLE.
    WM8904_MICSHORTTHR_t mic_short_thr; //Mic bias short circuit threshold(AVdd = 1.8V).
                                        //@arg THR_0_52MA
                                        //@arg THR_0_88MA
                                        //@arg THR_1_24MA
                                        //@arg THR_1_60MA
    WM8904_MICDETTHR_t mic_det_thr;     //Mic bias current detect threshold(AVdd = 1.8V).
                                        //@arg THR_0_07MA
                                        //@arg THR_0_26MA
                                        //@arg THR_0_45MA
                                        //@arg THR_0_64MA
                                        //@arg THR_0_83MA
                                        //@arg THR_1_02MA
                                        //@arg THR_1_21MA
                                        //@arg THR_1_40MA
    WM8904_MICBIAS_t mic_bias_sel;      //Selects Mic bias voltage.
                                        //@arg AVDD_1_6V
                                        //@arg AVDD_2_0V
                                        //@arg AVDD_2_1V
                                        //@arg AVDD_2_4V
                                        //@arg AVDD_2_7V
} wm8904MicBiasParamStruct_t;

//MCLK parameter structure.
typedef struct _wm8904MclkParamStruct_t {
    WM8904_MCLKDIV_t mclk_div;          //Enables divide by 2 on MCLK.
                                        //@arg MCLK_DIV1.
                                        //@arg MCLK_DIV2.
    WM8904_MCLKINV_t mclk_inv;          //MCLK invert.
                                        //@arg MCLK_NOTINVERTED.
                                        //@arg MCLK_INVERTED.
    //WM8904_SYSCLKSRC_t sysclk_src;      //System clock source.
                                        //@arg SYSCLK_MCLK.
                                        //@arg SYSCLK_FLL.
    //FunctionalState sysclk_ena;         //System clock enable.
                                        //@arg ENABLE.
                                        //@arg DISABLE.
} wm8904MclkParamStruct_t;

//TOCLK parameter structure.
typedef struct _wm8904ToClkParamStruct_t {
    WM8904_TOCLKDIV16_t div16;          //TOCLK rate divider(/16).
                                        //@arg TOCLK16_DIV1.
                                        //@arg TOCLK16_DIV16.
    WM8904_TOCLKMULTI_t multi;          //TOCLK rate multiplier.
                                        //@arg TOCLK_Fx1.
                                        //@arg TOCLK_Fx4.
    WM8904_TOCLKDIV2_t div2;            //TOCLK rate devider.
                                        //@arg TOCLK2_DIV2.
                                        //@arg TOCLK2_DIV1.
} wm8904ToClkParamStruct_t;

//System clock rate and sample rate parameter structure.
typedef struct _wm8904SysClkFsParamStruct_t {
    WM8904_SYSCLKRATE_t sysclk_rate;//sysclk_rate = sysclk / fs.
                                    //@arg SYSCLKRATE64.
                                    //@arg SYSCLKRATE128.
                                    //@arg SYSCLKRATE192.
                                    //@arg SYSCLKRATE256.
                                    //@arg SYSCLKRATE384.
                                    //@arg SYSCLKRATE512.
                                    //@arg SYSCLKRATE768.
                                    //@arg SYSCLKRATE1024.
                                    //@arg SYSCLKRATE1408.
                                    //@arg SYSCLKRATE1536.
    WM8904_SAMPLERATE_t fs;         //Sample rate.
                                    //@arg SAMPLERATE_8KHZ.
                                    //@arg SAMPLERATE_12KHZ.
                                    //@arg SAMPLERATE_16KHZ.
                                    //@arg SAMPLERATE_24KHZ.
                                    //@arg SAMPLERATE_32KHZ.
                                    //@arg SAMPLERATE_48KHZ.
} wm8904SysClkFsParamStruct_t;

//FLL parameter structure.
typedef struct _wm8904FllParamStruct_t {
    WM8904_FLLCLKSRC_t src;         //The reference clock source.
                                    //@arg FLL_CLKSRC_MCLK.
                                    //@arg FLL_CLKSRC_BCLK.
                                    //@arg FLL_CLKSRC_LRCLK.
    uint32_t freq_Hz;               //The input frequence.
    uint32_t fs_Hz;                 //The sample rate.
    WM8904_SYSCLKRATE_t ratio;      //This is MCLK / fs. The typical values are shown below:
                                    //@arg SYSCLKRATE64.
                                    //@arg SYSCLKRATE128.
                                    //@arg SYSCLKRATE192.
                                    //@arg SYSCLKRATE256.
                                    //@arg SYSCLKRATE384.
                                    //@arg SYSCLKRATE512.
                                    //@arg SYSCLKRATE768.
                                    //@arg SYSCLKRATE1024.
                                    //@arg SYSCLKRATE1408.
                                    //@arg SYSCLKRATE1536.
} wm8904FllParamStruct_t;

//I2S parameter structure.
typedef struct _wm8904I2sParamStruct_t {
    WM8904_MST_SLV_t mode;              //The I2S worked in master mode or slave mode.
                                        //@arg WM8904_SLVMODE.
                                        //@arg WM8904_MSTMODE.
    WM8904_AUDIOWORDLENGTH_t wl;        //Digital word length.
                                        //@arg WL_16BITS.
                                        //@arg WL_20BITS.
                                        //@arg WL_24BITS.
                                        //@arg WL_32BITS.
    WM8904_BCLKFREQ_t bclk_div;         //The divider of BCLK(Master mode).
    WM8904_BCLKINV_t bclk_inv;          //The polarity of BCLK.
                                        //@arg BCLK_NOTINVERT.
                                        //@arg BCLK_INVERT.
    uint16_t lrclk_div;                 //The divider of LRCLK(Master mode).
                                        //LRCLK output = BCLK / lrclk_div.
                                        //Valid range: 8 to 2047.
    WM8904_LRCLKINV_t lrclk_inv;        //The polarity of LRCLK.
                                        //@arg LRC_NOTINVERT.
                                        //@arg LRC_INVERT.
    FunctionalState comp_ena;           //ADC/DAC companding enable.
                                        //@arg ENABLE.
                                        //@arg DISABLE.
    WM8904_COMPANDINGTYPE_t comp_type;  //ADC/DAC companding type.
                                        //@arg U_LAW.
                                        //@arg A_LAW.
    FunctionalState loopback_ena;       //Loopback enable.
                                        //@arg ENABLE.
                                        //@arg DISABLE.
    WM8904_DACINBOOST_t dac_boost;      //DAC digital input volume boost.
                                        //@arg DAC_BOOST_0DB.
                                        //@arg DAC_BOOST_6DB.
                                        //@arg DAC_BOOST_12DB.
                                        //@arg DAC_BOOST_18DB.
} wm8904I2sParamStruct_t;

//DAC soft mute parameter structure.
typedef struct _wm8904DacSoftMuteParamStruct_t {
    WM8904_DACUNMUTERAMP_t unmute_ramp; //Change immediately or ramp up gradually.
                                        //@arg CHANGE_IMMEDIATELY Cause the DAC volume to change immediately to VOL setting.
                                        //@arg RAMPUP_GRADUALLY   Cause the DAC volume to ramp up gradually to VOL setting.
    WM8904_DACMUTERATE_t mute_rate;     //Ramp up fast or slow.
                                        //@arg FAST_RAMP  Ramp up fast.
                                        //@arg SLOW_RAMP  Ramp up slow.
} wm8904DacSoftMuteParamStruct_t;

//AD DA volume parameter structure.
typedef struct _wm8904AdDaVolParamStruct_t {
    WM8904_DA_AD_t io;              //The input or output line.
                                    //@arg WM8904_DA.
                                    //@arg WM8904_AD.
    WM8904_STEREO_t stereo;         //The left or right stereo.
                                    //@arg LEFT_STEREO.
                                    //@arg RIGHT_STEREO.
    float vol_dB;                   //The volume. The minimum volume is -71.625dB and the maximum is 0dB.
                                    //The volume step is 0.375dB.
                                    //This parameter is the real volume such as -71.625.
} wm8904AdDaVolParamStruct_t;

//Analogue output volume parameter structure.
typedef struct _wm8904AnaOutVolParamStruct_t {
    WM8904_OUTPORT_t port;          //Output port.
                                    //@arg OUTPUT_HP.
                                    //@arg OUTPUT_LINE.
    WM8904_STEREO_t stereo;         //Output stereo.
                                    //@arg LEFT_STEREO.
                                    //@arg RIGHT_STEREO.
    WM8904_HP_LINE_MUTE_t mute;     //Output mute or un-mute.
                                    //@arg OUTPUT_NOT_MUTE.
                                    //@arg OUTPUT_MUTE.
    FlagStatus vu;                  //Update output volume.
                                    //@arg RESET.
                                    //@arg SET.
    FunctionalState zc;             //Output zero-cross enable or disable.
                                    //@arg ENABLE.
                                    //@arg DISABLE.
    int16_t vol_dB;                 //Output volume.
                                    //Range is -57dB to +6dB with 1dB steps.
                                    //This parameter is the real volume such as -57.
} wm8904AnaOutVolParamStruct_t;

//Analogue output bypass parameter structure.
typedef struct _wm8904AnaOutBypParamStruct_t {
    WM8904_OUTPORT_t port;          //Output port.
                                    //@arg OUTPUT_HP.
                                    //@arg OUTPUT_LINE.
    WM8904_STEREO_t stereo;         //Output stereo.
                                    //@arg LEFT_STEREO.
                                    //@arg RIGHT_STEREO.
    WM8904_INSRC_TO_OUTMUX_t src;   //The input for output MUX.
                                    //@arg INPUT_DAC.
                                    //@arg INPUT_PGA.
} wm8904AnaOutBypParamStruct_t;

//DC servo parameter structure.
typedef struct _wm8904DcServoParamStruct_t {
    WM8904_OUTPORT_t port;          //Output port.
                                    //@arg OUTPUT_HP.
                                    //@arg OUTPUT_LINE.
    WM8904_STEREO_t stereo;         //Output stereo.
                                    //@arg LEFT_STEREO.
                                    //@arg RIGHT_STEREO.
    FlagStatus trig_start_up;       //If SET, the output will be within 1mV of AGND. 
                                    //This operation takes 86ms per channel.
                                    //@arg SET.
                                    //@arg RESET.
    FlagStatus trig_dac_wr;         //If SET, the DC offset correction to be set to the value of DCS_DAC_WR_VAL_n fields(R73(0x49) ~ R76(0x4C)).
                                    //This operation takes 2ms per channel.
                                    //@arg SET.
                                    //@arg RESET.
    float dac_wr_val;               //This is the DC offset corrention value.
                                    //Step:  0.25mV.
                                    //Range: +/-32mV
    FlagStatus trig_single;         //If SET, initiates a single DC offset measurement and adjustment.
    FlagStatus trig_series;         //If SET, initiates a series DC offset measurement and adjustment.
    uint8_t series_no;              //The number of DC servo operations.
                                    //Range: 1 ~ 128 updates.
    WM8904_DCS_TIMPERIOD_t period;  //Time between periodic updates.
                                    //Time is calculated as 0.256s x (2^PERIOD).
} wm8904DcServoParamStruct_t;

//Write sequence parameter structure.
typedef struct _wm8904CfgWseqParamStruct_t {
    uint8_t wseq_wr_index;          //Sequence write index. This is the memory location.
                                    //Range: 0 ~ 31.
    uint8_t reg_addr;               //Register address.
    WM8904_WRSEQ_WIDTH_t width;     //The data width.
                                    //@arg WRSEQ_WIDTH_1BIT.
                                    //@arg WRSEQ_WIDTH_2BITS.
                                    //@arg WRSEQ_WIDTH_3BITS.
                                    //@arg WRSEQ_WIDTH_4BITS.
                                    //@arg WRSEQ_WIDTH_5BITS.
                                    //@arg WRSEQ_WIDTH_6BITS.
                                    //@arg WRSEQ_WIDTH_7BITS.
                                    //@arg WRSEQ_WIDTH_8BITS.
    WM8904_WRSEQ_BITPOS_t start;    //The data start bit.
                                    //@arg WRSEQ_POS_BIT0.
                                    //@arg WRSEQ_POS_BIT1.
                                    //@arg WRSEQ_POS_BIT2.
                                    //@arg WRSEQ_POS_BIT3.
                                    //@arg WRSEQ_POS_BIT4.
                                    //@arg WRSEQ_POS_BIT5.
                                    //@arg WRSEQ_POS_BIT6.
                                    //@arg WRSEQ_POS_BIT7.
                                    //@arg WRSEQ_POS_BIT8.
                                    //@arg WRSEQ_POS_BIT9.
                                    //@arg WRSEQ_POS_BIT10.
                                    //@arg WRSEQ_POS_BIT11.
                                    //@arg WRSEQ_POS_BIT12.
                                    //@arg WRSEQ_POS_BIT13.
                                    //@arg WRSEQ_POS_BIT14.
                                    //@arg WRSEQ_POS_BIT15.
    uint8_t data;                   //The register value.
    uint8_t delay;                  //Time delay after executing this step.
                                    //Total delay time per step(including execution)
                                    //= 62.5us x (2^WSEQ_DELAY + 8).
    WM8904_SEQENDFLG_t eos;         //End of Sequence flag.
                                    //@arg SEQ_NOT_END.
                                    //@arg SEQ_END.
} wm8904CfgWseqParamStruct_t;

//Digital pulls parameter structure.
typedef struct _wm8904DigPullParamStruct_t {
    WM8904_DIGPORT_t port;          //The digital port.
                                    //@arg WM8904_BCLK.
                                    //@arg WM8904_LRCLK.
                                    //@arg WM8904_DACDAT.
                                    //@arg WM8904_MCLK.
    WM8904_DIGPORTSTATE_t state;    //The digital port state.
                                    //@arg WM8904_OD
                                    //@arg WM8904_PU.
                                    //@arg WM8904_PD.
} wm8904DigPullParamStruct_t;

//Interrupt parameter structure.
typedef struct _wm8904IntrParamStruct_t {
    WM8904_INTRSRC_t src;           //Interrupt source.
                                    //@arg INTR_MIC_DET
                                    //@arg INTR_MIC_SHRT
                                    //@arg INTR_FLL_LOCK
                                    //@arg INTR_GPI7
                                    //@arg INTR_GPI8
                                    //@arg INTR_GPIO1
                                    //@arg INTR_GPIO2
                                    //@arg INTR_GPIO3
                                    //@arg INTR_WSEQ
                                    //@arg INTR_GPIO4
    FunctionalState ena;            //Mask the interrupt or not.
                                    //@arg ENABLE.
                                    //@arg DISABLE.
    WM8904_INTRPOLARITY_t polarity; //The interrupt polarity.
                                    //@arg INTR_ACT_HIGH.
                                    //@arg INTR_ACT_LOW.
    FunctionalState debounce;       //The interrupt debounce.
                                    //@arg ENABLE.
                                    //@arg DISABLE.
} wm8904IntrParamStruct_t;


//APIs
void WM8904_SoftRst(void);
FlagStatus WM8904_VerifyId(void);
void WM8904_CfgBiasCtrl(wm8904BiasCtrlParamStruct_t* wm8904BiasCtrlParamStruct);
void WM8904_CfgVmidCtrl(wm8904VmidParamStruct_t* wm8904VmidParamStruct);
void WM8904_CfgMicBias(wm8904MicBiasParamStruct_t* wm8904MicBiasParamStruct);
void WM8904_AnaChnCmd(WM8904_ANACHN_t chn, FunctionalState status);
void WM8904_CfgMclk(wm8904MclkParamStruct_t* wm8904MclkParamStruct);
void WM8904_DspClkCmd(FunctionalState status);
void WM8904_GpioClkOutputCmd(FunctionalState status);
void WM8904_ToClkCmd(FunctionalState status);
void WM8904_CfgToClk(wm8904ToClkParamStruct_t* wm8904ToClkParamStruct);
void WM8904_SysClkCmd(FunctionalState status);
void WM8904_CfgSysClkSrc(WM8904_SYSCLKSRC_t src);
void WM8904_CfgSysClkRateAndFs(wm8904SysClkFsParamStruct_t* wm8904SysClkFsParamStruct);
void WM8904_FllCmd(FunctionalState status);
void WM8904_CfgFllOutFreq(wm8904FllParamStruct_t* wm8904FllParamStruct);
void WM8904_CfgI2sAudioInterface(wm8904I2sParamStruct_t* i2sParamStruct);
void WM8904_DacMuteCmd(FunctionalState status);
void WM8904_CfgDacSoftMute(wm8904DacSoftMuteParamStruct_t* wm8904DacSoftMuteParamStruct);
void WM8904_CfgAdDaVol(wm8904AdDaVolParamStruct_t* wm8904AdDaVolParamStruct);
void WM8904_DacMonoMixCmd(FunctionalState status);
void WM8904_DacSlopingStopBandFiltCmd(FunctionalState status);
void WM8904_CfgDacDeEmphasis(WM8904_DEEMPH_t de_emph);
void WM8904_DacSlopingStopBandFiltCmd(FunctionalState status);
void WM8904_CfgDacOsr128(WM8904_DACOSR_t osr_mode);
void WM8904_CfgAnaOutputVol(wm8904AnaOutVolParamStruct_t* wm8904AnaOutVolParamStruct);
void WM8904_CfgOutputByp(wm8904AnaOutBypParamStruct_t* wm8904AnaOutBypParamStruct);
void WM8904_DcServoCmd(WM8904_OUTPORT_t port, WM8904_STEREO_t stereo, FunctionalState status);
FlagStatus WM8904_GetDcServoStatus(WM8904_OUTPORT_t port, WM8904_STEREO_t stereo);
void WM8904_CfgDcServo(wm8904DcServoParamStruct_t* wm8904DcServoParamStruct);
void WM8904_InputStageCmd(WM8904_OUTPORT_t port, WM8904_STEREO_t stereo, FunctionalState status);
void WM8904_DelayStageCmd(WM8904_OUTPORT_t port, WM8904_STEREO_t stereo, FunctionalState status);
void WM8904_OutputStageCmd(WM8904_OUTPORT_t port, WM8904_STEREO_t stereo, FunctionalState status);
void WM8904_RmvShortCmd(WM8904_OUTPORT_t port, WM8904_STEREO_t stereo, WM8904_RMSHORT_t status);
void WM8904_ChargePumpCmd(FunctionalState status);
void WM8904_CfgChargePumpClass(WM8904_DYNPWRCTRL_t dyn_class);
WM8904_SEQBSY_t WM8904_GetWseqStatus(void);
uint16_t WM8904_GetWseqCurrentIndex(void);
void WM8904_CfgWseq(wm8904CfgWseqParamStruct_t* wm8904CfgWseqParamStruct);
void WM8904_WseqSta(uint16_t index);
void WM8904_WseqIntr(void);
void WM8904_CfgDigPortPulls(wm8904DigPullParamStruct_t* wm8904DigPullParamStruct);
FlagStatus WM8904_GetIntrFlg(WM8904_INTRSRC_t src);
void WM8904_ClrIntrFlg(WM8904_INTRSRC_t src);
void WM8904_CfgIntrFlg(wm8904IntrParamStruct_t* wm8904IntrParamStruct);


#ifdef __cplusplus
		}//extern "C" 
#endif

#endif

/************************ (C) COPYRIGHT ART *****END OF FILE****/
