/*********************************************************************************************************
*                                             Requests
*                             COPYRIGHT(c) 2019 Amped RF Technology (ART)
*                                        All Rights Reserved
*
* File    : requests.h
* Version : V1.0.0
* Brief   : This file contains all the function prototypes for web requests.
**********************************************************************************************************/
#ifndef _REQUESTS_H
#define _REQUESTS_H

#include <stdint.h>

#ifdef __cplusplus
	extern "C" {
#endif

//Requests options.
#define REQUESTS_OPT_TLS_ENABLE         (1)
#define REQUESTS_OPT_RECV_BUF_SIZE      (1024)
//TLS options.
#if REQUESTS_OPT_TLS_ENABLE
#define REQUESTS_OPT_TLS_VERIFY_CERT    (0)
#define REQUESTS_OPT_TLS_SUPPORT_H2     (0)
#endif

//Request parameters type.
typedef struct _RequestParam_t {
    const char *key;
    const char *val;
} RequestParam_t;
#define REQPARAMLEN(iv)  (sizeof(iv) / sizeof(iv[0]))

//Request post data callback type.
typedef int (*post_data_callback_t)(void *requests, uint8_t *buf, int len, void *usr_data);
//Before send data callback type.
typedef int (*before_send_callback_t)(void *requests, void *usr_data);
//After send data callback type.
typedef int (*after_send_callback_t)(void *requests, void *usr_data);
//Before receive data callback type.
typedef int (*before_recv_callback_t)(void *requests, void *usr_data);
//After receive data callback type.
typedef int (*after_recv_callback_t)(void *requests, void *usr_data);
//Receive data callback type.
typedef int (*recv_data_callback_t)(void *requests, const uint8_t *buf, const int len, void *usr_data);
//Receive header callback type.
typedef int (*recv_hdr_callback_t)(void *requests, const uint8_t *buf, const int len, void *usr_data);
//Receive body callback type.
typedef int (*recv_bdy_callback_t)(void *requests, const uint8_t *buf, const int len, void *usr_data);
//Send error callback type.
typedef int (*send_err_callback_t)(void *requests, void *usr_data);
//Receive error callback type.
typedef int (*recv_err_callback_t)(void *requests, void *usr_data);

#if REQUESTS_OPT_TLS_ENABLE
//Close notify callback type.
typedef int (*close_notify_callback_t)(void *requests, void *usr_data);
//Receive EOF callback type.
typedef int (*recv_eof_callback_t)(void *requests, void *usr_data);
#endif

//Request control information.
typedef struct _RequestCtrlInfo_t {
    void *usr_data;
    const char *protocol;
    const char *host;
    const char *path;
    const char *port;
    const char *send_content;
    const char *recv_content;
    int send_content_len;
    int recv_content_len;
    int sockfd;
    int (*get)(struct _RequestCtrlInfo_t *requests, const char *url, uint16_t port, RequestParam_t *param, int len, void* opt, void *usr_data);
    int (*post)(struct _RequestCtrlInfo_t *requests, const char *url, uint16_t port, post_data_callback_t callback, void* opt, void *usr_data);
    int status_code;            //Response status code.
    const char *content_length; //Content-Length.
    const char *content_type;   //Content-Type.
    before_send_callback_t  before_send_callback;
    after_send_callback_t   after_send_callback;
    before_recv_callback_t  before_recv_callback;
    after_recv_callback_t   after_recv_callback;
    recv_data_callback_t    recv_data_callback;
    recv_hdr_callback_t     recv_hdr_callback;
    recv_bdy_callback_t     recv_bdy_callback;
    send_err_callback_t     send_err_callback;
    recv_err_callback_t     recv_err_callback;
#if REQUESTS_OPT_TLS_ENABLE
    close_notify_callback_t close_notify_callback;
    recv_eof_callback_t     recv_eof_callback;
#endif
} RequestCtrlInfo_t;

//APIs.
RequestCtrlInfo_t* xRequestCreate(void);
void vRequestDelete(RequestCtrlInfo_t *requests);
void vRequestMountBeforeSendCallBack(RequestCtrlInfo_t *requests, before_send_callback_t callback);
void vRequestMountAfterSendCallBack(RequestCtrlInfo_t *requests, after_send_callback_t callback);
void vRequestMountBeforeRecvCallBack(RequestCtrlInfo_t *requests, before_recv_callback_t callback);
void vRequestMountAfterRecvCallBack(RequestCtrlInfo_t *requests, after_recv_callback_t callback);
void vRequestMountRecvDataCallBack(RequestCtrlInfo_t *requests, recv_data_callback_t callback);
void vRequestMountRecvHdrCallBack(RequestCtrlInfo_t *requests, recv_hdr_callback_t callback);
void vRequestMountRecvBdyCallBack(RequestCtrlInfo_t *requests, recv_bdy_callback_t callback);
void vRequestMountSendErrCallBack(RequestCtrlInfo_t *requests, send_err_callback_t callback);
void vRequestMountRecvErrCallBack(RequestCtrlInfo_t *requests, recv_err_callback_t callback);
#if REQUESTS_OPT_TLS_ENABLE
void vRequestMountCloseNotifyCallBack(RequestCtrlInfo_t *requests, close_notify_callback_t callback);
void vRequestMountRecvEofCallBack(RequestCtrlInfo_t *requests, recv_eof_callback_t callback);
#endif
int iTcpOpen(RequestCtrlInfo_t *requests, const char *host, const char *port);
void vTcpClose(RequestCtrlInfo_t *requests);
void vTcpMount(RequestCtrlInfo_t *requests, const uint8_t *buf, const int len);
void vTcpUnmount(RequestCtrlInfo_t *requests);
int iTcpSend(RequestCtrlInfo_t *requests);
int iTcpRecv(RequestCtrlInfo_t *requests);
#if REQUESTS_OPT_TLS_ENABLE
#include "mbedtls/config.h"
#include "mbedtls/platform.h"
#include "mbedtls/net_sockets.h"
#include "mbedtls/debug.h"
#include "mbedtls/ssl.h"
#include "mbedtls/entropy.h"
#include "mbedtls/ctr_drbg.h"
#include "mbedtls/error.h"
#include "mbedtls/certs.h"

int iTlsOpen(RequestCtrlInfo_t *requests, const char *host, const char *port, 
             mbedtls_net_context **server_fd, mbedtls_entropy_context **entropy, mbedtls_ctr_drbg_context **ctr_drbg, 
             mbedtls_ssl_context **ssl, mbedtls_ssl_config **conf, mbedtls_x509_crt **cacert);
void vTlsClose(RequestCtrlInfo_t *requests, mbedtls_net_context **server_fd, mbedtls_entropy_context **entropy, 
               mbedtls_ctr_drbg_context **ctr_drbg, mbedtls_ssl_context **ssl, mbedtls_ssl_config **conf, 
               mbedtls_x509_crt **cacert);
void vTlsMount(RequestCtrlInfo_t *requests, const uint8_t *buf, const int len);
void vTlsUnmount(RequestCtrlInfo_t *requests);
int iTlsSend(RequestCtrlInfo_t *requests, mbedtls_ssl_context *ssl);
int iTlsRecv(RequestCtrlInfo_t *requests, mbedtls_ssl_context *ssl);
#endif  //endif REQUESTS_OPT_TLS_ENABLE

#ifdef __cplusplus
	}//extern "C"
#endif

#endif

/******************************** (C) COPYRIGHT ART ****END OF FILE****/

