/*********************************************************************************************************
*                                  HTTP/2 C Library API for nghttp2
*                             COPYRIGHT(c) 2018 Amped RF Technology (ART)
*                                        All Rights Reserved
*
* File    : h2_lib.h
* Version : V1.0.0
* Brief   : This file contains all the function prototypes for http2 communication.
**********************************************************************************************************/
#ifndef _H2_LIB_H
#define _H2_LIB_H

#include <stdint.h>
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"
#include "nghttp2/nghttp2.h"
#include "mbed_lib.h"
#include "streamQueue.h"

#ifdef __cplusplus
	extern "C" {
#endif

//Return an array length.
#define ARRLEN(x) (sizeof(x) / sizeof(x[0]))
//Used to construct a header table
#define NAME_VALUE(NAME, VALUE)                 \
        {                                       \
            (uint8_t *)NAME, (uint8_t *)VALUE,  \
            sizeof(NAME) - 1, strlen(VALUE),    \
            NGHTTP2_NV_FLAG_NONE                \
        }

//Http2 received data package type enum.
typedef enum{
    H2_RX_HDR = 0, 
    H2_RX_BDY
} H2_RX_PKG_TYPE_e;

//Http2 session control information.
typedef struct _H2SessionCtrlInfo_t {
    mbed_ctx_t mctx;
    nghttp2_session *session;
    volatile bool h2RxStreamFlowCtrlStatus;  //Http2 receive stream flow control status.
    volatile bool h2RxHdrFlowCtrlStatus;     //Http2 receive header flow control status.
    volatile bool h2RxBdyFlowCtrlStatus;     //Http2 receive body flow control status.
    xSemaphoreHandle h2RxStreamFlowCtrlSem;  //Http2 receive stream flow control semaphore.
    xSemaphoreHandle h2RxHdrFlowCtrlSem;     //Http2 receive header flow control semaphore.
    xSemaphoreHandle h2RxBdyFlowCtrlSem;     //Http2 receive body flow control semaphore.
} H2SessionCtrlInfo_t;

//Http2 received header structure.
typedef struct _H2RxHdrDataStruct_t {
    char* key;
    uint32_t keyLen;
    char* value;
    uint32_t valueLen;
} H2RxHdrDataStruct_t;

//Http2 received data package attribute.
typedef struct _H2RxPkgAttr_t {
    streamNode_t node;
    streamBlock_t* block;
    H2_RX_PKG_TYPE_e pkgType;
} H2RxPkgAttr_t;

//Http2 stream control information.
typedef struct _H2RxStreamCtrlInfo_t {
    uint32_t streamId;
    streamList_t hdrList;
    streamList_t bdyList;
    streamBlock_t* block;
    streamQueue_t* webQueue;
    streamQueue_t* hdrQueue;
    streamQueue_t* bdyQueue;
    void* usr_data;
} H2RxStreamCtrlInfo_t;

//Http2 received header control information.
typedef struct _H2RxHdrCtrlInfo_t {
    H2RxPkgAttr_t pkgAttr;
    uint32_t streamId;
    H2RxHdrDataStruct_t data;
} H2RxHdrCtrlInfo_t;

//Http2 received body control information.
typedef struct _H2RxBdyCtrlInfo_t {
    H2RxPkgAttr_t pkgAttr;
    uint32_t streamId;
} H2RxBdyCtrlInfo_t;

//Http2 update window size information.
typedef struct _H2UpdateWinInfo_t {
    int id;
    uint32_t size;
} H2UpdateWinInfo_t;

//Variables.
extern H2SessionCtrlInfo_t h2SessionCtrlInfo;

//APIs.
H2RxStreamCtrlInfo_t* h2RxStreamCtrlAlloc(uint32_t id, void* usr_data, void* param);
void h2RxStreamCtrlFree(H2RxStreamCtrlInfo_t* info, void* param);
void h2RxStreamCtrlFreeAll(void *param);
uint32_t h2RxStreamCtrlGetWorkCnt(void *param);
H2RxStreamCtrlInfo_t* h2RxStreamCtrlSearch(uint32_t id, void* param);
H2RxHdrCtrlInfo_t* h2RxHdrCtrlAlloc(uint32_t id, void* param);
void h2RxHdrCtrlFree(H2RxHdrCtrlInfo_t* hdrInfo, void* param);
H2RxBdyCtrlInfo_t* h2RxBdyCtrlAlloc(uint32_t id, void* param);
void h2RxBdyCtrlFree(H2RxBdyCtrlInfo_t* bdyInfo, void* param);
void mutex_init (SemaphoreHandle_t  *mutex);
void mutex_free (SemaphoreHandle_t  *mutex);
int  mutex_lock (SemaphoreHandle_t *mutex);
int  mutex_unlock (SemaphoreHandle_t * mutex);
void h2SessionCtrlInfoInit(H2SessionCtrlInfo_t *info);
int h2_start(char *host);
int h2_stop(void);
int h2_update(H2UpdateWinInfo_t *info);
int h2_ping(void);
int h2_send_data(nghttp2_nv *hdr, uint8_t hdr_len, nghttp2_data_provider *bdy, void *stream_usr_data);
int h2_recv(void);

#ifdef __cplusplus
	}//extern "C"
#endif

#endif

/******************************** (C) COPYRIGHT ART ****END OF FILE****/

