#ifndef __SDK_SIGNAL_MNT_H__
#define __SDK_SIGNAL_MNT_H__

#ifndef CONFIG_ENABLE_SDK_SIGNAL
#define CONFIG_ENABLE_SDK_SIGNAL 1
#endif

#if (CONFIG_ENABLE_SDK_SIGNAL == 1)

#include "FreeRTOS.h"
#include "queue.h"

// =============================================================================
// Signal Definitions
// =============================================================================

/**
 * @brief SDK Signal IDs
 * @note  Add specific signals to this enum as needed.
 */
typedef enum {
    SIG_SDK_NONE = 0,
    
    // System Signals
    SIG_SDK_SYS_INIT_DONE,
    
    // WLAN Signals
    SIG_SDK_WLAN_CONNECTED,
    SIG_SDK_WLAN_DISCONNECTED,

    // Sta Join Signals
    SIG_SDK_JOIN_SUCESS,
    SIG_SDK_JOIN_FAILED,

    // IP addr Signals
    SIG_SDK_WLAN_IP_CHANGED,//after DHCP
    
    // UART Signals
    SIG_SDK_UART_DATA_RX,

    // MQTT Signals
    SIG_SDK_MQTT_DATA_UP,   // Uplink data from user app
    SIG_SDK_MQTT_DATA_DOWN, // Downlink data from network

    SIG_SDK_MQTT_CONNECT_OK,
    SIG_SDK_MQTT_DISCONNECT,
    
    // ... Add more signals here ...

    SIG_SDK_MAX
} sdk_signal_id_t;

/**
 * @brief Subscription Modes
 */
typedef enum {
    /**
     * @brief Repeat Mode:
     * The signal is delivered to the subscriber AND continues to the default handler (e.g., AT Task).
     * Used for monitoring events without interfering with system behavior.
     */
    SDK_SUB_MODE_REPEAT = 0,

    /**
     * @brief Redirect Mode:
     * The signal is intercepted and delivered ONLY to the subscriber. 
     * The default handler will NOT receive this signal.
     * Used when the user application takes full control of the event.
     */
    SDK_SUB_MODE_REDIRECT = 1

} sdk_sub_mode_t;

/**
 * @brief Signal Source Definitions
 */
typedef enum {
    SDK_SIGNAL_SRC_PHYSICAL_UART = 0, // External UART physical input
    SDK_SIGNAL_SRC_INTERNAL_APP  = 1, // Internal API/Application call
    SDK_SIGNAL_SRC_REMOTE_NET    = 2  // Remote network input (e.g. Telnet/Web)
} sdk_signal_src_t;

/**
 * @brief Structure of a Signal Message
 */
typedef struct {
    uint16_t id;        // Signal ID (from sdk_signal_id_t)
    uint8_t  source;    // Source of the signal (sdk_signal_src_t)
    void     *data;     // Pointer to payload data (optional)
    uint16_t data_len;  // Length of payload data (optional)
} sdk_signal_t;

// =============================================================================
// Public API
// =============================================================================

/**
 * @brief Initialize the SDK Signal Management System.
 * 
 * Creates the global dispatcher queue, mutex, and starts the dispatcher task.
 * Should be called during system initialization.
 * 
 * @return BaseType_t pdPASS on success, pdFAIL otherwise.
 */
BaseType_t sdk_signal_init(void);

/**
 * @brief Subscribe to a specific signal.
 * 
 * Allocates a subscription node from the system heap and adds it to the registry.
 * 
 * @param signal_id     The ID of the signal to subscribe to.
 * @param xTargetQueue  The FreeRTOS queue where the signal should be delivered.
 * @param mode          Subscription mode (REPEAT or REDIRECT).
 * @return BaseType_t   pdPASS on success, pdFAIL on error (e.g., out of memory).
 */
BaseType_t wlan_sdk_subscribe(uint16_t signal_id, QueueHandle_t xTargetQueue, sdk_sub_mode_t mode);

/**
 * @brief Unsubscribe from a specific signal.
 * 
 * Finds the subscription node matching the signal_id and target queue, 
 * removes it from the registry, and frees the memory.
 * 
 * @param signal_id     The ID of the signal to unsubscribe from.
 * @param xTargetQueue  The target queue associated with the subscription.
 * @return BaseType_t   pdPASS on success, pdFAIL if not found.
 */
BaseType_t wlan_sdk_unsubscribe(uint16_t signal_id, QueueHandle_t xTargetQueue);

/**
 * @brief Post a signal to the SDK Dispatcher.
 * 
 * This function is typically called by drivers, protocol stacks, or user tasks
 * to generate an event. The signal is sent to the global dispatch queue.
 * 
 * @param signal_id     The ID of the signal.
 * @param pData         Pointer to the payload data (can be NULL).
 * @param len           Length of the payload data (can be 0).
 * @return BaseType_t   pdPASS on success, pdFAIL if the queue is full.
 */
BaseType_t sdk_signal_post(uint16_t signal_id, void *pData, uint16_t len);

/**
 * @brief Post a signal with a specific source.
 */
BaseType_t sdk_signal_post_with_source(uint16_t signal_id, void *pData, uint16_t len, sdk_signal_src_t source);

/**
 * @brief Check if a signal has any subscribers and retrieve the redirection mode.
 * 
 * @param signal_id     The ID of the signal to check.
 * @param out_mode      Pointer to store the subscription mode (if subscriber exists).
 *                      If multiple subscribers exist, REDIRECT takes precedence.
 * @return BaseType_t   pdTRUE if at least one subscriber exists, pdFALSE otherwise.
 */
BaseType_t sdk_signal_has_subscriber(uint16_t signal_id, sdk_sub_mode_t *out_mode);

#else

// Stub macros when SDK Signal feature is disabled
#define sdk_signal_init()               (pdPASS)
#define wlan_sdk_subscribe(a, b, c)     (pdPASS)
#define wlan_sdk_unsubscribe(a, b)      (pdPASS)
#define sdk_signal_post(a, b, c)        (pdPASS)
#define sdk_signal_post_with_source(a, b, c, d)        (pdPASS)
#define sdk_signal_has_subscriber(a, b) (0)

#endif // CONFIG_ENABLE_SDK_SIGNAL

#endif // __SDK_SIGNAL_MNT_H__
