/***************************************************************************************************
 *                              MCU ACH118X driver library
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     ach118x_wdt.h
 * @brief    Functions prototypes and definitions for the WDT firmware library.
 * @version  V1.0.0
 * @date     05-Nov-2020
 ***************************************************************************************************/
#ifndef __ACH118X_WDT_H
#define __ACH118X_WDT_H

#ifdef __cplusplus
   extern "C" {
#endif

#include "ach118x_type.h"
#include "ach118x_map.h"

//WDT password register
#define WDT_PASSWORD_REG        *(uint32_t*)(WDT_REG_BASE + 0x300)
//WDT password
#define WDT_PASSWORD            (0x1ACCE551)

//------------------------------------------------------------------
// CR Bits
//------------------------------------------------------------------
// WDT interrupt enable bit.
#define WDT_INTEN_BIT_MASK      (0x01)
// WDT enable bit.
// This bit is used to enable and disable the DW_apb_wdt.
// When disabled, the counter does not decrement.
#define WDT_EN_BIT_MASK         (0x03)

//------------------------------------------------------------------
// EOI Bits
//------------------------------------------------------------------
// Clears the watchdog interrupt. This can be used to
// clear the interrupt without restarting the watchdog counter.
#define WDT_EOI_CLR_BIT_MASK    (0x01)

//------------------------------------------------------------------
// WDT_STAT Bits
//------------------------------------------------------------------
// This register shows the interrupt status of the WDT.
#define WDT_STAT_BIT_MASK       (0x01)

//WDT registers definition
typedef struct
{
    vu32    LOAD;           //0x00, Current counter value register
    vu32    Reserved_1;     //0x04
    vu32    CR;             //0x08, Interrupt and reset enable register
    vu32    EOI;            //0x0C, Interrupt clear register
    vu32    STAT_1;         //0x10, Interrupt status register0
    vu32    STAT;           //0x14, Interrupt status register1
} WDT_Typedef;


//------------------------------------------------------------------
// API prototypes
//------------------------------------------------------------------

/**
  * @brief  Unlock watchdog
  * @retval none
  */
void  WDT_Unlock(void);

/**
  * @brief  Set watchdog counter
  * @param  CountValue: counter value
  * @retval none
  */
void  WDT_SetTIM(uint32_t CountValue);

/**
  * @brief  Enable or disable watchdog
  * @param  NewState: Enable or Disable
  * @retval none
  */
void  WDT_Cmd(FunctionalState NewState);

/**
  * @brief  Interrupt enable or disable
  * @param  NewState: Enable or Disable
  * @retval none
  */
void  WDT_ITCmd(FunctionalState NewState);

/**
  * @brief  Get interrupt status
  * @retval SET or RESET
  */
FlagStatus  WDT_GetITFlag(void);

/**
  * @brief  Clear interrupt status
  * @retval none
  */
void  WDT_ClearITFlag(void);

#ifdef __cplusplus
    }//extern "C"
#endif

#endif // __ACH118X_WDT_H
/************************ (C) COPYRIGHT ART *****END OF FILE****/

