/***************************************************************************************************
 *                              MCU ACH118X driver library
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     ach118x_uart.h
 * @brief    Function prototypes and definitions for the UART firmware library.
 * @version  V1.0.0
 * @date     05-Nov-2020
 * @detail
 *   - Modeled after the industry-standard 16550.
 *   - Number of data bits supported per character (5-9).
 *   - Baud rate up to 2M baud.
 *   - Parity generation/checking - odd, even select or Stick Parity.
 *   - Number of stop bits (1, 1.5 or 2).
 *   - Interrupt generation.
 *   - Transmit and receive data FIFO size 256 bytes.
 *   - Auto Flow Control mode, as specified in the 16750 standard.
 *   - Programmable Transmit Holding Register Empty (THRE) interrupt.
 *     When FIFOs and the THRE Mode are selected and enabled, THRE Interrupts
 *     are active at or below a programmed TX FIFO threshold level. Additionally,
 *     the Line Status THRE switches from indicating TX FIFO empty to TX FIFO full,
 *     which allows software to set a threshold that keeps the transmitter FIFO
 *     from running empty whenever there is data to transmit.
 ***************************************************************************************************/
#ifndef __ACH118X_UART_H
#define __ACH118X_UART_H

#include "ach118x_type.h"

/*UART register define*/
typedef union
{
    vu32 RBR;   //0x00,
    vu32 THR;   //0x00,
    vu32 DLL;   //0x00,
} regRTD;

typedef union
{
    vu32 DLH;   //0x04,
    vu32 IER;   //0x04,
} regDI;

typedef union
{
    vu32 IIR;   //0x08,
    vu32 FCR;   //0x08,
} regIF;

typedef union
{
    vu32 SRBR[16];
    vu32 STHR[16];
} regSS;

typedef struct
{
    regRTD RTD;     //0x00 (RBR + THR + DLL)
    regDI  DI;      //0x04  (DLH+IER)
    regIF  IF;      //0x08  (IIR+FCR)
    vu32 LCR;       //0x0c,

    vu32 MCR;       //0x10,
    vu32 LSR;       //0x14,
    vu32 MSR;       //0x18,
    vu32 SCR;       //0x1c,

    vu32 LPDLL;     //0x20
    vu32 LPDLH;     //0x24
    vu32 Resever28; //0x28
    vu32 Resever2c; //0x2c

    regSS SS;       //0x30-0x6c

    vu32 FAR;       //0x70,
    vu32 TFR;       //0x74,
    vu32 RFW;       //0x78,
    vu32 USR;       //0x7c,

    vu32 TFL;       //0x80,
    vu32 RFL;       //0x84
    vu32 SRR;       //0x88
    vu32 SRTS;      //0x8c

    vu32 SBCR;      //0x90
    vu32 SDMAM;     //0x94
    vu32 SFE;       //0x98
    vu32 SRT;       //0x9c

    vu32 STET;      //0xa0
    vu32 HTX;       //0xa4
    vu32 DMASA;     //0xa8
    vu32 TCR;       //0xac

    vu32 DE_EN;     //0xb0
    vu32 RE_EN;     //0xb4;
    vu32 DET;       //0xb8;
    vu32 TAT;       //0xbc

    vu32 DLF;       //0xc0
    vu32 RAR;       //0xc4
    vu32 TAR;       //0xc8
    vu32 LCR_EXT;   //0xcc

    vu32 ReseverD0; //0xD0
    vu32 ReseverD4; //0xD4
    vu32 ReseverD8; //0xD8
    vu32 ReseverDC; //0xDC

    vu32 ReseverE0; //0xe0
    vu32 ReseverE4; //0xe4
    vu32 ReseverE8; //0xe8
    vu32 ReseverEC; //0xeC

    vu32 ReseverF0; //0xf0
    vu32 CPR;       //0xf4
    vu32 UCV;       //0xf8
    vu32 CTR;       //0xfc
}UART_Typedef;

typedef enum{
    UART_LEN_5 = 0,
    UART_LEN_6,
    UART_LEN_7,
    UART_LEN_8
}WordLenType;

typedef enum{
    UART_STOP_1 = 0x0,
    UART_STOP_2 = 0x4
}StopBitType;

typedef enum{
    UART_NONE_PARITY = 0x00,
    UART_EVEN_PARITY = 0x38,
    UART_ODD_PARITY  = 0x28
}ParityType;

typedef enum{
    UART_FLOWCON_DISABLE = 0x00,
    UART_FLOWCON_ENABLE  = 0x20,
}FlowConType;

typedef struct
{
    uint32_t UART_BaudRate;
    WordLenType UART_WordLength;
    StopBitType UART_StopBits;
    ParityType  UART_Parity;
    FlowConType UART_HardwareFlowControl;
} UART_InitTypeDef;

//IER (Interrupt Enable Register)
#define UART_ERBFI              (0x01)
#define UART_ETBEI              (0x02)
#define UART_ELSI               (0x04)
#define UART_EDSSI              (0x08)
#define UART_PTIME              (0x80)

//IER (Interrupt Enable Register)
#define UART_RCV_DATA_IT        (0x01)
#define UART_TX_EMPTY_IT        (0x02)
#define UART_RCV_LINE_IT        (0x04)
#define UART_MODEM_IT           (0x08)
#define UART_PROGRAMN_IT        (0x80)

//IIR(Interrupt Identity Register)
#define UART_MODEM_STATUS       (0x00)
#define UART_NO_INTPENDING      (0x01)
#define UART_THR_EMPTY          (0x02)
#define UART_RXD_AVAILABLE      (0x04)
#define UART_RXD_LINE_STATUS    (0x06)
#define UART_BUSY_DETECT        (0x07)
#define UART_CHAR_TIMEOUT       (0x0c)
#define UART_FIFOSE             (0xc0)

//FCR(FIFO Control Register)
#define UART_FIFOE              (0x01)
#define UART_RFIOR              (0x02)
#define UART_XFIFOR             (0x04)
#define UART_DMAM_MODE1         (0x08)        // Not used
#define UART_DMAM_MODE0         (~DMAM_MODE1) // Not used

//FCR[5:4] TX Empty Trigger
#define UART_TET_EMPTY          (0x00)
#define UART_TET_2CHAR          (0x10)
#define UART_TET_QFULL          (0x20)
#define UART_TET_HFULL          (0x30)

 //FCR[7:6] RCVR Trigger
#define UART_RCT_1CHAR          (0x00)
#define UART_RCT_QFULL          (0x40)
#define UART_RCT_HFULL          (0x80)
#define UART_RCT_LFULL          (0xc0)
//LCR(Line Control register)
//LCR[1:0]
/*
#define UART_DLS_5BIT           (0x00)
#define UART_DLS_6BIT           (0x01)
#define UART_DLS_7BIT           (0x02)
#define UART_DLS_8BIT           (0x03)
*/
//LCR[2]
/*
#define UART_STOP_1             (0x00)
#define UART_STOP_2             (0x04)
#define    UART_NONE_PARITY        (0x00)
#define    UART_EVEN_PARITY        (0x38)
#define    UART_ODD_PARITY         (0x28)
*/

#define UART_ENABLE_BAUDRATE    (0x80)
#define UART_DISABLE_BAUDRATE   (~UART_ENABLE_BAUDRATE)

//MCR(Modem Control Register)
#define UART_DTR_LOW            (0x01)
#define UART_RTS_LOW            (0x02)
#define UART_OUT1_LOW           (0x04)
#define UART_OUT2_LOW           (0x08)
#define UART_LOOP_BACK          (0x10)
#define UART_FLOW_CONTROL       (0x20)
#define UART_SIR_MODE_ENABLE    (0x40)

#define UART_ALL_CONFIG         (0xffff)/*The define is used to clear old configuaration,
                                          The following API can use the define:
                                            UART_ITConfig,
                                            UART_FIFOConfig,
                                            UART_LineConfig,
                                            UART_LineExtConfig,
                                            UART_ModemConfig */

//LSR(Line Status Register)
#define UART_DR_STATUS          (0x01)
#define UART_OE_STATUS          (0x02)
#define UART_PE_STATUS          (0x04)
#define UART_FE_STATUS          (0x08)
#define UART_BI_STATUS          (0x10)
#define UART_THRE_STATUS        (0x20)
#define UART_TEMT_STATUS        (0x40)
#define UART_RFE_STATUS         (0x80)

//MSR(Modem Status Register)
#define UART_DCTS_STATUS        (0x01)
#define UART_DDSR_STATUS        (0x02)
#define UART_TERI_STATUS        (0x04)
#define UART_DDCD_STATUS        (0x08)
#define UART_CTS_STATUS         (0x10)
#define UART_DSR_STATUS         (0x20)
#define UART_RI_STATUS          (0x40)
#define UART_DCD_STATUS         (0x80)
#if (UART_FIFO_ACCESS == 1)
    //FAR(FIFO Access Register)
    //#define UART_FIFO_ACCESS_ENABLE      (0x01)
    //#define UART_FIFO_ACCESS_DISABLE     (~FIFO_ACCESS_ENABLE)
#endif //END UART_FIFO_ACCESS

//USR(UART Status Register)
#define UART_RFF_STATUS         (0x10)
#define UART_RFNE_STATUS        (0x08)
#define UART_TFE_STATUS         (0x04)
#define UART_TFNF_STATUS        (0x02)
#define UART_BUSY_STATUS        (0x01)
#if (UART_SHADOW == 1)
//SRR(software reset register)
  #define UART_UR_RESET         (0x01)
  #define UART_RFR_RESET        (0x02)
  #define UART_XFR_RESET        (0x04)
//SRTS(Shadow request to send)
  #define UART_SRTS_HIGH        (0x01)
  #define UART_SRTS_LOW         (~SRTS_HIGH)
//SBCR(Shadow Break Control Register)
  #define UART_SBCR_HIGH        (0x01)
  #define UART_SBCR_LOW         (~SBCR_HIGH)
//SDMAM(Shadow DMA MODE)
  #define UART_SDMAM_MODE1      (0x01)
  #define UART_SDMAM_MODE0      (~SDMAM_MODE1)
//SFE(Shadow FIFO Enable)
  #define UART_SFIFO_ENABLE     (0x01)
  #define UART_SFIFO_DISABLE    (~SFIFO_ENABLE)
//SRT(Shadow RCVR Trigger)
  #define UART_SRCT_1CHAR       (0x00)
  #define UART_SRCT_QFULL       (0x01)
  #define UART_SRCT_HFULL       (0x02)
  #define UART_SRCT_LFULL       (0x03)
//STET(Shadow TX Empty Trigger)
  #define UART_STET_EMPTY       (0x00)
  #define UART_STET_2CHAR       (0x01)
  #define UART_STET_QFULL       (0x02)
  #define UART_STET_HFULL       (0x03)
#endif
//HTX(Halt TX)
#define UART_HALTTX_ENABLE           (0x01)
#define UART_HALTTX_DISABLE          (~(HALTTX_ENABLE))
//DMASA(DMA Software Acknowledge)
#define UART_DMASW_ACK_ENABLE        (0x01)
#define UART_DMASW_ACK_DISABLE       (~DMASW_ACK_ENABLE)
#if (UART_RS485==1)
//TCR(tramsceiver Control Register)
  #define UART_RS485_EN              (0x01)
  #define UART_RE_POL_HIGH           (0x02)
  #define UART_DE_POL_HIGH           (0x04)
  #define UART_XFER_MODE0            (0x00)
  #define UART_XFER_MODE1            (0x08)
  #define UART_XFER_MODE2            (0x18)
  #define UART_RS232_EN              (~RS485)
  #define UART_RE_POL_LOW            (~RE_POL_HIGH)
  #define UART_DE_POL_LOW            (~DE_POL_HIGH)
//DE_EN(Driver output Enable regsiter)
  #define UART_DE_ENABLE             (0x01)
  #define UART_DE_DISABLE            (~DE_ENABLE)
//RE_EN(Receiver output Enable regsiter)
  #define UART_RE_ENABLE             (0x01)
  #define UART_RE_DISABLE            (~RE_ENABLE)
//DET(Driver output enable)
typedef enum{
    ASSERTION   = 0,
    DEASSERTION = 1
}DeTimerType;

typedef enum{
    REtoDE = 0,
    DEtoRE = 1
}TATimerType;
#endif

//LCR_EXT(Line Extended Control Regsiter)
#define UART_DLS_BIT9_ENABLE        (0x01)
#define UART_ADDR_MATCH_MODE        (0x02)
#define UART_SEND_ADDR_1            (0x04)
#define UART_TRANSMIT_MODE1         (0x08)

//CPR(Component Parameter Register)
#define UART_APB_DATA_WIDTH_8BIT    (0x0000)
#define UART_APB_DATA_WIDTH_16BIT   (0x0001)
#define UART_APB_DATA_WIDTH_32BIT   (0x0002)
#define UART_AFCE_MODE              (0x0010)
#define UART_THRE_MODE              (0x0020)
#define UART_SIR_MODE               (0x0040)
#define UART_SIR_LP_MODE            (0x0080)
#define UART_ADDITIONAL_FEAT        (0x0100)
#define UART_FIFO_ACCESS            (0x0200)
#define UART_FIFO_STAT              (0x0400)
#define UART_SHADOW                 (0x0800)
#define UART_ADD_ENCODED_PARAMS     (0x1000)
#define UART_DMA_EXTRA              (0x2000)

#define UART1_ENABLE                (0x0000d7)
#define UART2_ENABLE                (0x550000)

#define UART_FOREVER                (0)

// macros & depricated functions
#define putstring(s)    SerialPutString(s)//ach_uart_send_string(UART0, s)
uint8_t ach_uart_read_byte(UART_Typedef* UARTx);
void ach_uart_send_string(UART_Typedef* UARTx, char *str);
void getline(char *buf, uint16_t bufmax);
int getchar(void);


//------------------------------------------------------------------
// API prototypes
//------------------------------------------------------------------

/**
  * @brief  Initialize UART
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  UART_InitStruct: Structure containing init parameters
  * @retval None
  */
void  UART_Init(UART_Typedef* UARTx, UART_InitTypeDef* UART_InitStruct);

/**
  * @brief  Send data
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  Data: Data to send
  * @param  Timeout: 0 to try forever, N to try N times
  * @retval None
  */
void  UART_SendData(UART_Typedef* UARTx, uint16_t Data, uint32_t Timeout);

/**
  * @brief  Receive data
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  Timeout: 0 to try forever, N to try N times
  * @retval Received data
  */
uint16_t  UART_ReceiveData(UART_Typedef* UARTx, uint32_t Timeout);

/**
  * @brief  Configure the interrupt
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  Method: Interrupt trigger method. Can be UART_ERBFI, UART_ETBEI,
  *                 UART_ELSI, UART_EDSSI, UART_PTIME
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void  UART_ITConfig(UART_Typedef* UARTx, uint16_t Method, FunctionalState NewState);

/**
  * @brief  Configure the FIFO trigger method
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  Method: FIFO trigger method. Can be UART_FIFOE, UART_RFIOR, UART_XFIFOR
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void  UART_FIFOConfig(UART_Typedef* UARTx, uint16_t Method, FunctionalState NewState);

/**
  * @brief  Configure the UART Line
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  UART_Line: DLAB, BREAK, STICK_PARITY, EPS, PEN, STOP, DLS
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void  UART_LineConfig(UART_Typedef* UARTx, uint16_t UART_Line, FunctionalState NewState);

/**
  * @brief  Configure the UART Ext Line
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  UART_Line: DLS_E, ADDR_MATCH, SEND_ADDR, TRANSMIT_MODE
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void  UART_LineExtConfig(UART_Typedef* UARTx, uint16_t UART_Line, FunctionalState NewState);

/**
  * @brief  Configure the Modem control register
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  UART_Line: DTR, RTS, OUT1, OUT2, LB, AFCE, SIRE
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void  UART_ModemConfig(UART_Typedef* UARTx, uint16_t UART_Modem, FunctionalState NewState);

/**
  * @brief  Get UART interrupt status
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  ITSource: Interrupt source can be DTR, RTS, OUT1, OUT2, LB, AFCE, SIRE
  * @retval SET or RESET
  */
ITStatus  UART_GetITStatus(UART_Typedef* UARTx, uint16_t ITSource);

/**
  * @brief  Gets status of specified USART Line
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  UART_Line: DR, OE, PE, FE, BI, THRE, TEMT, RFE, ADDR_RCVD
  * @retval SET or RESET
  */
FlagStatus  UART_GetLineStatus(UART_Typedef* UARTx, uint16_t UART_Line);

/**
  * @brief  Gets Modem status
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  UART_FLAG: DCTS, DDSR, TERI, DDCD, CTS, DSR, RI, DCD
  * @retval SET or RESET
  */
FlagStatus  UART_GetModemStatus(UART_Typedef* UARTx, uint16_t UART_FLAG);

/**
  * @brief  Get UART status
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  UART_FLAG: BUSY, TFNF, TFE, RFNE, RFF
  * @retval SET or RESET
  */
FlagStatus  UART_GetUartStatus(UART_Typedef* UARTx, uint16_t UART_FLAG);

/**
  * @brief  Check for component validity
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  UART_Param: Param value
  * @retval SET or RESET
  */
FlagStatus  UART_GetComponentParameter(UART_Typedef* UARTx, uint16_t UART_Param);

/**
  * @brief  Get component version
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @retval Version
  */
uint32_t  UART_GetComponentVersion(UART_Typedef* UARTx);

/**
  * @brief  Get component type
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @retval Component type
  */
uint32_t  UART_GetComponentType(UART_Typedef* UARTx);

/**
  * @brief  Enable or disable UART interrupt
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void  UART_ITCmd(UART_Typedef* UARTx, uint16_t UART_IT, FunctionalState NewState);

/**
  * @brief  Enable or disable UART modem
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  UART_Modem: DCTS, DDSR, TERI, DDCD, CTS DSR, RI, DCD
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void  UART_ModemCmd(UART_Typedef* UARTx, uint16_t UART_Modem, FunctionalState NewState);

/**
  * @brief  Check UART interrupt
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  UART_IT: IID, FIFOSE
  * @retval SET or RESET
  */
ITStatus  UART_GetITFlag(UART_Typedef* UARTx, uint16_t UART_IT);

//void  GKI_UartITCmd(UART_Typedef* UARTx, uint16_t UART_IT, FunctionalState NewState);

/**
  * @brief  Enable or disable UART FIFO
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @param  UART_IT: FIFOE, RFIFOR, XFIFOR, DMAM, TET, RCVR
  * @retval none
  */
void  UART_FIFOCmd(UART_Typedef* UARTx, uint16_t UART_FIFO, FunctionalState NewState);

/**
  * @brief  Enable UART System Interrupts. Set flags using
  *         SYSCFG_SYS_LEV_INTR_MASK_REG and NVIC_EnableIRQ.
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @retval none
  */
void  UART_SystemInterruptEnable(UART_Typedef* UARTx);

/**
  * @brief  Disable UART System Interrupts. Clear flags using
  *         SYSCFG_SYS_LEV_INTR_MASK_REG and NVIC_DisableIRQ.
  * @param  UARTx: where x can be 0, 1, 2 to select the UART
  * @retval none
  */
void  UART_SystemInterruptDisable(UART_Typedef* UARTx);


#if (UART_FIFO_ACCESS == 1)
    void     UART_FIFOAccessCmd(UART_Typedef* UARTx,FunctionalState NewState);
    uint8_t  UART_FIFOAccessReadTranFIFO(UART_Typedef* UARTx);
    void     UART_FIFOAccessWriteReceFIFO(UART_Typedef* UARTx,uint16_t data);
    uint32_t UART_FIFOAccessGetTraFIFOLevel(UART_Typedef* UARTx);
    uint32_t UART_FIFOAccessGetRecFIFOLevel(UART_Typedef* UARTx);
#endif //END UART_FIFO_ACCESS

void    UART_HaltTXCmd(UART_Typedef* UARTx,FunctionalState NewState);

#if (UART_SHADOW == 1)
    void     UART_SendData_Shadow(UART_Typedef* UARTx, uint16_t Data);
    uint16_t UART_ReceiveData_Shadow(UART_Typedef* UARTx);
    void     UART_SWReset_Shadow(UART_Typedef* UARTx, uint16_t UART_Shadow_option, FunctionalState NewState);
    void     UART_RTS_Shadow(UART_Typedef* UARTx,LevelState NewState);
#endif // END UART_SHADOW

#if (UART_RS485 == 1)
    void    UART_RS485_TranReceConfig(UART_Typedef* UARTx,uint16_t UART_OPTION, FunctionalState NewState);
    void    UART_RS485_DE_EN(UART_Typedef* UARTx);
    void    UART_RS485_RE_EN(UART_Typedef* UARTx);
    void    UART_RS485_SetDETimer(UART_Typedef* UARTx,DeTimerType option,uint16_t value);
    void    UART_RS485_SetTurnAroundTimer(UART_Typedef* UARTx,TATimerType option,uint16_t value);
#endif //END UART_RS485


#endif // __ACH118X_UART_H
/************************ (C) COPYRIGHT ART *****END OF FILE****/

