/***************************************************************************************************
 *                      MCU ACH118X driver library
 *              COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                         All Rights Reserved
 *
 * @file     ach118x_timer.h
 * @brief    Function prototypes for the Timer firmware library.
 * @version  V1.0.0
 * @date     18-Nov-2020
 ***************************************************************************************************/
#ifndef __ACH1180X_TIM_H
#define __ACH1180X_TIM_H

#include "ach118x_map.h"
#include "ach118x_type.h"

/*************************************************************************/
/*                   TIMER Definitions                                   */
/*************************************************************************/

// TIM 0-7
typedef struct {
    vu32   LoadCount;      // Load Count Register (32 bits)
    vu32   CurrentValue;   // Current Value Register (32 bits)
    vu32   ControlReg;     // Control Register (3 bits)
    vu32   EOI;            // End-of-Interrupt Register (1 bit)
    vu32   IntStatus;      // Interrupt Status Register (1 bit)
} TIM_Typedef;


// Control Register Flags
#define TIM_CTRL_NONE  0x00
#define TIM_ENABLE     0x01
#define TIM_COUNT_MODE 0x02 // or Free-Running mode
#define TIM_INT_MASK   0x04

//------------------------------------------------------------------
// API prototypes
//------------------------------------------------------------------

/**
  * @brief  Enable or disable the timer
  * @param  TIMx: where x can be 0-3 to select the timer.
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void  TIM_Cmd(TIM_Typedef *TIMx, FunctionalState NewState);

/**
  * @brief  Set the initial countdown timer value
  * @param  TIMx: where x can be 0-3 to select the timer.
  * @param  count: Initial value to set countdown timer
  * @retval none
  */
void  TIM_SetCounter(TIM_Typedef *TIMx, uint32_t count);

/**
  * @brief  Get countdown timer value
  * @param  TIMx: where x can be 0-3 to select the timer.
  * @retval Value of countdown timer
  */
uint32_t  TIM_GetCounter(TIM_Typedef *TIMx);

/**
  * @brief  Get timer interrupt state
  * @param  TIMx: where x can be 0-3 to select the timer.
  * @retval ENABLE or DISABLE
  */
uint8_t  TIM_GetITFlag(TIM_Typedef *TIMx);

/**
  * @brief  Clear timer interrupt state
  * @param  TIMx: where x can be 0-3 to select the timer.
  * @retval none
  */
void  TIM_ClearITFlag(TIM_Typedef *TIMx);

/**
  * @brief  Enable TIM System Interrupts. Set flags using
  *         SYS_INT_MASK and NVIC_EnableIRQ.
  * @param  TIMx: where x can be 0-3 to select the timer.
  * @retval none
  */
void  TIM_SystemInterruptEnable(TIM_Typedef *TIMx);

/**
  * @brief  Disable TIM System Interrupts. Clear flags using
  *         SYS_INT_MASK and NVIC_DisableIRQ.
  * @param  TIMx: where x can be 0-3 to select the timer.
  * @retval none
  */
void  TIM_SystemInterruptDisable(TIM_Typedef *TIMx);

#endif /* __ACH1180X_TIM_H */

/************************ (C) COPYRIGHT ART *****END OF FILE****/

