/***************************************************************************************************
 *                     MCU ACH118X driver library
 *             COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                        All Rights Reserved
 *
 * @file     ach118x_syscfg.h
 * @brief    System register definitions.
 * @version  V1.0.0
 * @date     05-Nov-2020
 * @detail
 *  - Data structures and the address mapping for SYSCFG
 *  - SYSCFG registers declarations and bits definition
 *  - Macros to access SYSCFG register hardware
 ***************************************************************************************************/
#ifndef __ACH118X_SYSCFG_H
#define __ACH118X_SYSCFG_H

#include "ach118x_type.h"

//------------------------------------------------------------------------------------
//           SYSCFG Register Address Map
//------------------------------------------------------------------------------------
#define SYSCFG_RST_REG                      0x00    /* reset_register */
#define SYSCFG_CFG_REG0                     0x04    /* sys_cfg_reg0 */
#define SYSCFG_EXT_CLK_DIV_REG              0x08    /* external_clock_divider, Reserved */
#define SYSCFG_CLK_CTRL_REG0                0x0C    /* clock_control_reg0 */
#define SYSCFG_CLK_CTRL_REG1                0x10    /* clock_control_reg1 */
#define SYSCFG_CODEC_SCAN_INP_REG0          0x14    /* codec_scan_reg0*/
#define SYSCFG_CODEC_SCAN_INP_REG1          0x18    /* codec_scan_reg1 */
#define SYSCFG_MEM_TEST_SECTOR0_REG0        0x1C    /* mem_test_sector0_reg0 */
#define SYSCFG_MEM_TEST_SECTOR0_REG1        0x20    /* mem_test_sector0_reg1 */
#define SYSCFG_MEM_TEST_SECTOR0_REG2        0x24    /* mem_test_sector0_reg2 */
#define SYSCFG_MEM_TEST_SECTOR1_REG0        0x28    /* mem_test_sector1_reg0 */
#define SYSCFG_MEM_TEST_SECTOR1_REG1        0x2C    /* mem_test_sector1_reg1 */
#define SYSCFG_MEM_TEST_SECTOR1_REG2        0x30    /* mem_test_sector1_reg2 */
#define SYSCFG_MEM_TEST_SECTOR2_REG0        0x34    /* mem_test_sector2_reg0 */
#define SYSCFG_MEM_TEST_SECTOR2_REG1        0x38    /* mem_test_sector2_reg1 */
#define SYSCFG_MEM_TEST_SECTOR2_REG2        0x3C    /* mem_test_sector2_reg2 */
#define SYSCFG_MPU_COUNTER_REG              0x40    /* pmu_counter */
#define SYSCFG_INTR_POLARITY_REG            0x44    /* interrupt_polarity */
#define SYSCFG_SYS_LEV_INTR_MASK_REG        0x48    /* sys_level_intr_mask_reg*/
#define SYSCFG_PLL0_REG0                    0x4C    /* pll0_reg0 */
#define SYSCFG_PLL0_REG1                    0x50    /* pll0_reg1 */
#define SYSCFG_REG20                        0x54    /* sys_cfg_reg20 */
#define SYSCFG_PLL1_REG0                    0x58    /* pll1_reg0 */
#define SYSCFG_PLL1_REG1                    0x5C    /* pll1_reg1 */
#define SYSCFG_SPREAD_SPECTRUM_PLL1_REG     0x60    /* spread_spectrum_pll1 */
#define SYSCFG_Reserved_64                  0x64    /* sys_cfg_reg24 */
#define SYSCFG_PERIPHERAL_RST_ENB_REG       0x68    /* peripheral_reset_enable */
#define SYSCFG_PAD_ENB_REG0                 0x6C    /* output_enable_reg0 */
#define SYSCFG_PAD_ENB_REG1                 0x70    /* output_enable_reg1 */
#define SYSCFG_REG28                        0x74    /* sys_cfg_reg28 */
#define SYSCFG_REG29                        0x78    /* sys_cfg_reg29 */
#define SYSCFG_REG30                        0x7C    /* sys_cfg_reg30 */
#define SYSCFG_REG31                        0x80    /* sys_cfg_reg31 */
#define SYSCFG_CLK_STATUS_REG               0x84    /* clock_status */
#define SYSCFG_INTR_STATUS_REG              0x88    /* interrupt_status */
#define SYSCFG_PLL1_SPREAD_SPECTRUM_REG     0x8C    /* pll1_spread_spectrum */
#define SYSCFG_STATUS_REG0                  0x90    /* status_reg0 */
#define SYSCFG_STATUS_REG1                  0x94    /* status_reg1 */
#define SYSCFG_REG37                        0x98    /* sys_cfg_reg37 */
#define SYSCFG_STATUS_REG2                  0x9C    /* status_reg2 */
#define SYSCFG_CLK_ENB_REG                  0xA0    /* clock_enable */
#define SYSCFG_REG40                        0xA4    /* sys_cfg_reg40 */
#define SYSCFG_CODEC_SCAN_OUT_REG0          0xA8    /* codec_scan_reg0 */
#define SYSCFG_CODEC_SCAN_OUT_REG1          0xAC    /* codec_scan_reg1 */


// ACH Sys Status
typedef struct {
    volatile uint32_t
        BOOT            : 1,
        MAIN_PLL        : 1,
        AUDIO_PLL       : 1,
        Reserved_3_31   : 29;
} SYS_CLK_STATUS_Typedef;

// ACH Sys Reset
typedef struct {
    volatile uint32_t           // Defaults:
        SSI_RST0_N      : 1,    // 0
        SSI_RST1_N      : 1,    // 0
        SSI_RST2_N      : 1,    // 0
        SSI2S0_SRESETN  : 1,    // 0
        I2S1_CH4_SRESETN: 1,    // 0
        IC_I2C_RST_N    : 1,    // 0
        TIMER_1_RESETN  : 1,    // 0
        RSTB            : 1,    // 0
        SHRESETN        : 1,    // 0
        MHRESETN        : 1,    // 0
        S0_DUT_PRESETN  : 1,    // 0
        S1_DUT_PRESETN  : 1,    // 0
        SYS_CFG_RSTN    : 1,    // 0
        SYSTEM_RESET    : 1,    // 0
        SPEED_UP_WDT_SYS_CFG: 1,// 0
        SECONDARY_WD_ENB: 1;    // 0
} SYS_RESET_Typedef;

// ACH Sys Clk Cntr 0
typedef struct {
    volatile uint32_t           // Defaults:
        Reserved_0_3    : 4,    //
        HCLK_sel        : 1,    // 0
        Reserved_5_31   : 27;
} SYS_CLK_CNTRL0_Typedef;

// ACH Sys Clk Cntr 1
typedef struct {
    volatile uint32_t               // Defaults:
        IO_SS_IN_N          : 1,    // 1 - bit 0
        IO_SS_IN_1          : 1,    // 1 - bit 1
        IO_SS_IN_2          : 1,    // 1 - bit 2
        IO_DSR_N_0          : 1,    // 0 - bit 3
        IO_DCD_N_0          : 1,    // 1 - bit 4
        IO_RI_N_0           : 1,    // 1 - bit 5
        IO_DSR_N_1          : 1,    // 0 - bit 6
        IO_DCD_N_1          : 1,    // 1 - bit 7
        IO_RI_N_1           : 1,    // 1 - bit 8
        IO_DSR_N_2          : 1,    // 0 - bit 9
        IO_DCD_N_2          : 1,    // 1 - bit 10
        IO_RI_N_2           : 1,    // 1 - bit 11
        SPEED_UP_WDT_ADC1   : 1,    // 0 - bit 12
        SPEED_UP_WDT_ADC2   : 1,    // 0 - bit 13
        SPEED_UP_WDT_AES    : 1,    // 0 - bit 14
        Reserved_15_17      : 3,    //   - bit 15-17
        INTFSEL             : 1,    // 1 - bit 18
        I2S1_CLK_STG1_SEL   : 1,    // 0 - bit 19  DIV4 = 1
        Reserved_20         : 1,    //   - bit 20
        CODEC_CLK_SEL       : 1,    // 0 - bit 21  Selection for CODEC clk
        I2S1_CLK_SEL        : 1,    // 0 - bit 22
        CODEC_WORD_SEL      : 1,    // 0 - bit 23
        CODEC_DATA_INPUT    : 1,    // 0 - bit 24
        I2S0_CLK_SEL        : 1,    // 0 - bit 25
        I2S0_CLK_STG1_SEL   : 1,    // 0 - bit 26  Muxed internal signal
        Reserved_27_31      : 5;    //   - bit 27-31
} SYS_CLK_CNTRL1_Typedef;

// ACH Sys Interrupt Mask
typedef struct {
    volatile uint32_t
        SPI0_IRQ       : 1,    // b0   SPI0 Interrupt mask
        SPI1_IRQ       : 1,    // b1   SPI1 Interrupt mask
        SPI2_IRQ       : 1,    // b2   SPI2 Interrupt mask
        I2S0_IRQ       : 1,    // b3   I2S0 mask
        UART0_IRQ      : 1,    // b4   UART0 Interrupt mask
        UART1_IRQ      : 1,    // b5   UART1 Interrupt mask
        UART2_IRQ      : 1,    // b6   UART2 Interrupt mask
        I2S1_IRQ       : 1,    // b7   I2S1 Interrupt mask
        I2C0_IRQ       : 1,    // b8   I2C0 Interrupt mask
        TIM0_IRQ       : 1,    // b9   Timer 1 Interrupt mask
        GPIO3_IRQ      : 1,    // b10  GPIO3 Interrupt mask
        WDOG_IRQ       : 1,    // b11  WDOG timer Interrupt mask
        ADC0_IRQ       : 1,    // b12  ADC0 Interrupt mask
        ADC1_IRQ       : 1,    // b13  ADC1 Interrupt mask
        AES_IRQ        : 1,    // b14  AES Interrupt mask
        RSRVD_15_IRQ   : 1,    // b15  Reserved
        RSRVD_16_IRQ   : 1,    // b16  Reserved
        DMAC_IRQ       : 1,    // b17  DMAC Interrupt mask
        TIM1_IRQ       : 1,    // b18  Timer 1 Interrupt mask
        TIM2_IRQ       : 1,    // b19  Timer 2 Interrupt mask
        TIM3_IRQ       : 1;    // b20  Timer 3 Interrupt mask
} SYS_IRQ_Typedef;

// ACH System registers
typedef struct
{
    SYS_RESET_Typedef       RESET;              /* 0x00  reset_register */
    vu32                    CFG0;               /* 0x04  sys_cfg_reg0 */
    vu32                    EX_CLOCK_DIV;       /* 0x08  external_clock_divider, Reserved */
    SYS_CLK_CNTRL0_Typedef  CLK_CTRL0;          /* 0x0C  clock_control_reg0 */
    SYS_CLK_CNTRL1_Typedef  CLK_CTRL1;          /* 0x10  clock_control_reg1 */
    vu32                    CODEC_SCAN0;        /* 0x14  codec_scan_reg0*/
    vu32                    CODEC_SCAN1;        /* 0x18  codec_scan_reg1 */
    vu32                    MEM_TEST_SEC0_REG0; /* 0x1C  mem_test_sector0_reg0 */
    vu32                    MEM_TEST_SEC0_REG1; /* 0x20  mem_test_sector0_reg1 */
    vu32                    MEM_TEST_SEC0_REG2; /* 0x24  mem_test_sector0_reg2 */
    vu32                    MEM_TEST_SEC1_REG0; /* 0x28  mem_test_sector1_reg0 */
    vu32                    MEM_TEST_SEC1_REG1; /* 0x2C  mem_test_sector1_reg1 */
    vu32                    MEM_TEST_SEC1_REG2; /* 0x30  mem_test_sector1_reg2 */
    vu32                    MEM_TEST_SEC2_REG0; /* 0x34  mem_test_sector2_reg0 */
    vu32                    MEM_TEST_SEC3_REG1; /* 0x38  mem_test_sector2_reg1 */
    vu32                    MEM_TEST_SEC2_REG2; /* 0x3C  mem_test_sector2_reg2 */
    vu32                    PMU_COUNTER;        /* 0x40  pmu_counter */
    SYS_IRQ_Typedef         INT_POLARITY;       /* 0x44  interrupt_polarity */
    SYS_IRQ_Typedef         INT_MASK;           /* 0x48  sys_level_intr_mask_reg*/
    vu32                    PLL0_REG0;          /* 0x4C  pll0_reg0 */
    vu32                    PLL0_REG1;          /* 0x50  pll0_reg1 */
    vu32                    CFG20;              /* 0x54  sys_cfg_reg20 */
    vu32                    PLL1_REG0;          /* 0x58  pll1_reg0 */
    vu32                    PLL1_REG1;          /* 0x5C  pll1_reg1 */
    vu32                    SPECT_PLL1;         /* 0x60  spread_spectrum_pll1 */
    vu32                    CFG24;              /* 0x64  sys_cfg_reg24 */
    vu32                    PERIPHERAL_RESET;   /* 0x68  peripheral_reset_enable */
    vu32                    REMAP0;             /* 0x6C  output_enable_reg0 */
    vu32                    REMAP1;             /* 0x70  output_enable_reg1 */
    vu32                    CFG28;              /* 0x74  sys_cfg_reg28 */
    vu32                    CFG29;              /* 0x78  sys_cfg_reg29 */
    vu32                    CFG30;              /* 0x7C  sys_cfg_reg30 */
    vu32                    CFG31;              /* 0x80  sys_cfg_reg31 */
    SYS_CLK_STATUS_Typedef  CLK_STATUS;         /* 0x84  clock_status */
    SYS_IRQ_Typedef         INT_STATUS;         /* 0x88  interrupt_status */
    vu32                    PLL1_SPECTRUM;      /* 0x8C  pll1_spread_spectrum */
    vu32                    STATUS0;            /* 0x90  status_reg0 */
    vu32                    STATUS1;            /* 0x94  status_reg1 */
    vu32                    CFG37;              /* 0x98  sys_cfg_reg37 */
    vu32                    STATUS2;            /* 0x9C  status_reg2 */
    vu32                    CLK_ENABLE;         /* 0xA0  clock_enable */
    vu32                    CFG40;              /* 0xA4  sys_cfg_reg40 */
    uc32                    CODEC_SCAN2;        /* 0xA8  codec_scan_reg0 */
    uc32                    CODEC_SCAN3;        /* 0xAC  codec_scan_reg1 */
}SYS_Typedef;



//------------------------------------------------------------------------------------
// Interrupt Mask bits
//------------------------------------------------------------------------------------
#define SPI0_IRQ_MASK       0x00000001          // b0   SPI0 Interrupt mask
#define SPI1_IRQ_MASK       0x00000002          // b1   SPI1 Interrupt mask
#define SPI2_IRQ_MASK       0x00000004          // b2   SPI2 Interrupt mask
#define I2S0_IRQ_MASK       0x00000008          // b3   I2S0 mask
#define UART0_IRQ_MASK      0x00000010          // b4   UART0 Interrupt mask
#define UART1_IRQ_MASK      0x00000020          // b5   UART1 Interrupt mask
#define UART2_IRQ_MASK      0x00000040          // b6   UART2 Interrupt mask
#define I2S1_IRQ_MASK       0x00000080          // b7   I2S1 Interrupt mask
#define I2C0_IRQ_MASK       0x00000100          // b8   I2C0 Interrupt mask
#define TIM_IRQ_MASK        0x00000200          // b9   Timer Interrupt mask
#define GPIO3_IRQ_MASK      0x00000400          // b10  GPIO3 Interrupt mask
#define WDOG_IRQ_MASK       0x00000800          // b11  WDOG timer Interrupt mask
#define ADC0_IRQ_MASK       0x00001000          // b12  ADC0 Interrupt mask
#define ADC1_IRQ_MASK       0x00002000          // b13  ADC1 Interrupt mask
#define AES_IRQ_MASK        0x00004000          // b14  AES Interrupt mask
#define RSRVD_15_IRQ_MASK   0x00008000          // b15  Reserved
#define RSRVD_16_IRQ_MASK   0x00010000          // b16  Reserved
#define DMAC_IRQ_MASK       0x00020000          // b17  DMAC Interrupt mask
#define GPIO2_00_IRQ_MASK   0x00040000          // b18  GPIO2 b1 Interrupt mask
#define GPIO2_01_IRQ_MASK   0x00080000          // b19  GPIO2 b2 Interrupt mask
#define GPIO2_02_IRQ_MASK   0x00100000          // b20  GPIO2 b3 Interrupt mask
#define GPIO2_03_IRQ_MASK   0x00200000          // b21  GPIO2 b4 Interrupt mask
#define GPIO2_04_IRQ_MASK   0x00400000          // b22  GPIO2 b5 Interrupt mask
#define GPIO2_05_IRQ_MASK   0x00800000          // b23  GPIO2 b6 Interrupt mask
#define GPIO2_06_IRQ_MASK   0x01000000          // b24  GPIO2 b7 Interrupt mask
#define GPIO2_07_IRQ_MASK   0x02000000          // b25  GPIO2 b8 Interrupt mask
#define GPIO2_08_IRQ_MASK   0x04000000          // b26  GPIO2 b9 Interrupt mask
#define GPIO2_09_IRQ_MASK   0x08000000          // b27  GPIO2 b10 Interrupt mask
#define GPIO2_10_IRQ_MASK   0x10000000          // b28  GPIO2 b11 Interrupt mask
#define GPIO2_11_IRQ_MASK   0x20000000          // b29  GPIO2 b12 Interrupt mask
#define GPIO2_12_IRQ_MASK   0x40000000          // b30  GPIO2 b13 Interrupt mask


//------------------------------------------------------------------------------------
//    SysCfg & PMU Register Definitions  (0x4003 5800 - 0x4003 5FFF)
//------------------------------------------------------------------------------------

#define SYS_CFG_RST_REG_BASE       (SYSCFG_REG_BASE | SYSCFG_RST_REG)
typedef struct {
    uint32_t                    // Defaults:
        SSI_RST0_N      : 1,    // 0
        SSI_RST1_N      : 1,    // 0
        SSI_RST2_N      : 1,    // 0
        SSI2S0_SRESETN  : 1,    // 0
        I2S1_CH4_SRESETN: 1,    // 0
        IC_I2C_RST_N    : 1,    // 0
        TIMER_1_RESETN  : 1,    // 0
        RSTB            : 1,    // 0
        SHRESETN        : 1,    // 0
        MHRESETN        : 1,    // 0
        S0_DUT_PRESETN  : 1,    // 0
        S1_DUT_PRESETN  : 1,    // 0
        SYS_CFG_RSTN    : 1,    // 0
        SYSTEM_RESET    : 1,    // 0
        SPEED_UP_WDT_SYS_CFG: 1,// 0
        SECONDARY_WD_ENB: 1;    // 0
} SYS_CFG_RST_REG_Typedef;
#define SYS_CFG_RST_REG  ((volatile SYS_CFG_RST_REG_Typedef *) SYS_CFG_RST_REG_BASE)


//============================================================================

#define CLK_CNTRL_REG1_BASE       (SYSCFG_REG_BASE | SYSCFG_CLK_CTRL_REG1)
typedef struct {
    uint32_t                        // Defaults:
        IO_SS_IN_N          : 1,    // 1 - bit 0
        IO_SS_IN_1          : 1,    // 1 - bit 1
        IO_SS_IN_2          : 1,    // 1 - bit 2
        IO_DSR_N_0          : 1,    // 0 - bit 3
        IO_DCD_N_0          : 1,    // 1 - bit 4
        IO_RI_N_0           : 1,    // 1 - bit 5
        IO_DSR_N_1          : 1,    // 0 - bit 6
        IO_DCD_N_1          : 1,    // 1 - bit 7
        IO_RI_N_1           : 1,    // 1 - bit 8
        IO_DSR_N_2          : 1,    // 0 - bit 9
        IO_DCD_N_2          : 1,    // 1 - bit 10
        IO_RI_N_2           : 1,    // 1 - bit 11
        SPEED_UP_WDT_ADC1   : 1,    // 0 - bit 12
        SPEED_UP_WDT_ADC2   : 1,    // 0 - bit 13
        SPEED_UP_WDT_AES    : 1,    // 0 - bit 14
        Reserved_15_17      : 3,    //   - bit 15-17
        INTFSEL             : 1,    // 1 - bit 18
        I2S1_CLK_STG1_SEL   : 1,    // 0 - bit 19  DIV4 = 1
        Reserved_20         : 1,    //   - bit 20
        CODEC_CLK_SEL       : 1,    // 0 - bit 21  Selection for CODEC clk
        I2S1_CLK_SEL        : 1,    // 0 - bit 22
        CODEC_WORD_SEL      : 1,    // 0 - bit 23
        CODEC_DATA_INPUT    : 1,    // 0 - bit 24
        I2S0_CLK_SEL        : 1,    // 0 - bit 25
        I2S0_CLK_STG1_SEL   : 1,    // 0 - bit 26  Muxed internal signal
        Reserved_27_31      : 5;    //   - bit 27-31
} CLK_CNTRL_REG1_Typedef;
#define CLK_CNTRL_REG1  ((volatile CLK_CNTRL_REG1_Typedef *) CLK_CNTRL_REG1_BASE)

//============================================================================
#define PLL0_REG0_BASE        (SYSCFG_REG_BASE | SYSCFG_PLL0_REG0)  // Main
#define PLL1_REG0_BASE        (SYSCFG_REG_BASE | SYSCFG_PLL1_REG0)  // Audio
typedef struct {
    uint32_t                    // Defaults:
        REFDIV          : 6,    // 1
        FBDIV           : 12,   // 23
        POSTDIV1        : 3,    // 3
        POSTDIV2        : 3,    // 2
        DIVVAL          : 4,
        Reserved_24_31  : 1;
} PLL_REG0_Typedef;
#define PLL0_REG0_VAULE   ((volatile PLL_REG0_Typedef *)  PLL0_REG0_BASE)
#define PLL1_REG0_VAULE   ((volatile PLL_REG0_Typedef *)  PLL1_REG0_BASE)

//============================================================================
#define PLL0_REG1_BASE       (SYSCFG_REG_BASE | SYSCFG_PLL0_REG1)   // Main
#define PLL1_REG1_BASE       (SYSCFG_REG_BASE | SYSCFG_PLL1_REG1)   // Audio
typedef struct {
    uint32_t                    // Defaults:
        FRAC            : 24,   // 2653381
        PD              : 1,    // 1
        DACPD           : 1,    // 0
        DSMPD           : 1,    // 0
        FOUTPOSTDIVPD   : 1,    // 0
        FOUT4PHASEPD    : 1,    // 0
        FOUTVCOPD       : 1,    // 0
        BYPASS          : 1,    // 0
        Reserved_31     : 1;
} PLL_REG1_Typedef;
#define PLL0_REG1_VAULE  ((volatile PLL_REG1_Typedef *) PLL0_REG1_BASE)
#define PLL1_REG1_VAULE  ((volatile PLL_REG1_Typedef *) PLL1_REG1_BASE)

//============================================================================

#define OUTPUT_ENABLE_REG0_BASE       (SYSCFG_REG_BASE | SYSCFG_PAD_ENB_REG0)
typedef struct {
    uint32_t                    // Defaults:
        P01             : 2,    // 0
        P02             : 2,    // 0
        P03             : 2,    // 0
        P04             : 2,    // 0
        Reserved_8_15   : 8,
        P09             : 2,    // 0
        P10             : 2,    // 0
        P11             : 2,    // 0
        P12             : 2,    // 0
        P13             : 2,    // 0
        P14             : 2,    // 0
        P15             : 2,    // 0
        P16             : 2;    // 0
} OUTPUT_ENABLE_REG0_Typedef;
#define OUTPUT_ENABLE_REG0  ((volatile OUTPUT_ENABLE_REG0_Typedef *) OUTPUT_ENABLE_REG0_BASE)

//============================================================================
/*
#define OUTPUT_ENABLE_REG1_BASE       (SYSCFG_REG_BASE | SYSCFG_PAD_ENB_REG1)
typedef struct {
    uint32_t                    // Defaults:
        P17             : 2,    // 0
        Reserved_2_13   : 12,
        P24             : 2,
        P25             : 2,
        P26             : 2,
        P27             : 2,
        Reserved_22_31  : 10;
} OUTPUT_ENABLE_REG1_Typedef;
#define OUTPUT_ENABLE_REG1  ((volatile OUTPUT_ENABLE_REG1_Typedef *) OUTPUT_ENABLE_REG1_BASE)
*/

//============================================================================


typedef struct{
    uint8_t PINMAP;
    uint8_t PINMUX;
    uint8_t PININPUT;
    uint8_t PINOUTPUT;
}PIN_MAP;

typedef enum{
    RESET_SPI0=0,
    RESET_SPI1,
    RESET_SPI2,
    RESET_I2S0,
    RESET_I2S1,
    RESET_I2C0,
    RESET_Timer,
    RESET_Codec,
    RESET_AhbSlave,
    RESET_AhbMaster,
    RESET_ApbPeriperals_1,
    RESET_ApbPeriperals_2,
    RESET_ApbSysCfg,
    RESET_System,
    RESET_All=0x3fff
}ResetInterface;

typedef enum{
    IRQ_SPI0=0,
    IRQ_SPI1,
    IRQ_SPI2,
    IRQ_I2S0,
    IRQ_UART0,
    IRQ_UART1,
    IRQ_UART2,
    IRQ_I2S1,
    IRQ_I2C0,
    IRQ_Timer,
    IRQ_GPIO,//IRQ_GPIO3,
    IRQ_WatchDog,
    IRQ_ADC0,
    IRQ_ADC1,
    IRQ_AES,
    IRQ_Reserved_15,
    IRQ_Reserved_16,
    IRQ_DMAC,
    IRQ_GPIO2,    /*13 bit for GPI02 MASK_12_BIT(18)*/
    IRQ_All=32
}IrqFlag;

typedef enum{
    CLOCK_AHB_DMAC=0,
    CLOCK_APB0_SPI0,
    CLOCK_APB0_SPI1,
    CLOCK_APB0_I2S0,
    CLOCK_APB0_UART0,
    CLOCK_APB0_UART1,
    CLOCK_APB0_I2S1,
    CLOCK_APB0_SPI2,
    CLOCK_APB0_I2C0,
    CLOCK_APB1_WatchDog,
    CLOCK_APB1_ADC0,
    CLOCK_APB1_ADC1,
    CLOCK_APB1_AES,
    CLOCK_APB0_UART2,
    CLOCK_26MToPLL0,
    CLOCK_26MToPLL1,
    CLOCK_APB0_CodecPclk,
    CLOCK_AllOn=32,
    CLOCK_AllOff
}ClockFlag;

#define MASK_BIT(mask, shift) ((uint32_t)(mask) << (shift))
#define MASK_1_BIT(org) ((uint32_t)0x1 << (org))
#define MASK_2_BIT(org) ((uint32_t)0x3 << (org))
#define MASK_3_BIT(org) ((uint32_t)0x7 << (org))
#define MASK_4_BIT(org) ((uint32_t)0xf << (org))
#define MASK_5_BIT(org) ((uint32_t)0x1f << (org))
#define MASK_6_BIT(org) ((uint32_t)0x3f << (org))
#define MASK_7_BIT(org) ((uint32_t)0x7f << (org))
#define MASK_8_BIT(org) ((uint32_t)0xff << (org))
#define MASK_9_BIT(org)    ((uint32_t)0x1ff << (org))
#define MASK_10_BIT(org) ((uint32_t)0x3ff << (org))
#define MASK_11_BIT(org) ((uint32_t)0x7ff << (org))
#define MASK_12_BIT(org) ((uint32_t)0xfff << (org))
#define MASK_13_BIT(org) ((uint32_t)0x1fff << (org))
#define MASK_14_BIT(org) ((uint32_t)0x3fff << (org))
#define MASK_15_BIT(org) ((uint32_t)0x7fff << (org))
#define MASK_16_BIT(org) ((uint32_t)0xffff << (org))
#define MASK_17_BIT(org) ((uint32_t)0x1ffff << (org))
#define MASK_18_BIT(org) ((uint32_t)0x3ffff << (org))
#define MASK_19_BIT(org) ((uint32_t)0x7ffff << (org))
#define MASK_20_BIT(org) ((uint32_t)0xfffff << (org))
#define MASK_21_BIT(org) ((uint32_t)0x1fffff << (org))
#define MASK_22_BIT(org) ((uint32_t)0x3fffff << (org))
#define MASK_23_BIT(org) ((uint32_t)0x7fffff << (org))
#define MASK_24_BIT(org) ((uint32_t)0xffffff << (org))
#define MASK_25_BIT(org) ((uint32_t)0x1ffffff << (org))
#define MASK_26_BIT(org) ((uint32_t)0x3ffffff << (org))
#define MASK_27_BIT(org) ((uint32_t)0x7ffffff << (org))
#define MASK_28_BIT(org) ((uint32_t)0xfffffff << (org))
#define MASK_29_BIT(org) ((uint32_t)0x1fffffff << (org))
#define MASK_30_BIT(org) ((uint32_t)0x3fffffff << (org))
#define MASK_31_BIT(org) ((uint32_t)0x7fffffff << (org))
#ifndef MASK_BIT
    #define MASK_BIT(mask, shift) ((uint32_t)(mask) << (shift))
#endif


//I2S clock select config
typedef enum{I2S1_PLL1_sel, I2S1_PLL1_4div_sel, I2S1_SynCodec_sel} I2S1ClockSelect;
void SYS_I2S1SelectClock(I2S1ClockSelect sel);
typedef enum{I2S0_PLL1_sel, I2S0_PLL1_4div_sel, I2S0_I2S0CK_sel} I2S0ClockSelect;
void SYS_I2S0SelectClock(I2S0ClockSelect sel);

//Inner Codec port select config
typedef enum{Parallel_sel, Serial_sel} CodecInterfaceSelect;
void SYS_CodecSelectInterface(CodecInterfaceSelect sel);
typedef enum{Codec_PLL1_sel, Codec_MClk1_sel} CodecClockSelect;
void SYS_CodecSelectClock(CodecClockSelect sel);
typedef enum{I2S1Ws_sel, I2S1WsSyno_sel} CodecWsSelect;
void SYS_CodecSelectWS(CodecWsSelect sel);
typedef enum{ I2S1_sel, I2S1TxSyno_sel } CodecDataInSelect;
void SYS_CodecSelectDataIn(CodecDataInSelect sel);

//Periperals or interfaces reset config
void SYS_ResetInterface(ResetInterface Flag);
void SYS_SetPeripheralResetForWdt(ResetInterface Flag, FunctionalState NewState);

//System clock select config
typedef enum {CLOCK_SRC_Osc = 0, CLOCK_SRC_Pll0} SystemClockSelect;
void SYS_SystemClockSourceSelect(SystemClockSelect Sel);

//Pin remap config
void SYS_SetPinRemap0(uint32_t PinNum, uint32_t Sel);
void SYS_SetPinRemap1(uint32_t PinNum, uint32_t Sel);
void SYS_PinRemap(uint8_t PinNum, uint8_t mode);

//Get boot_pin and Pll status
FlagStatus SYS_GetBootPinFlag(void);
FlagStatus SYS_GetPLL0LockStatus(void);
FlagStatus SYS_GetPLL1LockStatus(void);

//Default Pll0 and Pll1 config
void SYS_InitPLL0_100Mhz(void);
void SYS_InitPLL1_12d288Mhz(void);
void SYS_SetPLL0_xxxMhz(uint16_t Mhz);

//peripherals' clock config
void SYS_SetClockEnableReg(uint32_t value);

//watchdog clock config
void SYS_WdtClockCmd(FunctionalState NewState);

//peripherals's reset config when watchdog triggered
void SYS_WdtHardwareResetCmd(FunctionalState NewState);
void SYS_SetITMaskFlag(IrqFlag Flag, FlagStatus NewStatus);
#define WDT_RESET_CTRL_Bit            MASK_1_BIT(15)

void SYS_SetClock(ClockFlag Flag, FunctionalState NewState);

//output_enable_reg0
#define PINMUX_PIN_01   (0)
#define PINMUX_PIN_02   (2)
#define PINMUX_PIN_03   (4)
#define PINMUX_PIN_04   (6)
#define PINMUX_PIN_09   (16)
#define PINMUX_PIN_10   (18)
#define PINMUX_PIN_11   (20)
#define PINMUX_PIN_12   (22)
#define PINMUX_PIN_13   (24)
#define PINMUX_PIN_14   (26)
#define PINMUX_PIN_15   (28)
#define PINMUX_PIN_16   (30)
#define REMAP0_PIN_01_Sel_SPI0_SDO          (0x00)
#define REMAP0_PIN_01_Sel_UART1_CTS         (0x01)
#define REMAP0_PIN_02_Sel_SPI0_SSN          (0x00)
#define REMAP0_PIN_02_Sel_UART1_RTS         (0x01)
#define REMAP0_PIN_03_Sel_SPI0_SCK          (0x00)
#define REMAP0_PIN_03_Sel_UART1_TX          (0x01)
#define REMAP0_PIN_04_Sel_SPI0_SDI          (0x00)
#define REMAP0_PIN_04_Sel_UART1_RX          (0x01)
#define REMAP0_PIN_04_Sel_GPIO3_8_OUTPUT    (0x02)
#define REMAP0_PIN_04_Sel_GPIO3_8_INPUT     (0x03)
#define REMAP0_PIN_09_Sel_JTDI              (0x00)
#define REMAP0_PIN_09_Sel_UART2_RX          (0x01)
#define REMAP0_PIN_09_Sel_GPIO3_10_OUTPUT   (0x02)
#define REMAP0_PIN_09_Sel_GPIO3_10_INPUT    (0x03)
#define REMAP0_PIN_10_Sel_JTDO              (0x00)
#define REMAP0_PIN_10_Sel_UART2_TX          (0x01)
#define REMAP0_PIN_10_Sel_GPIO3_11_OUTPUT   (0x02)
#define REMAP0_PIN_10_Sel_GPIO3_11_INPUT    (0x03)
#define REMAP0_PIN_11_Sel_TRSTn             (0x00)
#define REMAP0_PIN_11_Sel_UART2_CTS         (0x01)
#define REMAP0_PIN_11_Sel_GPIO3_9_OUTPUT    (0x02)
#define REMAP0_PIN_11_Sel_GPIO3_9_INPUT     (0x03)
#define REMAP0_PIN_12_Sel_GPIO3_0_OUTPUT    (0x00)
#define REMAP0_PIN_12_Sel_UART2_RTS         (0x01)
#define REMAP0_PIN_12_Sel_GPIO3_0_INPUT     (0x02)
//#define REMAP0_PIN_12_Sel_GPIO3_9_INPUT            (0x11)
#define REMAP0_PIN_13_Sel_GPIO3_1_OUTPUT            (0x00)
#define REMAP0_PIN_13_Sel_I2S0_WS                   (0x01)
#define REMAP0_PIN_13_Sel_SPI1_SSN                  (0x02)
#define REMAP0_PIN_13_Sel_GPIO3_1_INPUT             (0x03)
#define REMAP0_PIN_13_Sel_I2S1_WsSynoCodec_INPUT    (0x03)
#define REMAP0_PIN_14_Sel_GPIO3_2_OUTPUT            (0x00)
#define REMAP0_PIN_14_Sel_I2S0_ClkSelStg1_OUPTUT    (0x01)
#define REMAP0_PIN_14_Sel_SPI1_SCK                  (0x02)
#define REMAP0_PIN_14_Sel_GPIO3_2_INPUT             (0x03)
#define REMAP0_PIN_14_Sel_I2S1_ClkSynoCodec_INPUT   (0x03)
#define REMAP0_PIN_14_Sel_I2S0_CLOCK                (0x03)
#define REMAP0_PIN_15_Sel_GPIO3_3_OUTPUT            (0x00)
#define REMAP0_PIN_15_Sel_GPIO3_3_INPUT             (0x01)
#define REMAP0_PIN_15_Sel_SPI1_SDI                  (0x02)
#define REMAP0_PIN_15_Sel_I2S0_RX                   (0x02)
#define REMAP0_PIN_15_Sel_I2S1_RxSynoCodec_INPUT    (0x03)
#define REMAP0_PIN_16_Sel_GPIO3_4_OUTPUT            (0x00)
#define REMAP0_PIN_16_Sel_I2S0_TX                   (0x01)
#define REMAP0_PIN_16_Sel_SPI1_SDO                  (0x02)
#define REMAP0_PIN_16_Sel_GPIO3_4_INPUT             (0x03)
#define REMAP0_PIN_16_Sel_I2S1_TxSynoCodec_INPUT    (0x03)

//output_enable_reg0 and output_enable_reg1
#define PINMUX_PIN_17   (0)
#define PINMUX_PIN_24   (14)
#define PINMUX_PIN_25   (16)
#define PINMUX_PIN_26   (18)
#define PINMUX_PIN_27   (20)
#define REMAP1_PIN_17_Sel_GPIO3_5_OUTPUT            (0x00)
#define REMAP1_PIN_17_Sel_GPIO3_5_INPUT             (0x01)
#define REMAP1_PIN_17_Sel_MsClk1SynoCodec           (0x02)
#define REMAP1_PIN_24_Sel_GPIO3_6_OUTPUT            (0x01)//(0x00)
#define REMAP1_PIN_24_Sel_TCK_SWCLK                 (0x02)
#define REMAP1_PIN_24_Sel_GPIO3_6_INPUT             (0x03)
#define REMAP1_PIN_25_Sel_SWDIOTMS_OUTPUT           (0x00)
#define REMAP1_PIN_25_Sel_GPIO3_7_OUTPUT            (0x01)
#define REMAP1_PIN_25_Sel_GPIO3_7_INPUT             (0x02)
#define REMAP1_PIN_25_Sel_SWDIOTMS_INPUT            (0x03)
#define REMAP1_PIN_26_I2C0_SDA                      (0x00)
//#define REMAP1_PIN_25_Sel_GPIO3_7_OUTPUT            (0x01)
#define REMAP1_PIN_27_I2C0_SCL                      (0x00)
//spread_spectrum_pll1

#define PLL1_SpeadAmplitude     MASK_5_BIT(0)
#define PLL1_DownSpread         MASK_1_BIT(5)
#define PLL1_WaveSelect         MASK_1_BIT(6)
#define PLL1_ExtWaveVal         MASK_8_BIT(7)
#define PLL1_ExtMaxVal          MASK_8_BIT(15)
#define PLL1_DisableSpread      MASK_1_BIT(23)
#define PLL1_WatchdogClock      MASK_1_BIT(26)

//------------------------------------------------------------------
// API prototypes
//------------------------------------------------------------------

/**
  * @brief  Get interrupt flag polarity
  * @param  Flag: Interrupt flag
  * @retval Interrupt flag
  */
uint32_t  SYS_GetITPolarityFlag(IrqFlag Flag);

/**
  * @brief  Set interrupt flag polarity
  * @param  Flag: Interrupt flag
  * @retval none
  */
void  SYS_SetITPolarityFlag(IrqFlag Flag, FlagStatus);

/**
  * @brief  Get interrupt request mask
  * @param  Flag: Interrupt request mask
  * @retval Interrupt request mask
  */
uint32_t  SYS_GetITMaskFlag(IrqFlag IntFlag);

/**
  * @brief  Set interrupt status
  * @param  IntFlag: Interrupt flag
  * @param  IntFlag: Interrupt flag
  * @retval none
  */
void  SYS_SetITMaskFlag(IrqFlag IntFlag, FlagStatus NewStatus);

/**
  * @brief  Get system interrrupt status
  * @retval System interrupt status
  */
uint32_t     SYS_GetITStatus(void);

/**
  * @brief  Get system interrrupt status flag
  * @param  Flag: System interrupt flag
  * @retval SET or RESET
  */
FlagStatus  SYS_GetITFlag(IrqFlag Flag);

/**
  * @brief  Clear system interrrupt flag
  * @param  Flag: System interrupt flag
  * @retval none
  */
void         SYS_ClearITFlag(IrqFlag IntFlag);

#endif /* __ACH118X_SYSCFG_H */
/************************ (C) COPYRIGHT ART *****END OF FILE****/

