/***************************************************************************************************
 *                              MCU ACH118X driver library
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     ach118x_spi.h
 * @brief    SPI features and registers.
 * @version  V1.0.0
 * @date     05-Nov-2020
 * @detail
 *   - Master/Slave operations supported: Enables serial communication with serial-master
 *     or serial-slave peripheral devices.
 *   - Independent masking of interrupts: Master collision, transmit FIFO overflow, transmit FIFO empty,
 *     receive FIFO full, receive FIFO underflow, and receive FIFO overflow interrupts can all be masked
 *     independently.
 *   - Programmable delay on the sample time of the received serial data bit (rxd);
 *     enables programmable  control of routing delays resulting in higher serial data-bit rates.
 *   - Serial interface operation � Choice of Motorola SPI, Texas Instruments Synchronous Serial Protocol
 *     or National Semiconductor Microwire.
 *   - Clock bit-rate: Dynamic control of the serial bit rate of the data transfer; used in only
 *     serial-master mode of operation.
 *   - Data Item size (4 to 32 bits) � Item size of each data transfer under the control of the programmer.
 *   - FIFO depth 22 words deep. The FIFO width is fixed at 16/32 bits, depending upon the SSI_MAX_XFER_SIZE
 *     parameter.
 *   - Number of slave select outputs: 1 to 16 serial slave-select output signals can be generated.
 *   - Hardware/software slave-select: Dedicated hardware slave-select lines can be used or software
 *     control can be used to target the serial-slave device.
 ***************************************************************************************************/
#ifndef __ACH118X_SPI_H
#define __ACH118X_SPI_H

#include "ach118x_type.h"

#define SPI_FIFO_DEPTH   (128)    // SPI FIFO Depth

/**
  * @brief  SPI Init structure definition
  */
typedef struct
{
    uint16_t TRANS_TYPE;
    uint16_t ADDR_L;
    uint16_t INST_L;
    uint16_t WAIT_CYCLES;
    //-------------------
    uint16_t SPI_FRF;
    uint16_t DFS_32;
    uint16_t CFS;
    uint16_t SRL;
    uint16_t SLV_OE;
    uint16_t TMOD;
    uint16_t SCPOL;
    uint16_t SCPH;
    uint16_t FRF;
    uint16_t DFS;
    //-------------------
    uint16_t NumOfDataFrame;
    uint16_t ClockDivider;
    uint16_t TX_FIFO_ThresholdLevel;
    uint16_t RX_FIFO_ThresholdLevel;
    uint32_t SlaveSlect;
}SPI_InitTypeDef;

/*------------------------ Serial Peripheral Interface -----------------------*/
/*SPI register define*/

typedef struct {
    vu32
        DFS                 : 4,
        FRF                 : 2,
        SCPH                : 1,
        SCPOL               : 1,
        TMOD                : 2,
        SLV_OE              : 1,
        SRL                 : 1,
        CFS                 : 4,
        DFS_32              : 5,
        SPI_FRF             : 2,
        Reserved_31_23      : 9;
} SSI_CTRLR0_Typedef;

typedef struct {
    vu32
        TRANS_TYPE          : 2,
        ADDR_L              : 4,
        Reserved_7_6        : 2,
        INST_L              : 2,
        Reserved_10         : 1,
        WAIT_CYCLES         : 4,
        Reserved_31_15      : 17;
} SPI_CTRLR0_Typedef;


#define SSI_BUFFER_LEN        36
typedef struct
{
    SSI_CTRLR0_Typedef CTRLR0;      // 0x00  Control Register 0
    vu32 CTRLR1;                    // 0x04  Control Register 1
    vu32 SSIENR;                    // 0x08  SSI Enable register
    vu32 MWCR;                      // 0x0c  Microwire Control Register
    vu32 SER;                       // 0x10  Slave Enable Register
    vu32 BAUDR;                     // 0x14  Baud Rate Select
    vu32 TXFTLR;                    // 0x18  Transmit FIFO Threshold Level
    vu32 RXFTLR;                    // 0x1c  Receive FIFO Threshold Level
    vu32 TXFLR;                     // 0x20  Transmit FIFO Level Register
    vu32 RXFLR;                     // 0x24  Receive FIFO Level Register
    vu32 SR;                        // 0x28  Status Register
    vu32 IMR;                       // 0x2c  Interrupt Mask Register
    vu32 ISR;                       // 0x30  Interrupt Status Register
    vu32 RISR;                      // 0x34  Raw Interrupt Status Register
    vu32 TXOICR;                    // 0x38  Transmit FIFO Overflow Interrupt Clear Register
    vu32 RXOICR;                    // 0x3c  Receive FIFO Overflow Interrupt Clear Register
    vu32 RXUICR;                    // 0x40  Receive FIFO Underflow Interrupt Clear Register
    vu32 MSTICR;                    // 0x44  Multi-Master Interrupt Clear Register
    vu32 ICR;                       // 0x48  Interrupt Clear Register
    vu32 DMACR;                     // 0x4c  DMA Control Register
    vu32 DMATDLR;                   // 0x50  DMA Transmit Data Level register
    vu32 DMARDLR;                   // 0x54  DMA Receive Data Level register
    vu32 IDR;                       // 0x58  Identification Register
    vu32 SSI_COMP_VERSION;          // 0x5c  coreKit version ID register
    vu32 DR[SSI_BUFFER_LEN];        // 0x60-0xEC Data Register
    vu32 RX_SAMPLE_DLY;             // 0xf0  RXD Sample Delay Register
    SPI_CTRLR0_Typedef SPI_CTRLR0;  // 0xf4  SPI control register
    vu32 RSVD_1;                    // 0xf8  Reserved
    vu32 RSVD_2;                    // 0xfc  Reserved
}SPI_Typedef;

/**********Register and Field Description for SPI************/

/**CTRLR0 Bit**/
#define CTRLR0_DQ_WAIR_BIT_SHIFT        (28)
#define CTRLR0_DQ_WAIR_BIT_MASK            (0xf << CTRLR0_DQ_WAIR_BIT_SHIFT)
typedef enum
{
    Wait_0_Cycles = 0,
    Wait_1_Cycle,
    Wait_2_Cycles,
    Wait_3_Cycles,
    Wait_4_Cycles,
    Wait_5_Cycles,
    Wait_6_Cycles,
    Wait_7_Cycles,
    Wait_8_Cycles,
    Wait_9_Cycles,
    Wait_10_Cycles,
    Wait_11_Cycles,
    Wait_12_Cycles,
    Wait_13_Cycles,
    Wait_14_Cycles,
    Wait_15_Cycles
}DQ_WAIT_CYCLES;

#define CTRLR0_DQ_INST_BIT_SHIFT        (26)
#define CTRLR0_DQ_INST_BIT_MASK            (0x3 << CTRLR0_DQ_INST_BIT_SHIFT)
typedef enum
{
    Inst_len_0_bit = 0,
    Inst_len_4_bits,
    Inst_len_8_bits,
    Inst_len_12_bits,
}DQ_INST_LEN;

#define CTRLR0_SPI_TRANS_BIT_SHIFT        (24)
#define CTRLR0_SPI_TRANS_BIT_MASK        (0x3 << CTRLR0_SPI_TRANS_BIT_SHIFT)
typedef enum
{
    TRANS_Standard_Mode = 0,
    TRANS_Mixed_Mode,
    TRANS_FRF_Mode,
    TRANS_Resvered_Mode,
}SPI_TRANS_MODE;


#define CTRLR0_SPI_FRF_BIT_SHIFT        (21)
#define CTRLR0_SPI_FRF_BIT_MASK            (0x3 << CTRLR0_SPI_FRF_BIT_SHIFT)
typedef enum
{
    SPI_FRF_Standard = 0,
    SPI_FRF_Dual, // not supported
    SPI_FRF_Quad, // not supported
}SPI_FRF;

#define CTRLR0_DFS_32_BIT_SHIFT            (16)
#define CTRLR0_DFS_32_BIT_MASK            (0x1f << CTRLR0_DFS_32_BIT_SHIFT)
typedef enum
{
    DF32_4_bit = 0x3,
    DF32_5_bit,
    DF32_6_bit,
    DF32_7_bit,
    DF32_8_bit,
    DF32_9_bit,
    DF32_10_bit,
    DF32_11_bit,
    DF32_12_bit,
    DF32_13_bit,
    DF32_14_bit,
    DF32_15_bit,
    DF32_16_bit,
    DF32_17_bit,
    DF32_18_bit,
    DF32_19_bit,
    DF32_20_bit,
    DF32_21_bit,
    DF32_22_bit,
    DF32_23_bit,
    DF32_24_bit,
    DF32_25_bit,
    DF32_26_bit,
    DF32_27_bit,
    DF32_28_bit,
    DF32_29_bit,
    DF32_30_bit,
    DF32_31_bit,
    DF32_32_bit,
}SPI_DFS32;
#define SPI_DFS_BIT(x)    (x-1)

#define CTRLR0_CFS_BIT_SHIFT            (12)
#define CTRLR0_CFS_BIT_MASK                (0xf << CTRLR0_CFS_BIT_SHIFT)
typedef enum
{
    CFS_Address_0_bit = 0,
    CFS_Address_4_bit,
    CFS_Address_8_bit,
    CFS_Address_12_bit,
    CFS_Address_16_bit,
    CFS_Address_24_bit,
    CFS_Address_28_bit,
    CFS_Address_32_bit,
    CFS_Address_36_bit,
    CFS_Address_40_bit,
    CFS_Address_44_bit,
    CFS_Address_48_bit,
    CFS_Address_52_bit,
    CFS_Address_56_bit,
    CFS_Address_60_bit
}SPI_CFS_ADDRESS_WIDTH;

typedef enum
{
    DFS_Control_1_bit=0,
    DFS_Control_2_bit,
    DFS_Control_3_bit,
    DFS_Control_4_bit,
    DFS_Control_5_bit,
    DFS_Control_6_bit,
    DFS_Control_7_bit,
    DFS_Control_8_bit,
    DFS_Control_9_bit,
    DFS_Control_10_bit,
    DFS_Control_11_bit,
    DFS_Control_12_bit,
    DFS_Control_13_bit,
    DFS_Control_14_bit,
    DFS_Control_15_bit,
    DFS_Control_16_bit
}SPI_DFS32_CTRL_WORD;

#define CTRLR0_SRL_BIT_SHIFT            (11)
#define CTRLR0_SRL_BIT_MASK                (0x1 << CTRLR0_SRL_BIT_SHIFT)
#define SRL_NORMAL_Mode                    (0)
#define SRL_Test_Mode                    (1)

#define CTRLR0_SLV_OE_BIT_SHIFT            (10)
#define CTRLR0_SLV_OE_BIT_MASK            (0x1 << CTRLR0_SLV_OE_BIT_SHIFT)
#define SLAVE_TXD_Enable                (0)
#define SLAVE_TXD_Disable                (1)

#define CTRLR0_TMOD_BIT_SHIFT            (8)
#define CTRLR0_TMOD_BIT_MASK            (0x3 << CTRLR0_TMOD_BIT_SHIFT)
typedef enum
{
    Mode_Tx_and_Rx = 0,
    Mode_Tx_only,
    Mode_Rx_only,
    Mode_E2PROM_read
}SPI_TXRX_MODE;

#define CTRLR0_SCPOL_BIT_SHIFT            (7)
#define CTRLR0_SCPOL_BIT_MASK            (0x1 << CTRLR0_SCPOL_BIT_SHIFT)
typedef enum
{
    SCPOL_low = 0,
    SCPOL_high,
}SPI_SCPOL;

#define CTRLR0_SCPH_BIT_SHIFT            (6)
#define CTRLR0_SCPH_BIT_MASK            (0x1 << CTRLR0_SCPH_BIT_SHIFT)
typedef enum
{
    SCPH_middle_1st = 0,
    SCPH_start_1st,
}SPI_SCPH;

#define CTRLR0_FRF_BIT_SHIFT            (4)
#define CTRLR0_FRF_BIT_MASK                (0x3 << CTRLR0_FRF_BIT_SHIFT)
typedef enum
{
    FRF_Motorola = 0,
    FRF_Texas,
    FRF_Microwire,
    FRF_Reserved
}FRF;


#define CTRLR0_DFS_BIT_SHIFT            (0)
#define CTRLR0_DFS_BIT_MASK                (0xf << CTRLR0_DFS_BIT_SHIFT)
typedef enum
{
    DFS16_4_bit=0x3,
    DFS16_5_bit,
    DFS16_6_bit,
    DFS16_7_bit,
    DFS16_8_bit,
    DFS16_9_bit,
    DFS16_10_bit,
    DFS16_11_bit,
    DFS16_12_bit,
    DFS16_13_bit,
    DFS16_14_bit,
    DFS16_15_bit,
    DFS16_16_bit
}SPI_DFS;

/**CTRLR1 Bit**/
#define CTRLR1_NDF_BIT_SHIFT        (0)
#define CTRLR1_NDF_BIT_MASK            (0xff << CTRLR1_NDF_BIT_SHIFT)

/**SSIENR Bit**/
#define CTRLR1_ENABLE_BIT_SHIFT        (0)
#define SSIENR_ENABLE_BIT_MASK        (1 << CTRLR1_ENABLE_BIT_SHIFT)
#define SSI_ENABLE    (0)
#define SSI_DISABLE    (1)

/**MWCR Bit**/
#define MWCR_MHS_BIT_SHIFT            (2)
#define MWCR_MHS_BIT_MASK            (1 << MWCR_MHS_BIT_SHIFT)
#define MHS_Disable                    (0)
#define MHS_Enable                    (1)

#define MWCR_MOD_BIT_SHIFT            (1)
#define MWCR_MOD_BIT_MASK            (1 << MWCR_MOD_BIT_SHIFT)
#define MOD_receieve                (0)
#define MOD_transmit                (1)

#define MWCR_MWMOD_BIT_SHIFT        (0)
#define MWCR_MWMOD_BIT_MASK            (1 << MWCR_MWMOD_BIT_SHIFT)
#define MWMOD_non_sequential        (0)
#define MWMOD_sequential            (1)

/**SER Bit**/
#define SSI_NUM_SLAVES                (1)//This dues to IP hardware
#define SSI_NUM_SLAVES_BIT_SHIFT    (0)
#define SSI_NUM_SLAVES_BIT_MASK        (SSI_NUM_SLAVES << SSI_NUM_SLAVES_BIT_SHIFT)
#define SLAVE_Not_selected            (0)
#define SLAVE_selected                (1)

/**BAUDR Bit**/
#define BAUDR_SCKDV_BIT_SHIFT        (0)
#define BAUDR_SCKDV_BIT_MASK        (0xff << BAUDR_SCKDV_BIT_SHIFT)

/**TXFTLR Bit**/
//#define TX_ABW                    (7)
#define TXFTLR_TFT_BIT_SHIFT        (0)
#define TXFTLR_TFT_BIT_MASK            (0xff << TXFTLR_TFT_BIT_SHIFT)

/**TXFTLR Bit**/
//#define TX_ABW                    (7)
#define TXFTLR_TXTFL_BIT_SHIFT        (0)
#define TXFTLR_TXTFL_BIT_MASK        (0xff << TXFTLR_TXTFL_BIT_SHIFT)

/**RXFLR Bit**/
//#define RX_ABW                    (7)
#define RXFLR_RXTFL_BIT_SHIFT        (0)
#define RXFLR_RXTFL_BIT_MASK        (0xff << RXFTLR_RTXTFL_BIT_SHIFT)

/**SR Bit**/
#define SSI_SR_BUSY     0x01    // SSI Busy Flag
#define SSI_SR_TFNF     0x02    // Transmit FIFO Not Full
#define SSI_SR_TFE      0x04    // Transmit FIFO Empty
#define SSI_SR_RFNE     0x08    // Receive FIFO Not Empty
#define SSI_SR_RFF      0x10    // Receive FIFO Full
#define SSI_SR_TXE      0x20    // Transmission Error
#define SSI_SR_DCOL     0x40    // Data Collision Error

#define SR_DCOL_BIT_SHIFT       (6)
#define SR_DCOL_BIT_MASK        (1 << SR_DCOL_BIT_SHIFT)
#define SR_DCOL_Flag            (1 << SR_DCOL_BIT_SHIFT)

#define SR_TXE_BIT_SHIFT        (5)
#define SR_TXE_BIT_MASK         (1 << SR_TXE_BIT_SHIFT)
#define SR_TXE_Flag             (1 << SR_TXE_BIT_SHIFT)

#define SR_RFF_BIT_SHIFT        (4)
#define SR_RFF_BIT_MASK         (1 << SR_RFF_BIT_SHIFT)
#define SR_RFF_Flag             (1 << SR_RFF_BIT_SHIFT)

#define SR_RFNE_BIT_SHIFT       (3)
#define SR_RFNE_BIT_MASK        (1 << SR_RFNE_BIT_SHIFT)
#define SR_RFNE_Flag            (1 << SR_RFNE_BIT_SHIFT)

#define SR_TFE_BIT_SHIFT        (2)
#define SR_TFE_BIT_MASK         (1 << SR_TFE_BIT_SHIFT)
#define SR_TFE_Flag             (1 << SR_TFE_BIT_SHIFT)

#define SR_TFNF_BIT_SHIFT       (1)
#define SR_TFNF_BIT_MASK        (1 << SR_TFNF_BIT_SHIFT)
#define SR_TFNF_Flag            (1 << SR_TFNF_BIT_SHIFT)

#define SR_BUSY_BIT_SHIFT       (0)
#define SR_BUSY_BIT_MASK        (1 << SR_TFNE_BIT_SHIFT)
#define SR_BUSY_Flag            (1 << SR_TFNE_BIT_SHIFT)

//For register IMR ISR RISR
#define SSI_IT_TXE      0x01    // Transmit FIFO Empty
#define SSI_IT_TXO      0x02    // Transmit FIFO Overflow
#define SSI_IT_RXU      0x04    // Receive FIFO Underflow
#define SSI_IT_RXO      0x08    // Receive FIFO Overflow
#define SSI_IT_RXF      0x10    // Receive FIFO Full
#define SSI_IT_MST      0x20    // Multi-Master Contention

#define IT_MST_BIT_SHIFT        (5)
#define IT_MASK_MST_Flag        (1 << IT_MST_BIT_SHIFT)
#define IT_RAW_MST_Flag         (1 << IT_MST_BIT_SHIFT)
//#define IT_MST_Flag                (1 << IT_MST_BIT_SHIFT)

#define IT_RXF_BIT_SHIFT        (4)
#define IT_MASK_RXF_Flag        (1 << IT_RXF_BIT_SHIFT)
#define IT_RAW_RXF_Flag         (1 << IT_RXF_BIT_SHIFT)
//#define IT_RXF_Flag                (1 << IT_RXF_BIT_SHIFT)

#define IT_RXO_BIT_SHIFT        (3)
#define IT_MASK_RXO_Flag        (1 << IT_RXO_BIT_SHIFT)
#define IT_RAW_RXO_Flag         (1 << IT_RXO_BIT_SHIFT)
//#define IT_RXO_Flag                (1 << IT_RXO_BIT_SHIFT)

#define IT_RXU_BIT_SHIFT        (2)
#define IT_MASK_RXU_Flag        (1 << IT_RXU_BIT_SHIFT)
#define IT_RAW_RXU_Flag         (1 << IT_RXU_BIT_SHIFT)
//#define IT_RXU_Flag                (1 << IT_RXU_BIT_SHIFT)

#define IT_TXO_BIT_SHIFT        (1)
#define IT_MASK_TXO_Flag        (1 << IT_TXO_BIT_SHIFT)
#define IT_RAW_TXO_Flag         (1 << IT_TXO_BIT_SHIFT)
//#define IT_TXO_Flag                (1 << IT_TXO_BIT_SHIFT)

#define IT_TXE_BIT_SHIFT        (1)
#define IT_MASK_TXE_Flag        (1 << IT_TXE_BIT_SHIFT)
#define IT_RAW_TXE_Flag         (1 << IT_TXE_BIT_SHIFT)
//#define IT_TXE_Flag                (1 << IT_TXE_BIT_SHIFT)
typedef enum{
    IT_TXE_Flag=0x01,
    IT_TXO_Flag=0x02,
    IT_RXU_Flag=0x04,
    IT_RXO_Flag=0x08,
    IT_RXF_Flag=0x10,
    IT_MST_Flag=0x20,
    IT_ALL_Flag=0x3f
}IT_Flag;
typedef enum{
    ST_BUSY_Flag=0x01,
    ST_TFNF_Flag=0x02,
    ST_TFE_Flag=0x04,
    ST_RFNE_Flag=0x08,
    ST_RFF_Flag=0x10,
    ST_TXE_Flag=0x20,
    ST_DCOL_Flag=0x40,
    ST_ALL_Flag=0x7f
}ST_Flag;


//------------------------------------------------------------------
// API prototypes
//------------------------------------------------------------------

/**
  * @brief  Initialize one structure of the SPIx peripheral with their default reset values.
  * @param  SPI_InitStruct: one SPI initialisation structure
  * @retval none
  */
void  SPI_StructInit(SPI_InitTypeDef* SPI_InitStruct);

/**
  * @brief  Enables or disables the specified SPI peripheral.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  NewState: new state of the SPIx peripheral.
  *          This parameter can be: ENABLE or DISABLE.
  * @retval none
  */
void  SPI_Cmd(SPI_Typedef* SPIx, FunctionalState NewState);


/**
  * @brief  Enable or disable SPIx peripheral.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  SPI_Mask_IT: Its options are
            SR_DCOL_Flag
            SR_TXE_Flag
            SR_RFF_Flag
            SR_RFNE_Flag
            SR_TFE_Flag
            SR_TFNE_Flag
            SR_BUSY_Flag

  * @retval FlagStatus: SET or RESET
  */
FlagStatus  SPI_GetStatusFlag(SPI_Typedef* SPIx, ST_Flag SPI_FLAG);

/**
  * @brief  Enable or disable SPIx peripheral.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  SPI_Mask_IT: Its options are IT_MASK_MST_Flag | IT_MASK_RXF_Flag | IT_MASK_RXO_Flag | IT_MASK_RXU_Flag | IT_MASK_TXO_Flag | IT_MASK_TXE_Flag.
  * @retval SET or RESET
  */
FlagStatus  SPI_GetITMaskFlag(SPI_Typedef* SPIx, uint8_t SPI_Mask_IT);

/**
  * @brief  Enable or disable SPIx peripheral.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  SPI_Mask_IT: Its options are IT_MASK_MST_Flag | IT_MASK_RXF_Flag | IT_MASK_RXO_Flag | IT_MASK_RXU_Flag | IT_MASK_TXO_Flag | IT_MASK_TXE_Flag.
  * @param  NewState: new state of the IT Mask.
  *          This parameter can be: ENABLE or DISABLE.
  * @retval none
  */
void  SPI_ITMaskConfig(SPI_Typedef* SPIx, uint8_t SPI_Mask_IT, FunctionalState NewState);

/**
  * @brief  Enable or disable SPIx peripheral.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  SPI_Mask_IT: Its options are IT_RAW_MST_Flag | IT_RAW_RXF_Flag | IT_RAW_RXO_Flag | IT_RAW_RXU_Flag | IT_RAW_TXO_Flag | IT_RAW_TXE_Flag.
  * @retval SET or RESET
  */
FlagStatus  SPI_GetRawITFlag(SPI_Typedef* SPIx, uint8_t SPI_Mask_IT);


/**
  * @brief  Enable or Disable SPIx peripheral.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  SPI_Mask_IT: Its options are IT_RAW_MST_Flag | IT_RAW_RXF_Flag | IT_RAW_RXO_Flag | IT_RAW_RXU_Flag | IT_RAW_TXO_Flag | IT_RAW_TXE_Flag.
  * @param  NewState: new state of the raw IT.
  *          This parameter can be: ENABLE or DISABLE.
  * @retval none
  */
void  SPI_RawITConfig(SPI_Typedef* SPIx, uint8_t SPI_Raw_IT, FunctionalState NewState);

/**
  * @brief  Clear interrupt register.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @retval FlagStatus: SET or RESET
  */
FlagStatus  SPI_ClearTxFifoOverflowIT(SPI_Typedef* SPIx);

/**
  * @brief  Clear interrupt register.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @retval FlagStatus: SET or RESET
  */
FlagStatus  SPI_ClearRxFifoOverflowIT(SPI_Typedef* SPIx);

/**
  * @brief  Clear interrupt register.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @retval FlagStatus: SET or RESET
  */
FlagStatus  SPI_ClearRxFifoUnderflowIT(SPI_Typedef* SPIx);

/**
  * @brief  Clear interrupt register.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @retval FlagStatus: SET or RESET
  */
FlagStatus  SPI_ClearMultimasterIT(SPI_Typedef* SPIx);

/**
  * @brief  Clear interrupt register.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @retval FlagStatus: SET or RESET
  */
FlagStatus  SPI_ClearITs(SPI_Typedef* SPIx);

/**
  * @brief  SPI send one data
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  Data: Data to send.
  * @retval none
  */
void  SPI_SendData(SPI_Typedef* SPIx, uint32_t Data);

/**
  * @brief  SPI send datas
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  pData: Pointer to buffer, which save datas to send.
  * @param  pData: The lenght of data to send.
  * @retval none
  */
void  SPI_SendDataBuf(SPI_Typedef* SPIx, uint8_t *pData, uint16_t len);

/**
  * @brief  SPI get one data
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  pData: Pointer to buffer, which saves data.
  * @retval none
  */
uint32_t  SPI_GetData(SPI_Typedef* SPIx);

/**
  * @brief  SPI get datas
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  pData: Pointer to buffer, which saves data.
  * @param  pData: The lenght of data to get.
  * @retval none
  */
void  SPI_GetDataBuf(SPI_Typedef* SPIx, uint8_t *pData, uint16_t len);

/**
  * @brief  Send and receive data over SPI
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  pSndBuf: Buffer of data to send.
  * @param  SndLen: Length of data to send.
  * @param  pRcvBuf: Buffer of data to receive.
  * @param  RcvLen: Length of data to receive.
  * @retval none
  */
void  SPI_SendAndGetData(SPI_Typedef* SPIx, uint8_t *pSndBuf, uint32_t SndLen, uint8_t *pRcvBuf, uint32_t RcvLen);

/**
  * @brief  Send and receive data over SPI when data is less or equal to SPI_FIFO_DEPTH
  *         This routine is quick, fast, and short.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  pSndBuf: Buffer of data to send.
  * @param  SndLen: Length of data to send.
  * @param  pRcvBuf: Buffer of data to receive.
  * @param  RcvLen: Length of data to receive.
  * @retval none
  */
void  SPI_SendAndGetDataShort(SPI_Typedef* SPIx, uint8_t *pSndBuf, uint32_t SndLen, uint8_t *pRcvBuf, uint32_t RcvLen);

/**
  * @brief  Send and receive data over SPI. Use SPIx IRQ Handler.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  pSndBuf: Buffer of data to send.
  * @param  SndLen: Length of data to send.
  * @param  pRcvBuf: Buffer of data to receive.
  * @param  RcvLen: Length of data to receive.
  * @retval 0 if success, <0 if error
  */
int  SPI_SendAndGetDataIRQ(SPI_Typedef* SPIx, uint8_t *pSndBuf, uint32_t SndLen, uint8_t *pRcvBuf, uint32_t RcvLen);

/**
  * @brief  Fills each SPI_InitStruct member with its default value.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  SPI_InitStruct: pointer to a SPI_InitTypeDef structure which will be initialized.
  * @retval none
  */
void  SPI_Init(SPI_Typedef* SPIx, SPI_InitTypeDef* SPI_InitStruct);

/**
  * @brief  Get SPI component ID
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @retval Component ID
  */
uint32_t  SPI_GetComponentID(SPI_Typedef* SPIx);

/**
  * @brief  Get SPI component version
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @retval Component version
  */
uint32_t  SPI_GetComponentVersion(SPI_Typedef* SPIx);

/**
  * @brief  Enable SPI System Interrupts. Set flags using
  *         SYSCFG_SYS_LEV_INTR_MASK_REG and NVIC_EnableIRQ.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @retval none
  */
void  SPI_SystemInterruptEnable(SPI_Typedef* SPIx);

/**
  * @brief  Disable SPI System Interrupts. Clear flags using
  *         SYSCFG_SYS_LEV_INTR_MASK_REG and NVIC_DisableIRQ.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @retval none
  */
void  SPI_SystemInterruptDisable(SPI_Typedef* SPIx);

/**
  * @brief  Set SPI interrupt mask bit.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void  SPI_ITCmd(SPI_Typedef* SPIx, IT_Flag Flag, FunctionalState NewState);

/**
  * @brief  Get SPI status flag.
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  Flag: SET or RESET
  * @retval none
  */
FlagStatus  SPI_GetITFlag(SPI_Typedef* SPIx, IT_Flag Flag);

/**
  * @brief  SPI get datas
  * @param  SPIx: where x can be 0, 1, 2 to select the SPI peripheral.
  * @param  pSndBuf: Pointer to buffer, which saves data to send.
  * @param  pData: The lenght of data to send.
  * @param  pRcvBuf: Pointer to buffer, which saves data to get.
  * @param  pData: The lenght of data to get.
  * @param  ShiftIdx: shift between sending and getting action.
  * @retval none
  */
void  SPI_SendAndGetDatas(SPI_Typedef* SPIx, uint8_t *pSndBuf, uint32_t SndLen, uint8_t *pRcvBuf, uint32_t RcvLen, uint16_t ShiftIdx);


#endif /* __ACH118X_SPI_H */
/************************ (C) COPYRIGHT ART *****END OF FILE****/

