/***************************************************************************************************
 *                              MCU ACH118X driver library
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     ach118x_misc.h
 * @brief    Miscellaneous functions prototypes for the ACH118X library.
 * @version  V1.0.0
 * @date     19 October 2020
 ***************************************************************************************************/
#ifndef __ACH118X_MISC_H
#define __ACH118X_MISC_H

#ifdef __cplusplus
 extern "C" {
#endif

#include "ach118x.h"

/**
  * @brief  NVIC Init Structure definition
  */

typedef struct
{
  uint8_t NVIC_IRQChannel;                    /*!< Specifies the IRQ channel to be enabled or disabled.
                                                   This parameter can be an enumerator of @ref IRQn_Type
                                                   enumeration (For the complete STM32 Devices IRQ Channels
                                                   list, please refer to stm32f4xx.h file) */

  uint8_t NVIC_IRQChannelPreemptionPriority;  /*!< Specifies the pre-emption priority for the IRQ channel
                                                   specified in NVIC_IRQChannel. This parameter can be a value
                                                   between 0 and 15 as described in the table @ref MISC_NVIC_Priority_Table
                                                   A lower priority value indicates a higher priority */

  uint8_t NVIC_IRQChannelSubPriority;         /*!< Specifies the subpriority level for the IRQ channel specified
                                                   in NVIC_IRQChannel. This parameter can be a value
                                                   between 0 and 15 as described in the table @ref MISC_NVIC_Priority_Table
                                                   A lower priority value indicates a higher priority */

  FunctionalState NVIC_IRQChannelCmd;         /*!< Specifies whether the IRQ channel defined in NVIC_IRQChannel
                                                   will be enabled or disabled.
                                                   This parameter can be set either to ENABLE or DISABLE */
} NVIC_InitTypedef;



/** @defgroup MISC_Vector_Table_Base
  * @{
  */

#define NVIC_VectTab_RAM             ((uint32_t)0x20000000)
#define NVIC_VectTab_FLASH           ((uint32_t)0x0800C000)
#define IS_NVIC_VECTTAB(VECTTAB) (((VECTTAB) == NVIC_VectTab_RAM) || \
                                  ((VECTTAB) == NVIC_VectTab_FLASH))
/**
  * @}
  */


/** @defgroup MISC_System_Low_Power
  * @{
  */

#define NVIC_LP_SEVONPEND            ((uint8_t)0x10)
#define NVIC_LP_SLEEPDEEP            ((uint8_t)0x04)
#define NVIC_LP_SLEEPONEXIT          ((uint8_t)0x02)
#define IS_NVIC_LP(LP) (((LP) == NVIC_LP_SEVONPEND) || \
                        ((LP) == NVIC_LP_SLEEPDEEP) || \
                        ((LP) == NVIC_LP_SLEEPONEXIT))
/**
  * @}
  */


/** @defgroup MISC_Preemption_Priority_Group
  * @{
  */

#define NVIC_PriorityGroup_0         ((uint32_t)0x700) /*!< 0 bits for pre-emption priority
                                                            4 bits for subpriority */
#define NVIC_PriorityGroup_1         ((uint32_t)0x600) /*!< 1 bits for pre-emption priority
                                                            3 bits for subpriority */
#define NVIC_PriorityGroup_2         ((uint32_t)0x500) /*!< 2 bits for pre-emption priority
                                                            2 bits for subpriority */
#define NVIC_PriorityGroup_3         ((uint32_t)0x400) /*!< 3 bits for pre-emption priority
                                                            1 bits for subpriority */
#define NVIC_PriorityGroup_4         ((uint32_t)0x300) /*!< 4 bits for pre-emption priority
                                                            0 bits for subpriority */

#define IS_NVIC_PRIORITY_GROUP(GROUP) (((GROUP) == NVIC_PriorityGroup_0) || \
                                       ((GROUP) == NVIC_PriorityGroup_1) || \
                                       ((GROUP) == NVIC_PriorityGroup_2) || \
                                       ((GROUP) == NVIC_PriorityGroup_3) || \
                                       ((GROUP) == NVIC_PriorityGroup_4))

#define IS_NVIC_PREEMPTION_PRIORITY(PRIORITY)  ((PRIORITY) < 0x10)

#define IS_NVIC_SUB_PRIORITY(PRIORITY)  ((PRIORITY) < 0x10)

#define IS_NVIC_OFFSET(OFFSET)  ((OFFSET) < 0x000FFFFF)

/**
  * @}
  */


/** @defgroup MISC_SysTick_clock_source
  * @{
  */

#define SysTick_CLKSource_HCLK_Div8    ((uint32_t)0xFFFFFFFB)
#define SysTick_CLKSource_HCLK         ((uint32_t)0x00000004)
#define IS_SYSTICK_CLK_SOURCE(SOURCE) (((SOURCE) == SysTick_CLKSource_HCLK) || \
                                       ((SOURCE) == SysTick_CLKSource_HCLK_Div8))
/**
  * @}
  */


//------------------------------------------------------------------
// API prototypes
//------------------------------------------------------------------

/**
  * @brief  Configures the priority grouping
  * @param  NVIC_PriorityGroup: NVIC_PriorityGroup_<0..4>
  * @retval none
  */
void NVIC_PriorityGroupConfig(uint32_t NVIC_PriorityGroup);

/**
  * @brief  Initializes the NVIC peripheral
  * @param  NVIC_InitStruct: Structure of parameters
  * @retval none
  */
void NVIC_Init(NVIC_InitTypedef* NVIC_InitStruct);

/**
  * @brief  Sets the vector table location and Offset
  * @param  NVIC_VectTab: Address of vector table
  * @param  Offset: Offset
  * @retval none
  */
void NVIC_SetVectorTable(uint32_t NVIC_VectTab, uint32_t Offset);

/**
  * @brief  Selects the condition for the system to enter low power mode
  * @param  LowPowerMode: NVIC_LP_SEVONPEND, NVIC_LP_SLEEPDEEP, NVIC_LP_SLEEPONEXIT
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void NVIC_SystemLPConfig(uint8_t LowPowerMode, FunctionalState NewState);

/**
  * @brief  Configures the SysTick clock source
  * @param  SysTick_CLKSource: SysTick_CLKSource_HCLK, SysTick_CLKSource_HCLK_Div8
  * @retval none
  */
void SysTick_CLKSourceConfig(uint32_t SysTick_CLKSource);

#ifdef __cplusplus
}
#endif

#endif /* __ACH118X_MISC_H */


/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/

