/***************************************************************************************************
 *                              MCU ACH118X driver library
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     ach118x_i2s.h
 * @brief    I2S Definitions & Features.
 * @version  V1.0.0
 * @date     05-Nov-2020
 * @detail
 *   - I2S0 is used for an externally exposed CODEC on the chip package.
 *   - I2S1 is the internal I2S for the integrated CODEC. I2S1 is master mode
 *     only, and the CODEC is the slave.
 *   - I2S Justified format only
 *   - Audio data resolutions of 12, 16, 20, 24, and 32 bit modes
 *   - I2S transmitter and/or receiver based on the Philips I2S serial protocol
 *   - Full duplex communication
 *   - Configurable FIFO depth of 2, 4, 8, and 16 words, where the data width
 *     is determined by I2S_RX_WORDSIZE_x or I2S_TX_WORDSIZE_x.
 *   - I2S maximum FIFO size is 16 samples for each channel.
 *   - The FIFO thresholds are programmible.
 ***************************************************************************************************/
#ifndef __ACH118X_I2S_H
#define __ACH118X_I2S_H

#include "ach118x.h"

typedef union {
    vu32 RBR;                   // Receive buffer register
    vu32 THR;                   // Transmit holding register
} I2S_RBR_THR_Typedef;

typedef struct {
    I2S_RBR_THR_Typedef LFT;    // 0x020    Left recieve/transmit register
    I2S_RBR_THR_Typedef RGT;    // 0x024    Right recieve/transmit register
    vu32 RER;                   // 0x028    Receive Enable Register
    vu32 TER;                   // 0x02C    Transmit Enable Register
    vu32 RCR;                   // 0x030    Receive Configuration Register
    vu32 TCR;                   // 0x034    Transmit Configuration Register
    vu32 ISR;                   // 0x038    Interrupt Status Register
    vu32 IMR;                   // 0x03C    Interrupt Mask Register
    vu32 ROR;                   // 0x040    Receive Overrun Register
    vu32 TOR;                   // 0x044    Transmit Overrun Register
    vu32 RFCR;                  // 0x048    Receive FIFO Configuration Register
    vu32 TFCR;                  // 0x04C    Transmit FIFO Configuration Register
    vu32 RFF;                   // 0x050    Receive FIFO Flush
    vu32 TFF;                   // 0x054    Transmit FIFO Flush
    vu32 Reserved_058;          // 0x058
    vu32 Reserved_05C;          // 0x05C
} I2S_Channel_Typedef;

typedef struct {
    vu32 IER;                   // 0x000    DW_apb_i2s Enable Register
    vu32 IRER;                  // 0x004    I2S Receiver Block Enable Register
    vu32 ITER;                  // 0x008    I2S Transmitter Block Enable Register
    vu32 CER;                   // 0x00C    Clock Configuration Register
    vu32 CCR;                   // 0x010    Receiver Block FIFO Register
    vu32 RXFFR;                 // 0x014    Receiver Block FIFO Register
    vu32 TXFFR;                 // 0x018    Transmitter Block FIFO Register
    vu32 Reserved_01C;          // 0x01C
    I2S_Channel_Typedef Ch0;    // 0x020 - 0x05F
    I2S_Channel_Typedef Ch1;    // 0x060 - 0x09F
    I2S_Channel_Typedef Ch2;    // 0x0A0 - 0x0DF
    I2S_Channel_Typedef Ch3;    // 0x0E0 - 0x11F
    vu32 Reserved_120_1BF[40];  // 0x120 - 0x1BF
    vu32 RXDMA;                 // 0x1C0    Receiver Block DMA Register
    vu32 RRXDMA;                // 0x1C4    Reset Receiver Block DMA Register
    vu32 TXDMA;                 // 0x1C8    Transmitter Block DMA Register
    vu32 RTXDMA;                // 0x1CC    Reset Transmitter Block DMA Register
    vu32 Reserved_1D0_1EF[8];   // 0x1D0 - 0x1EF
    vu32 COMP_PARAM_2;          // 0x1F0    Component Parameter 2 Register
    vu32 COMP_PARAM_1;          // 0x1F4    Component Parameter 1 Register
    vu32 COMP_VERSION;          // 0x1F8    Component Version ID
    vu32 COMP_TYPE;             // 0x1FC    DesignWare Component Type - 0x445701a0
} I2S_Typedef;

#define I2S_IER_IEN         0x01        // IER - DW_apb_i2s enable
#define I2S_IRER_RXEN       0x01        // IRER - Receiver block enable
#define I2S_ITER_TXEN       0x01        // ITER - Transmitter block enable
#define I2S_TER_TXCHEN      0x01        // TERx - Transmit channel enable
#define I2S_CER_CLKEN       0x01        // CER - Clock generation enable

//ISRx
#define I2S_ISR_RXDA        0x01        // Status of Receive Data Available interrupt
#define I2S_ISR_RXFO        0x02        // Status of Data Overrun interrupt for the RX channel
#define I2S_ISR_TFE         0x10        // Status of Transmit Empty Trigger interrupt
#define I2S_ISR_TFO         0x20        // Status of Data Overrun interrupt for the TX channel

//IMRx
#define I2S_IMR_RXDAM       0x01        // Masks RX FIFO Data Available interrupt
#define I2S_IMR_RXFOM       0x02        // Masks RX FIFO Overrun interrupt
#define I2S_IMR_TXFEM       0x10        // Masks TX FIFO Empty interrupt
#define I2S_IMR_TXFOM       0x20        // Masks TX FIFO Overrun interrupt

//CCR
#define I2S_CCR_WSS_16      0x00
#define I2S_CCR_WSS_24      0x08
#define I2S_CCR_WSS_32      0x10

#define I2S_CCR_SCLKG_NONE  0x00
#define I2S_CCR_SCLKG_12    0x01
#define I2S_CCR_SCLKG_16    0x02
#define I2S_CCR_SCLKG_20    0x03
#define I2S_CCR_SCLKG_24    0x04


//------------------------------------------------------------------
// API prototypes
//------------------------------------------------------------------

/**
  * @brief  Get component ID
  * @param  I2Sx: where x can be 0, 1 to select the I2S peripheral.
  * @retval Component ID
  */
uint32_t  I2S_GetComponentID(I2S_Typedef *i2s);

#endif // __ACH118X_I2S_H
/************************ (C) COPYRIGHT ART *****END OF FILE****/

