/***************************************************************************************************
 *                              MCU ACH118X driver library
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     ach118x_i2c.h
 * @brief    I2C Definitions.
 * @version  V1.0.0
 * @date     18-Nov-2020
 * @detail
 *  - Data structures and the address mapping for I2C
 *  - Macros and prototypes to access I2C hardware
 ***************************************************************************************************/
#ifndef __ACH118X_I2C_H
#define __ACH118X_I2C_H

#include "ach118x_type.h"
#include "ach118x_map.h"

/*I2C register define*/
typedef struct
{
    vu32 CON;               //0x0,I2C Control Register
    vu32 TAR;               //0x4,I2C Target Address Register
    vu32 SAR;               //0x8,I2C Slave Address Register
    vu32 HS_MADDR;          //0xc,I2C High Speed Master Mode Code Address Register
    vu32 DATA_CMD;          //0x10,I2C Rx/Tx Data Buffer and Command Register
    vu32 SS_SCL_HCNT;       //0x14,Standard Speed I2C Clock SCL High Count Register
    vu32 SS_SCL_LCNT;       //0x18,Standard Speed I2C Clock SCL Low Count Register
    vu32 FS_SCL_HCNT;       //0x1c, Fast Mode or Fast Mode Plus I2C Clock SCL High Count Register
    vu32 FS_SCL_LCNT;       //0x20,Fast Mode or Fast Mode Plus I2C Clock SCL Low Count Register
    vu32 HS_SCL_HCNT;       //0x24,High Speed I2C Clock SCL High Count Register
    vu32 HS_SCL_LCNT;       //0x28,High Speed I2C Clock SCL Low Count Register
    vu32 INTR_STAT;         //0x2c,I2C Interrupt Status Register
    vu32 INTR_MASK;         //0x30, I2C Interrupt Mask Register
    vu32 RAW_INTR_STAT;     //0x34,I2C Raw Interrupt Status Register
    vu32 RX_TL;             //0x38,I2C Receive FIFO Threshold Register
    vu32 TX_TL;             //0x3c,I2C Transmit FIFO Threshold Registe
    vu32 CLR_INTR;          //0x40,Clear Combined and Individual Interrupt Register
    vu32 CLR_RX_UNDER;      //0x44,Clear RX_UNDER Interrupt Register
    vu32 CLR_RX_OVER;       //0x48,Clear RX_OVER Interrupt Register
    vu32 CLR_TX_OVER;       //0x4c,Clear TX_OVER Interrupt Register
    vu32 CLR_RD_REQ;        //0x50,Clear RD_REQ Interrupt Register
    vu32 CLR_TX_ABRT;       //0x54,Clear TX_ABRT Interrupt Register
    vu32 CLR_RX_DONE;       //0x58,Clear RX_DONE Interrupt Register
    vu32 CLR_ACTIVITY;      //0x5c,Clear ACTIVITY Interrupt Register
    vu32 CLR_STOP_DET;      //0x60,Clear STOP_DET Interrupt Register
    vu32 CLR_START_DET;     //0x64,Clear START_DET Interrupt Register
    vu32 CLR_GEN_CALL;      //0x68,Clear GEN_CALL Interrupt Register
    vu32 ENABLE;            //0x6c, I2C Enable Register
    vu32 STATUS;            //0x70, I2C Status Register
    vu32 TXFLR;             //0x74, I2C Transmit FIFO Level Register
    vu32 RXFLR;             //0x78,I2C Receive FIFO Level Register
    vu32 SDA_HOLD;          //0x7c,I2C SDA Hold Time Length Register
    vu32 TX_ABRT_SOURCE;    //0x80, I2C Transmit Abort Source Register
    vu32 SLV_DATA_NACK_ONLY;//0x84,Generate Slave Data NACK Register
    vu32 DMA_CR;            //0x88,DMA Control Register
    vu32 DMA_TDLR;          //0x8c,DMA Transmit Data Level Register
    vu32 DMA_RDLR;          //0x90,I2C Receive Data Level Register
    vu32 SDA_SETUP;         //0x94,I2C SDA Setup Register
    vu32 ACK_GENERAL_CALL;  //0x98,I2C ACK General Call Register
    vu32 ENABLE_STATUS;     //0x9c,I2C Enable Status Register
    vu32 FS_SPKLEN;         //0xA0,I2C SS and FS Spike Suppression Limit Register
    vu32 HS_SPKLEN;         //0xA4,I2C HS Spike Suppression Limit Register
    vu32 CLR_RESTART_DET;   //0xA8,Clear RESTART_DET Interrupt Register
    vu32 ReseverAC;         //

    vu32 ReseverB0;//
    vu32 ReseverB4;//
    vu32 ReseverB8;//
    vu32 ReseverBC;//

    vu32 ReseverC0;//
    vu32 ReseverC4;//
    vu32 ReseverC8;//
    vu32 ReseverCC;//

    vu32 ReseverD0;//
    vu32 ReseverD4;//
    vu32 ReseverD8;//
    vu32 ReseverDC;//

    vu32 ReseverE0;//
    vu32 ReseverE4;//
    vu32 ReseverE8;//
    vu32 ReseverEC;//

    vu32 ReseverF0;//
    vu32 COMP_PARAM_1;      //0xf4,Clear RESTART_DET Interrupt Register
    vu32 COMP_VERSION;      //0xf8,I2C Component Version Register
    vu32 COMP_TYPE;         //0xfc, I2C Component Type Register
}I2C_Typedef;

typedef enum {I2C_MASTER = 0, I2C_SLAVE = !I2C_MASTER} RoleType;
#define I2C_MASTER_MASK(X)      ((X) |= (0x41))
#define I2C_SLAVE_MASK(X)       ((X) &= (0xBE))

typedef enum {I2C_ADDR_7BIT = 0, I2C_ADDR_10BIT = !I2C_ADDR_7BIT} AddrType;
#define I2C_ADDR_7BIT_MASK(X)   ((X) &= (0xE7))
#define I2C_ADDR_10BIT_MASK(X)  ((X) |= (0x18))

typedef enum {
    I2C_DutyCycle_1_4 = 0,
    I2C_DutyCycle_1_2 = 1,
    I2C_DutyCycle_3_4 = 3
} DutyCycleType;

#define I2C_STANDARD_MODE_MASK(X)   (X) &= (0xFB);(X) |= (0x02)
#define I2C_FAST_MODE_MASK(X)       (X) |= (0x04);(X) &= (0xFD)
#define I2C_HIGH_MODE_MASK(X)       ((X) |= (0x06))
#define I2C_RESTART_EN_MASK(X)      ((X) |= (0x20))

typedef struct
{
    RoleType I2C_Role;
    AddrType I2C_AddrType;
    uint32_t I2C_ClockSpeed;
    DutyCycleType I2C_DutyCycle;
    uint16_t I2C_AckAddress;
    uint16_t I2C_OwnAddress;
    FunctionalState I2C_Ack;
}I2C_InitTypeDef;

//IC_CLR_INTR
#define I2C_INTR                            (0x0000)
//IC_RAW_INTR_STAT
#define I2C_RX_UNDER                        (0x0001)
#define I2C_RX_OVER                         (0x0002)
#define I2C_RX_FULL                         (0x0004)
#define I2C_TX_OVER                         (0x0008)
#define I2C_TX_EMPTY                        (0x0010)
#define I2C_RD_REQ                          (0x0020)
#define I2C_TX_ABRT                         (0x0040)
#define I2C_RX_DONE                         (0x0080)
#define I2C_ACTIVITY                        (0x0100)
#define I2C_STOP_DET                        (0x0200)
#define I2C_START_DET                       (0x0400)
#define I2C_GEN_CALL                        (0x0800)
#define I2C_RESTART_DET                     (0x1000)
#define I2C_MST_ON_HOLD                     (0x2000)
//CLR_RESTART_DET
//#define RESTART_DET                     (0xFFFF)

//IC_STATUS  ( I2C Status Register)
#define I2C_SLV_HOLD_RX_FIFO_FULL           (0x400)
#define I2C_SLV_HOLD_TX_FIFO_EMPT           (0x200)
#define I2C_MST_HOLD_RX_FIFO_FULL           (0x100)
#define I2C_MST_HOLD_TX_FIFO_EMPTY          (0x080)
#define I2C_SLV_ACTIVITY                    (0x040)
#define I2C_MST_ACTIVITY                    (0x020)
#define I2C_REC_FIFO_FULL                   (0x010)
#define I2C_REC_FIFP_NOT_EMPTY              (0x008)
#define I2C_TRA_FIFO_EMPTY                  (0x004)
#define I2C_TRA_FIFO_NOT_FULL               (0x002)
#define I2C_ACTIVITY_STATUS                 (0x001)

//IC_COMP_PARAM_1
    //BIT[1:0]
#define I2C_APB_DATA_WIDTH_8BIT             (0x00)
#define I2C_APB_DATA_WIDTH_16BIT            (0x01)
#define I2C_APB_DATA_WIDTH_32BIT            (0x02)
    //BIT[3:2]
#define I2C_MAX_SPEED_MODE_STANDARD         (0x01<<2)
#define I2C_MAX_SPEED_MODE_FAST             (0x02<<2)
#define I2C_MAX_SPEED_MODE_HIGH             (0x03<<2)

#define I2C_HC_COUNT_VALUES                 (0x1<<4)
#define I2C_INTR_IO                         (0x1<<5)
#define I2C_HAS_DMA                         (0x1<<6)
#define I2C_ADD_ENCODED_PARAM               (0x1<<7)

//------------------------------------------------------------------
// API prototypes
//------------------------------------------------------------------

/**
  * @brief  Initialize I2C with specific parameter
  * @param  I2Cx: I2C port
  * @param  I2C_InitStruct: Parameter structure
  * @retval none
  */
void  I2C_Init(I2C_Typedef* I2Cx, I2C_InitTypeDef* I2C_InitStruct);

/**
  * @brief  Send one byte via i2c interface
  * @param  I2Cx: I2C port
  * @param  Data: Data
  * @retval none
  */
void  I2C_SendData(I2C_Typedef* I2Cx, uint8_t Data);

/**
  * @brief  Read one byte from i2c interface
  * @param  I2Cx: I2C port
  * @retval Byte read from I2C register
  */
uint8_t  I2C_ReceiveData(I2C_Typedef* I2Cx);

/**
  * @brief  Checks whether the specified I2C interrupt has occurred or not.
  * @param  I2Cx: I2C port
  * @param  I2C_IT: Interrupt source
  * @retval SET or RESET
  */
ITStatus  I2C_GetITStatus(I2C_Typedef* I2Cx, uint16_t I2C_IT);

/**
  * @brief  Configure the i2c interrupt
  * @param  I2Cx: I2C port
  * @param  I2C_IT: ENABLE or DISABLE
  * @retval none
  */
void  I2C_ITConfig(I2C_Typedef* I2Cx, uint16_t I2C_IT, FunctionalState NewState);

/**
  * @brief  Controls the level of entries (or below) that trigger the TX_EMPTY interrupt
  * @param  I2Cx: I2C port
  * @param  I2C_Level: Transmit FIFO Threshold Level
  * @retval none
  */
void  I2C_TranFIFOLevelConfig(I2C_Typedef* I2Cx, uint8_t I2C_Level);

/**
  * @brief  Controls the level of entries (or above) that triggers the RX_FULL interrupt
  * @param  I2Cx: I2C port
  * @param  I2C_Level: Receive FIFO Threshold Level
  * @retval none
  */
void  I2C_ReceFIFOLevelConfig(I2C_Typedef* I2Cx, uint8_t I2C_Level);

/**
  * @brief  Clear the i2c interrupt flag
  * @param  I2Cx: I2C port
  * @param  I2C_IT: Interrupt source
  * @retval Value read from interrupt source register
  */
uint8_t  I2C_ClearITStatus(I2C_Typedef* I2Cx, uint16_t I2C_IT);

/**
  * @brief  Enable or Disable I2C interface
  * @param  I2Cx: I2C port
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void        I2C_Cmd(I2C_Typedef* I2Cx, FunctionalState NewState);

/**
  * @brief  Used to indicate the current transfer status and FIFO status
  * @param  I2Cx: I2C port
  * @param  I2C_FLAG: Transfer or FIFO status
  * @retval SET or RESET
  */
FlagStatus  I2C_GetFlagStatus(I2C_Typedef* I2Cx, uint16_t I2C_FLAG);

/**
  * @brief  Get the number of valid data entries in the transmit FIFO buffer
  * @param  I2Cx: I2C port
  * @retval The number of valid data
  */
uint32_t  I2C_GetTranFIFOLevel(I2C_Typedef* I2Cx);

/**
  * @brief  Get the number of valid data entries in the receive FIFO buffer
  * @param  I2Cx: I2C port
  * @retval The number of valid data
  */
uint32_t  I2C_GetReceFIFOLevel(I2C_Typedef* I2Cx);

/**
  * @brief  Control the hold time of SDA during transmit in both slave and
  *         master mode.extend the SDA transition (if any) whenever SCL is HIGH in
  *         the receiver in either master or slave mode.
  * @param  I2Cx: I2C port
  * @param  RxHoldTime: Hold time as a reciever
  * @param  TxHoldTime: Hold time as a transmitter
  * @retval none
  */
void  I2C_SetSDAHoldTime(I2C_Typedef* I2Cx, uint16_t RxHoldTime, uint16_t TxHoldTime);

/**
  * @brief  Get the source of the TX_ABRT bit.
  * @param  I2Cx: I2C port
  * @param  AbrtSource: Transmit abort source
  * @retval SET or RESET
  */
FlagStatus  I2C_GetTXAbrtSource(I2C_Typedef* I2Cx, uint16_t AbrtSource);

/**
  * @brief  Enable or Disable generate a NACK for the data part of a transfer
  *         when it is acting as a slave-receiver
  * @param  I2Cx: I2C port
  * @param  NewState:  ENABLE or DISABLE
  * @retval none
  */
void  I2C_NAcknowledge(I2C_Typedef* I2Cx, FunctionalState NewState);

/**
  * @brief  Enable or Disable I2C interface Acknowledge function
  * @param  I2Cx: I2C port
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void  I2C_Acknowledge(I2C_Typedef* I2Cx, FunctionalState NewState);

/**
  * @brief  Set the duration that measured in ic_clk cycles, of the longest spike
  *         that is filtered out by the spike suppression logic when the component is
  *         operating in standard mode, fast mode, or fast mode plus
  * @param  I2Cx: I2C port
  * @param  SpkLen: Spike length
  * @retval none
  */
void  I2C_FS_SPKLENConfig(I2C_Typedef* I2Cx, uint8_t SpkLen);

/**
  * @brief  Set the duration that measured in ic_clk cycles, of the longest spike
  *         that is filtered out by the spike suppression logic when the component is
  *         operating in high speed mode.
  * @param  I2Cx: I2C port
  * @param  SpkLen: Spike length
  * @retval none
  */
void  I2C_HS_SPKLENConfig(I2C_Typedef* I2Cx, uint8_t SpkLen);

/**
  * @brief  Get rx buffer depth
  * @param  I2Cx: I2C port
  * @retval Depth
  */
uint8_t  I2C_GetRxBufferDepth(I2C_Typedef* I2Cx);

/**
  * @brief  Get tx buffer depth
  * @param  I2Cx: I2C port
  * @retval Depth
  */
uint8_t I2C_GetTxBufferDepth(I2C_Typedef* I2Cx);

/**
  * @brief  Get component parameter
  * @param  I2Cx: I2C port
  * @param  I2C_Param: parameter
  * @retval SET or RESET
  */
FlagStatus  I2C_GetComponentParameter(I2C_Typedef* I2Cx, uint16_t I2C_Param);

/**
  * @brief  Get the I2C Component Version
  * @param  I2Cx: I2C port
  * @retval I2C Component Version
  */
uint32_t    I2C_GetComponentVersion(I2C_Typedef* I2Cx);

/**
  * @brief  Designware Component Type number = 0x44_57_01_40.
  * @param  I2Cx: I2C port
  * @retval I2C Component Type
  */
uint32_t    I2C_GetComponentType(I2C_Typedef* I2Cx);

#endif // __ACH118X_I2C_H
/************************ (C) COPYRIGHT ART *****END OF FILE****/

