/***************************************************************************************************
 *                              MCU ACH118X driver library
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     ach118x_file.h
 * @brief    Data structures and prototypes for fopen, fclose, and other file functions.
 * @version  V1.0.0
 * @date     05-Nov-2020
 ***************************************************************************************************/
#ifndef __ACH118X_FILE_H
#define __ACH118X_FILE_H

#include <stdio.h>

/*************************************************************************/
/*                   File Definitions                                    */
/*************************************************************************/
#define FILENAME_MAXSIZE 54

// Directory file entry (64 bytes)
typedef struct {
    uint8_t     EntryType;                  // Type of entry
    uint8_t     FileAttribute;              // Attribute of file
    uint16_t    FileStartSector;            // Sector start of file
    uint32_t    FileSize;                   // Size of file in bytes
    char        FileName[FILENAME_MAXSIZE]; // Name of file
    uint16_t    CRC16;
} FileEntry_t;

// File handle info
typedef struct {
    uint16_t FileEntry;
    char     FileName[FILENAME_MAXSIZE];
    int16_t  Mode;
    uint32_t StartAddr;
    uint32_t Size;
    uint32_t Offset;
} FILE_HANDLE_t;


typedef struct {
    FileEntry_t FileEntry;
    uint32_t  EntryAddr;
} DEFAULT_APP_t;


// Directory entry types
#define ENTRY_DELETED           0x00
#define ENTRY_VALID             0x0F
#define ENTRY_UNUSED            0xFF

// Directory file types
#define FILE_FLASHLOADER        0
#define FILE_APPLICATION        1
#define FILE_GENERIC            2

// Directory entry attributes
#define ATTR_MASK_ALL           0x03
#define ATTR_EXE                0x01
#define ATTR_DEFAULT            0x02
#define ATTR_DEFAULT_EXE        0x03
#define ATTR_OFFSET             1
#define ATTR_GENERIC_FILE       0xFF
#define ATTR_EXE_FILE           0xFE
#define ATTR_DEFAULT_EXE_FILE   0xFC


#define SELECT_EXE              0
#define SELECT_EXE_DEFAULT      1
#define SELECT_ALL              3

#define CREATE_ENTRY_OK         0
#define CREATE_ENTRY_FAIL       -1

#ifndef M25D80_SECTOR_SIZE
#define M25D80_SECTOR_SIZE      4096
#endif

#define SECTOR_SIZE             (M25D80_SECTOR_SIZE)
#define SECTOR0_ADDR            (0)
#define SECTOR1_ADDR            (SECTOR_SIZE)
#define PERS_VARS_SECTOR        (3)             // Persistent Variable sector
#define PERS_VARS_SIZE          (4*SECTOR_SIZE)
#define DIR_SECTOR              (2)             // Directory sector
#define DIR_ADDR                (DIR_SECTOR*SECTOR_SIZE)
#define DIR_SIZE                (SECTOR_SIZE)
#define ENTRYMAX                (DIR_SIZE / sizeof(FileEntry_t))
#define APP_EXECUTE_ADDR        (0x20002000)
#define OTP_ADDR                (void *)(0x20001F80)
#define OTP_SIZE                (128)
#define MAC_AREA_ADDR           (0x20001FC0)
#define pMAC_Area               (uint8_t *)MAC_AREA_ADDR
#define MAC_AREA_SIZE           (64)
#define MAC_FLASH_ADDR          (0x1FC0)
#define GetSectorAddr(n)        (n*SECTOR_SIZE)


// File handle globals
#define FILE_HANDLE_OFFSET  3
#define FILE_HANDLE_MAX     16
extern FILE_HANDLE_t *__FILE_HANDLE[];

/***************************************************************/
/*  Standard supported prototypes                              */
/***************************************************************/
/* int fgets(unsigned char *sbuf, int size, FILE *stream);     */
/* int fwrite(void *ptr, int n, int plen, FILE *stream);       */
size_t fwrite(const void *ptr, size_t size, size_t nmemb, FILE *stream);
/* FILE *fopen(char *filename, char *opt);                     */
FILE *fopen(const char *filename, const char *opt);
/* int fseek(FILE *stream, long offset, int whence);           */
/* long ftell(FILE *stream);                                   */
/* int fclose(FILE *stream);                                   */
/* int fread(void *ptr, int size, int nmemb, FILE * stream);   */
/* int ferror(FILE *stream);                                   */
/* void fprintf(FILE *f, char *fmt, ...);                      */
/* void fflush(FILE *f);                                       */
/* void fputc(int c, FILE *f);                                 */
/***************************************************************/

//------------------------------------------------------------------
// API prototypes
//------------------------------------------------------------------

/**
  * @brief  Get file info
  * @param  fp: File handle after open
  * @param  pFileInfo: File handle info
  * @retval File size or -1 if error
  */
//int GetFileHandleInfo(FILE *fp, FILE_HANDLE_t *pFileInfo);
void InitDirectory(void);
void RewriteDirectory(void);
void ReadDirectory(void);

int ReadDirectory_n(void);
void InitDir(void);
int NextFile(int n, char *ext);

#endif /* __ACH118X_FILE_H */
/************************ (C) COPYRIGHT ART *****END OF FILE****/

