/***************************************************************************************************
 *                              MCU ACH118X driver library
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     ach118x_exti.c
 * @brief    Function prototypes for the EXTI firmware library.
 * @version  V1.0.0
 * @date     05-Nov-2020
 ***************************************************************************************************/
#ifndef _ACH118X_EXTI_H
#define _ACH118X_EXTI_H

#ifdef __cplusplus
    extern "C" {
#endif

#include "ach118x_type.h"
#include "ach118x_map.h"

typedef enum{
    PIN_GPIO_0 = 0x01,
    PIN_GPIO_1 = 0x02,
    PIN_GPIO_2 = 0x04,
    PIN_GPIO_3 = 0x08,
    PIN_GPIO_All=0xf
}ExtiPin;

typedef enum{
    TYPE_Level = 0x0,
    TYPE_Edge,
}ExtiType;

typedef enum{
    POLARITY_Low = 0x0,
    POLARITY_High,
}ExtiPolarity;

typedef enum{
    DIS_Debounce = 0x0,
    EN_Debounce,
}ExtiDebounce;

typedef enum{
    DIS_2edge = 0x0,
    EN_2edge,
}Exti2Edge;

//Typedef initialisation strcutrue.
typedef struct
{
    FunctionalState Cmd;
    ExtiPin         Pin;
    ExtiType        Type;
    ExtiPolarity    Polarity;
    ExtiDebounce    EnDebounce;
    Exti2Edge       En2Edge;
} EXTI_InitTypeDef;

//GPIO external interrupt registers definition */
typedef struct
{
    vu32    ENABLE;             //0x30, Interrupt enable register.
    vu32    MASK;               //0x34, Interrupt mask register.
    vu32    TYPE;               //0x38, Interrupt level register.
    vu32    POLARITY;           //0x3C, Interrupt polarity register.
    vu32    STATUS;             //0x40, Interrupt status of PORTA.
    vu32    RSTATUS;            //0x44, Raw interrupt status of PORTA(premasking)
    vu32    DEBOUNCE;           //0x48, Debounce enable register.
    vu32    EOI;                //0x4C, PORTA clear interrupt register.
    vu32    Reserved0;
    vu32    Reserved1;
    vu32    Reserved2;
    vu32    Reserved3;
    vu32    LSS;                //0x60, Level-sensitive synchronization enable register.
    vu32    Reserved4;
    vu32    EDGE;               //0x68, Interrupt both edge type.
} GPIO_IntTypeDef;


//------------------------------------------------------------------
// API prototypes
//------------------------------------------------------------------

/**
  * @brief  Reset EXTI with reset value
  * @retval none
  */
void  EXTI_DeInit(void);

/**
  * @brief  Initialise EXTI with EXTI_InitStruct value
  * @param  EXTI_InitStruct: Parameter structure for initialization
  * @retval none
  */
void  EXTI_Init(EXTI_InitTypeDef* EXTI_InitStruct);

/**
  * @brief  Enable or disable external interrupt
  * @param  ExtiPin: PIN_GPIO_n where n = 0-3, ALL
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void  EXTI_Cmd(ExtiPin pin, FunctionalState NewState);

/**
  * @brief  Get external interrupt status
  * @param  ExtiPin: PIN_GPIO_n where n = 0-3, ALL
  * @retval none
  */
FlagStatus  EXTI_GetITFlag(ExtiPin pin);

/**
  * @brief  Clear external interrupt status
  * @param  ExtiPin: PIN_GPIO_n where n = 0-3, ALL
  * @retval none
  */
void  EXTI_ClearITFlag(ExtiPin pin);


#ifdef __cplusplus
    }//extern "C"
#endif

#endif

/************************ (C) COPYRIGHT ART *****END OF FILE****/

