/***************************************************************************************************
 *                              MCU ACH118X driver library
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     ach118x_dma.h
 * @brief    Function prototypes for the DMA firmware library.
 * @version  V1.0.0
 * @date     05-Nov-2020
 ***************************************************************************************************/
#ifndef __ACH118X_DMA_H
#define __ACH118X_DMA_H

#ifdef __cplusplus
    extern "C" {
#endif

#include "ach118x_type.h"
#include "ach118x_map.h"

//------------------------------------------------------------------------------
// DW_ahb_dmac Parameters
//------------------------------------------------------------------------------
#define DMAH_NUM_HS_INT         0   // Num of handshake interfaces
#define DMAH_WLAST_EN           0   // En handshake for completion
#define DMAH_INTR_IO            2   // should be 3 or F
#define DMAH_CHx_STW            32  // Should be 0
#define DMAH_CHx_DTW            32  // Should be 0
#define DMAH_CHx_FC             0   // Hardcodes the DMA as the block flow controlling device
#define DMAH_CHx_MAX_BLK_SIZE   31
#define DMAH_CHx_FIFO_DEPTH     16

typedef struct {
    uint64_t Undefined_0     : 5;
    uint64_t CH_PRIOR        : 3; // Channel priority
    uint64_t CH_SUSP         : 1; // Channel Suspend
    uint64_t FIFO_EMPTY      : 1; //
    uint64_t HS_SEL_DST      : 1; // Destination Software or Hardware Handshaking Select
    uint64_t HS_SEL_SRC      : 1; // Source Software or Hardware Handshaking Select
    uint64_t LOCK_CH_L       : 2; // Channel Lock Level (unused: DMAH_CHx_LOCK_EN=0)
    uint64_t LOCK_B_L        : 2; // Bus Lock Level (unused: DMAH_CHx_LOCK_EN=0)
    uint64_t LOCK_CH         : 1; // Channel Lock Bit
    uint64_t LOCK_B          : 1; // Bus Lock Bit
    uint64_t DST_HS_POL      : 1; // Destination Handshaking Interface Polarity
    uint64_t SRC_HS_POL      : 1; // Source Handshaking Interface Polarity
    uint64_t MAX_ABRST       : 10;// Maximum AMBA Burst Length (unused: DMAH_MABRST=0)
    uint64_t RELOAD_SRC      : 1; // Automatic Source Reload (unused: DMAH_CHx_MULTI_BLK_EN=0)
    uint64_t RELOAD_DST      : 1; // Automatic Destination Reload (unused: DMAH_CHx_MULTI_BLK_EN=0)
    uint64_t FCMODE          : 1; // Flow Control Mode
    uint64_t FIFO_MODE       : 1; // FIFO Mode Select
    uint64_t PROTCTL         : 3; // Protection Control
    uint64_t DS_UPD_EN       : 1; // Destination Status Update Enable (unused: DMAH_NUM_HS_INT=0)
    uint64_t SS_UPD_EN       : 1; // Source Status Update Enable (unused: DMAH_NUM_HS_INT=0)
    uint64_t SRC_PER         : 4; // H/W handshaking interface (unused: DMAH_NUM_HS_INT=0)
    uint64_t DEST_PER        : 4; // H/W handshaking interface (unused: DMAH_NUM_HS_INT=0)
    uint64_t Undefined_47    : 17;
} DMA_CFG_Typedef;


typedef struct {
    uint64_t INT_EN          : 1; // Interrupt Enable Bit
    uint64_t DST_TR_WIDTH    : 3; // Destination Transfer Width (unused: DMAH_CHx_DTW=32)
    uint64_t SRC_TR_WIDTH    : 3; // Source Transfer Width (unused: DMAH_CHx_STW=32)
    uint64_t DINC            : 2; // Destination Address Increment
    uint64_t SINC            : 2; // Source Address Increment
    uint64_t DEST_MSIZE      : 3; // Destination Burst Transaction Length
    uint64_t SRC_MSIZE       : 3; // Source Burst Transaction Length
    uint64_t SRC_GATHER_EN   : 1; // Source gather enable bit (unused: DMAH_CHx_SRC_GAT_EN=0)
    uint64_t DST_SCATTER_EN  : 1; // Destination scatter enable bit (unused: DMAH_CHx_DST_SCA_EN=0)
    uint64_t Undefined_19    : 1; //
    uint64_t TT_FC           : 3; // Transfer Type and Flow Control
    uint64_t DMS             : 2; // Destination Master Select
    uint64_t SMS             : 2; // Source Master Select
    uint64_t LLP_DST_EN      : 1; // Block chaining is enabled on the destination side
    uint64_t LLP_SRC_EN      : 1; // Block chaining is enabled on the source side
    uint64_t Undefined_29    : 3; //
    uint64_t BLOCK_TS        : 12;// Block Transfer Size (DMAH_CHx_MAX_BLK_SIZE=31)
    uint64_t DONE            : 1; //
    uint64_t Undefined_45    : 19;
} DMA_CTL_Typedef;

typedef struct
{
      uint32_t DTW         : 3;
    uint32_t STW         : 3;
    uint32_t STAT_DST    : 1;
    uint32_t STAT_SRC    : 1;
    uint32_t DST_SCA     : 1;
    uint32_t SRC_GAT     : 1;
    uint32_t CH_LOCK     : 1;
    uint32_t MUlTI_BLK   : 1;
    uint32_t CTL_WB      : 1;
    uint32_t HC_LLP      : 1;
    uint32_t FC          : 2;
    uint32_t MULTI_SIZE  : 3;
    uint32_t DMS         : 3;
    uint32_t LMS         : 3;
    uint32_t SMS         : 3;
    uint32_t FIFO_DEPTH  : 3;
}DMA_CHAN_PARAM_T1;

typedef union
{
      uint32_t data;
    DMA_CHAN_PARAM_T1 param;
}DMA_ParamKid1_Typedef;

typedef struct
{
      uint32_t CH0 : 4;
    uint32_t CH1 : 4;
    uint32_t CH2 : 4;
    uint32_t CH3 : 4;
    uint32_t CH4 : 4;
    uint32_t CH5 : 4;
    uint32_t CH6 : 4;
    uint32_t CH7 : 4;
}DMA_CHAN_PARAM_T2;

typedef union
{
      uint32_t data;
    DMA_CHAN_PARAM_T2 param;
} DMA_ParamKid2_Typedef;

typedef struct
{
      uint32_t BIG_ENDIAN_         : 1;
    uint32_t INTR_IOn            : 2;
    uint32_t MABRST              : 1;
    uint32_t Reserved            : 4;
    uint32_t NUM_CHANNELS        : 3;
    uint32_t NUM_MASTER_INT      : 2;
    uint32_t S_WIDTH             : 2;
    uint32_t M4_WIDTH            : 2;
    uint32_t M3_WIDTH            : 2;
    uint32_t M2_WIDTH            : 2;
    uint32_t M1_WIDTH            : 2;
    uint32_t NUM_HS_INT          : 5;
    uint32_t ADD_ENCODED_PARAMSn : 1;
    uint32_t STATIC_ENDIAN_SELECT : 1;
} DMA_CHAN_PARAM_T3;

typedef union
{
      uint32_t data;
    DMA_CHAN_PARAM_T3 param;
} DMA_ParamKid3_Typedef;

typedef union
{
      uint32_t param6_RegL;
      DMA_ParamKid1_Typedef param6_ch7;
    DMA_ParamKid1_Typedef param5_ch6;
    DMA_ParamKid1_Typedef param5_ch5;
    DMA_ParamKid1_Typedef param4_ch4;
    DMA_ParamKid1_Typedef param4_ch3;
    DMA_ParamKid1_Typedef param3_ch2;
    DMA_ParamKid1_Typedef param3_ch1;
    DMA_ParamKid1_Typedef param2_ch0;
    DMA_ParamKid2_Typedef param2_blk_type;
    DMA_ParamKid2_Typedef param1_blk_size;
    DMA_ParamKid3_Typedef param1_top_level;
    uint32_t dma_type;
    uint32_t dma_version;
} DMA_Param_Typedef;

typedef enum {
    Channel_0 = 0x01,
    Channel_1 = 0x02,
    Channel_2 = 0x04,
    Channel_3 = 0x08,
    Channel_4 = 0x10,
    Channel_5 = 0x20,
    Channel_6 = 0x40,
    Channel_7 = 0x80,
} DMA_Channel;

typedef enum {
    IT_Tfr = 0x00,
    IT_Blk = 0x01,
    IT_Src = 0x02,
    IT_Dst = 0x03,
    IT_Err = 0x04
} CH_IT_Type;

typedef enum {
    FUNC_Disable = 0x00,
    FUNC_Enable  = 0x01,
} FUNC_EN;

typedef enum {
    TR_8Bits   = 0x00,
    TR_16Bits  = 0x01,
    TR_32Bits  = 0x02,
    TR_64Bits  = 0x03,
    TR_128Bits = 0x04,
    TR_256Bits = 0x05
} TR_WIDTH;

typedef enum {
    Increment    = 0x00,
    DecrementDis = 0x01,
    NoChange     = 0x02,
} ADDR_INCREMENT;

typedef enum {
    MSIZE_1     = 0x00,
    MSIZE_4     = 0x01,
    MSIZE_8     = 0x02,
    MSIZE_16    = 0x03,
    MSIZE_32    = 0x04,
    MSIZE_64    = 0x05,
    MSIZE_128   = 0x06,
    MSIZE_256   = 0x07
} TR_MSIZE;

typedef enum {              // FLOW CONTROLLER
    M2M_DMA     = 0x00,     // DW_ahb_dmac
    M2P_DMA     = 0x01,     // DW_ahb_dmac
    P2M_DMA     = 0x02,     // DW_ahb_dmac
    P2P_DMA     = 0x03,     // DW_ahb_dmac
    P2M_P       = 0x04,     // Peripheral
    P2P_SrcP    = 0x05,     // Source Peripheral
    M2P_P       = 0x06,     // Peripheral
    P2P_DstP    = 0x07,     // Destination Peripheral
} TT_FC;

typedef enum {
    Master_1    = 0,
    Master_2    = 1,
    Master_3    = 2,
    Master_4    = 3
} MASTER;

typedef enum {
    BLOCK_TS_8Bits   = 0x00,
    BLOCK_TS_16Bits  = 0x01,
    BLOCK_TS_32Bits  = 0x02,
    BLOCK_TS_64Bits  = 0x03,
    BLOCK_TS_128Bits = 0x04,
    BLOCK_TS_256Bits = 0x05
} xxxBLOCK_TS;

typedef enum {
    HS_HW = 0x00,
    HS_SW = 0x01,
} HS_SEL;

typedef enum {
    ACTIVE_High  = 0x00,
    ACTIVE_Low   = 0x01,
} HWHS_POL;

typedef enum {
    Compl_DMA_Tfr = 0x00,
    DMA_Blk_Tfr   = 0x01,
    DMA_Blk_Tsc   = 0x02,
} LOCK_LEVEL;

typedef enum {
    FC_MODE_0    = 0x00,
    FC_MODE_1    = 0x01,
} FC_MODE;

typedef enum {
    FIFO_MODE_0  = 0x00,
    FIFO_MODE_1  = 0x01,
} FIFO_MODE;


typedef struct
{
    // LLP
    volatile uint32_t       LLPLoc;

    // Ctrl
    volatile uint32_t       SrcAddress;
    volatile uint32_t       DstAddress;
    volatile FUNC_EN        IntEnable;
    volatile TR_WIDTH       DstTrWidth;
    volatile TR_WIDTH       SrcTrWidth;
    volatile ADDR_INCREMENT DstAddrIncOpt;
    volatile ADDR_INCREMENT SrcAddrIncOpt;
    volatile TR_MSIZE       DstTrSize;
    volatile TR_MSIZE       SrcTrSize;
    volatile FUNC_EN        SrcGatherEn;
    volatile FUNC_EN        DstScatterEn;
    volatile TT_FC          FlowCtrlType;
    volatile MASTER         DstMasterOpt;
    volatile MASTER         SrcMasterOpt;
    volatile FUNC_EN        DstLLPEn;
    volatile FUNC_EN        SrcLLPEn;
    volatile uint32_t       BlockTsize;

    // Cfg
    volatile uint8_t        ChannelPrior;
    volatile HS_SEL         DstHsSel;
    volatile HS_SEL         SrcHsSel;
    volatile LOCK_LEVEL     ChannelLockLevel;
    volatile LOCK_LEVEL     BusLockLevel;
    volatile FUNC_EN        ChannelLockEn;
    volatile FUNC_EN        BusLockEn;
    volatile HWHS_POL       SrcHsPol;
    volatile HWHS_POL       DstHsPol;
    volatile uint16_t       BurstLenMax;
    volatile FUNC_EN        ReloadSrcEn;
    volatile FUNC_EN        ReloadDstEn;
    volatile FC_MODE        FcMode;
    volatile uint8_t        FifoMode;
    volatile FUNC_EN        SrcStatusEn;
    volatile FUNC_EN        DstStatusEn;
    volatile uint8_t        SrcPER;
    volatile uint8_t        DstPER;
}DMA_ChannelInit_Typedef;


//-------------------------
//  Channel structure
//-------------------------
typedef struct
{
    volatile uint64_t           SAR;
    volatile uint64_t           DAR;
    volatile uint64_t           LLP;
    volatile DMA_CTL_Typedef    CTL;
    volatile uint64_t           SSTAT;
    volatile uint64_t           DSTAT;
    volatile uint64_t           SSTATAR;
    volatile uint64_t           DSTATAR;
    volatile DMA_CFG_Typedef    CFG;
    volatile uint64_t           SGR;
    volatile uint64_t           DSR;
} DMA_Channel_Typedef;


typedef struct
{
    volatile uint64_t   RawTfr_Reg;
    volatile uint64_t   RawBlock_Reg;
    volatile uint64_t   RawSrcTran_Reg;
    volatile uint64_t   RawDstTran_Reg;
    volatile uint64_t   RawErrTran_Reg;
    volatile uint64_t   StatusTfr_Reg;
    volatile uint64_t   StatusBlock_Reg;
    volatile uint64_t   StatusSrcTran_Reg;
    volatile uint64_t   StatusDstTran_Reg;
    volatile uint64_t   StatusErrTran_Reg;
    volatile uint64_t   MaskTfr_Reg;
    volatile uint64_t   MaskBlock_Reg;
    volatile uint64_t   MaskSrcTran_Reg;
    volatile uint64_t   MaskDstTran_Reg;
    volatile uint64_t   MaskErrTran_Reg;
    volatile uint64_t   ClearTfr_Reg;
    volatile uint64_t   ClearBlock_Reg;
    volatile uint64_t   ClearSrcTran_Reg;
    volatile uint64_t   ClearDstTran_Reg;
    volatile uint64_t   ClearErrTran_Reg;
    volatile uint64_t   StatusInt_Reg;
    volatile uint64_t   ReqSrc_Reg;
    volatile uint64_t   ReqDst_Reg;
    volatile uint64_t   SglReqSrc_Reg;
    volatile uint64_t   SglReqDst_Reg;
    volatile uint64_t   LstReqSrc_Reg;
    volatile uint64_t   LstReqDst_Reg;
    volatile uint64_t   DmaCfg_Reg;
    volatile uint64_t   ChEn_Reg;
    volatile uint64_t   DmaId_Reg;
    volatile uint64_t   DmaTest_Reg;
    volatile uint64_t   Reserved_3B8;
    volatile uint64_t   Reserved_3C0;
    volatile uint64_t   DMA_COMP_PARAMS_6;
    volatile uint64_t   DMA_COMP_PARAMS_5;
    volatile uint64_t   DMA_COMP_PARAMS_4;
    volatile uint64_t   DMA_COMP_PARAMS_3;
    volatile uint64_t   DMA_COMP_PARAMS_2;
    volatile uint64_t   DMA_COMP_PARAMS_1;
    volatile uint64_t   DMA_ComponentID;
} DMA_Typedef;


//------------------------------------------------------------------
// CFG Bits
//------------------------------------------------------------------
// DMA interrupt enable bit.
#define INT_EN_BIT_MASK           (0x02)
// DMA enable bit.
#define DMAC_EN_BIT_MASK           (0x01)

//------------------------------------------------------------------
// API prototypes
//------------------------------------------------------------------

/**
  * @brief  Initialize DMA structure with default values
  * @param  DMA_ChannelInitStruct:
  * @retval none
  */
void  DMA_ChannelStructDeinit(DMA_ChannelInit_Typedef *DMA_ChannelInitStruct );

/**
  * @brief  Initialize DMA channel with initialization structure
  * @param  Channel: DMA_CHANNELx where x can be 0-7
  * @retval none
  */
void  DMA_ChannelInit(DMA_Channel_Typedef *Channel, DMA_ChannelInit_Typedef *DMA_ChannelInitStruct);

/**
  * @brief  Enable or disable DMA
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void  DMA_Cmd(FunctionalState NewState);

/**
  * @brief  Enable or disable DMA channel
  * @param  Channel: 0-7
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void  DMA_ChannelCmd(DMA_Channel Channel, FunctionalState NewState);

/**
  * @brief  Enable or disable DMA channel interrupt
  * @param  Channel: Channel_x where x can be 0-7
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void  DMA_ITCmd(FunctionalState NewState);

/**
  * @brief  Enable or disable the channel DMA interrupt
  * @param  Channel: Channel_x where x can be 0-7
  * @param  NewState: ENABLE or DISABLE
  * @retval none
  */
void  DMA_ChannelITCmd(DMA_Channel Channel, CH_IT_Type Channel_IT, FunctionalState NewState);

/**
  * @brief  Get channel interrupt status
  * @param  Channel: Channel_x where x can be 0-7
  * @param  Channel_IT: IT_Blk, IT_Dst, IT_Err, IT_Src, IT_Tfr
  * @retval IT_Tfr, IT_Blk, IT_Src, IT_Dst, IT_Err
  */
uint64_t  DMA_GetChannelITStatus(DMA_Channel Channel, CH_IT_Type Channel_IT);

/**
  * @brief  Get channel raw status
  * @param  Channel: DMA_CHANNELx where x can be 0-7
  * @param  CH_IT_Type: IT_Tfr, IT_Blk, IT_Src, IT_Dst, IT_Err
  * @retval IT_Tfr, IT_Blk, IT_Src, IT_Dst, IT_Err
  */
uint64_t  DMA_GetChannelITRawStatus(DMA_Channel Channel, CH_IT_Type Channel_IT);

/**
  * @brief  Clear channel interrupt status
  * @param  Channel: DMA_CHANNELx where x can be 0-7
  * @param  CH_IT_Type: IT_Tfr, IT_Blk, IT_Src, IT_Dst, IT_Err
  * @retval none
  */
void  DMA_ClearChannelITStatus(DMA_Channel Channel, CH_IT_Type Channel_IT);

/**
  * @brief  Check if any channel DMA interrupt is set
  * @param  Channel: DMA_CHANNELx where x can be 0-7
  * @param  Channel_IT: IT_Blk, IT_Dst, IT_Err, IT_Src, IT_Tfr
  * @retval IT_Blk, IT_Dst, IT_Err, IT_Src, IT_Tfr
  */
uint64_t DMA_IsAnyChannelITSet(CH_IT_Type Channel_IT);

/**
  * @brief  Enable DMA System Interrupts. Set flags using
  *         SYSCFG_SYS_LEV_INTR_MASK_REG and NVIC_EnableIRQ.
  * @retval none
  */
void  DMA_SystemInterruptEnable(void);

/**
  * @brief  Disable DMA System Interrupts. Clear flags using
  *         SYSCFG_SYS_LEV_INTR_MASK_REG and NVIC_DisableIRQ.
  * @retval none
  */
void  DMA_SystemInterruptDisable(void);


#ifdef __cplusplus
    }//extern "C"
#endif

#endif // __ACH118X_DMA_H
/************************ (C) COPYRIGHT ART *****END OF FILE****/

