/***************************************************************************************************
 *                              MCU ACH118X driver library
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     ach118x_device.h
 * @brief    Device addresses and interrupt numbers.
 * @version  V1.0.0
 * @date     05-Nov-2020
 * @detail
 *           - Data structures and the address mapping for all peripherals
 *           - peripherals registers declarations and bits definition
 *           - Macros to access peripheral's registers hardware
 ***************************************************************************************************/
#ifndef __ACH118X_DEVICE_H
#define __ACH118X_DEVICE_H

/**
  * @brief Configuration of the Cortex-M4 Processor and Core Peripherals
  */
#define __CM4_REV                 0x0001U  /* Core revision r0p1                            */
#define __MPU_PRESENT             1U       /* ACH provides an MPU                           */
#define __NVIC_PRIO_BITS          4U//3U       /* ACH uses 3 Bits for the Priority Levels       */
#define __Vendor_SysTickConfig    0U       /* Set to 1 if different SysTick Config is used  */
#define __FPU_PRESENT             1U       /* FPU present                                   */

/**
 * @brief ACH Interrupt Number Definition, according to the selected device
 *        in @ref Library_configuration_section
 */
typedef enum
{
/******  Cortex-M4 Processor Exceptions Numbers *****************************************/
    NonMaskableInt_IRQn     = -14,      /* Non Maskable Interrupt                       */
    MemoryManagement_IRQn   = -12,      /* Memory Management Interrupt                  */
    BusFault_IRQn           = -11,      /* Bus Fault Interrupt                          */
    UsageFault_IRQn         = -10,      /* Usage Fault Interrupt                        */
    SVCall_IRQn             = -5,       /* SV Call Interrupt                            */
    DebugMonitor_IRQn       = -4,       /* Debug Monitor Interrupt                      */
    PendSV_IRQn             = -2,       /* Pend SV Interrupt                            */
    SysTick_IRQn            = -1,       /* System Tick Interrupt                        */
/******  ACH specific Interrupt Numbers *************************************************/
    INTERNAL_GPIO_IRQn      = 0,        /* Internal GPIO 0, GPIO 1, GPIO 2 interrupt    */
    SPI0_IRQn               = 1,        /* SPI0 Interrupt, io_ssi_intr                  */
    SPI1_IRQn               = 2,        /* SPI1, Interrupt, io_ssi_intr_1               */
    SPI2_IRQn               = 3,        /* SPI2, Interrupt, io_ssi_intr_2               */
    I2S0_IRQn               = 4,        /* I2S External Interrupt, io_i2s_intr          */
    UART0_IRQn              = 5,        /* Uart0 Interrupt, io_uart_intr                */
    UART1_IRQn              = 6,        /* Uart1, Interrupt, io_uart_intr_1             */
    UART2_IRQn              = 7,        /* Uart2 Interrupt, io_uart_intr_2              */
    I2S1_IRQn               = 8,        /* I2S Codec Interrupt,  io_i2s_intr_1          */
    I2C_IRQn                = 9,        /* I2C Interrupt, io_i2c_intr                   */
    TIM0_IRQn                = 10,       /* Timer Interrupt, io_timer_intr               */
    GPIO_IRQn               = 11,       /* GPIO interrupt, gpio_intr                    */
    WDT_IRQn                = 12,       /* WDT interrupt, wdog_intr                     */
    ADC0_IRQn               = 13,       /* ADC1 interrupt, io_adc1_intr                 */
    ADC1_IRQn               = 14,       /* ADC2 interrupt, io_adc2_intr                 */
    AES_IRQn                = 15,       /* AES interrupt, io_aes_intr                   */
    DMAC_IRQn               = 18,       /* DMAC interrupt, dmac_intr                    */
    TIM1_IRQn               = 19,       /* TIM 1                                        */
    TIM2_IRQn               = 20,       /* TIM 2                                        */
    TIM3_IRQn               = 21        /* TIM 3                                        */
} IRQn_Type;


#include "core_cm4.h"             /* Cortex-M4 processor and core peripherals */
#include "system_ach118x.h"
#include <stdint.h>

/* Peripheral_memory_map  */
#define SRAM_BASE               0x20000000U                 // SRAM base address
#define SRAM_TOP                0x200CFFFFU                 // SRAM top address

/* Peripheral memory map */
#define APB0_PERIPH_BASE        0x40020000U                 // 0x4002 xxxx
#define APB1_PERIPH_BASE        0x40030000U                 // 0x4003 xxxx
#define AHB_PERIPH_BASE         0x40040000U                 // 0x4004 xxxx

#define SPI0_REG_BASE           (APB0_PERIPH_BASE | 0x1800) // 0x4002 1800
#define SPI1_REG_BASE           (APB0_PERIPH_BASE | 0x2200) // 0x4002 2200
#define I2S0_REG_BASE           (APB0_PERIPH_BASE | 0x2800) // 0x4002 2800
#define UART0_REG_BASE          (APB0_PERIPH_BASE | 0x3000) // 0x4002 3000 main uart
#define UART1_REG_BASE          (APB0_PERIPH_BASE | 0x3800) // 0x4002 3800
#define I2S1_REG_BASE           (APB0_PERIPH_BASE | 0x4000) // 0x4002 4000 codec
#define SPI2_REG_BASE           (APB0_PERIPH_BASE | 0x4800) // 0x4002 4800 flash
#define I2C0_REG_BASE           (APB0_PERIPH_BASE | 0x5000) // 0x4002 5000
#define CODEC_REG_BASE          (APB0_PERIPH_BASE | 0x5800) // 0x4002 5800
#define UART2_REG_BASE          (APB0_PERIPH_BASE | 0x6000) // 0x4002 6000

#define GPIO3_REG_BASE          (APB1_PERIPH_BASE | 0x1800) // 0x4003 1800
#define GPIO_INT_BASE           (APB1_PERIPH_BASE | 0x1830)    // 0x4003 1830
#define WDT_REG_BASE            (APB1_PERIPH_BASE | 0x3000) // 0x4003 3000
#define ADC0_REG_BASE           (APB1_PERIPH_BASE | 0x3800) // 0x4003 3800
#define ADC1_REG_BASE           (APB1_PERIPH_BASE | 0x4000) // 0x4003 4000
#define AES_REG_BASE            (APB1_PERIPH_BASE | 0x4800) // 0x4003 4800
#define SYSCFG_REG_BASE         (APB1_PERIPH_BASE | 0x5800) // 0x4003 5800
#define TIM_REG_BASE            (APB1_PERIPH_BASE | 0x6000) // 0x4003 6000
#define DMAC_REG_BASE           (AHB_PERIPH_BASE)           // 0x4004 0000

#define TIM0_REG_BASE           (TIM_REG_BASE + 0x00)
#define TIM1_REG_BASE           (TIM_REG_BASE + 0x14)
#define TIM2_REG_BASE           (TIM_REG_BASE + 0x28)
#define TIM3_REG_BASE           (TIM_REG_BASE + 0x3C)

#define DMA_CHANNEL0_BASE       (DMAC_REG_BASE + 0x000)
#define DMA_CHANNEL1_BASE       (DMAC_REG_BASE + 0x058)
#define DMA_CHANNEL2_BASE       (DMAC_REG_BASE + 0x0b0)
#define DMA_CHANNEL3_BASE       (DMAC_REG_BASE + 0x108)
#define DMA_CHANNEL4_BASE       (DMAC_REG_BASE + 0x160)
#define DMA_CHANNEL5_BASE       (DMAC_REG_BASE + 0x1b8)
#define DMA_CHANNEL6_BASE       (DMAC_REG_BASE + 0x210)
#define DMA_CHANNEL7_BASE       (DMAC_REG_BASE + 0x268)
#define DMA_REG_BASE            (DMAC_REG_BASE + 0x2c0)
#define DMA_PARAM_BASE          (DMAC_REG_BASE + 0x3c8)


#endif /* __ACH118X_DEVICE_H */
/************************ (C) COPYRIGHT ART *****END OF FILE****/

