/***************************************************************************************************
 *                              MCU ACH118X driver library
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     ach118x_codec.h
 * @brief    Function prototypes for the CODEC firmware library.
 * @version  V1.0.0
 * @date     05-Nov-2020
 * @detail
 *  Developers should use these API functions below to configure the integrated audio CODEC in the ACH1180
 *
 *  void codec_master_reset(void);
 *   - Reset the CODEC block
 *
 *  void codec_init(void);
 *   - Reset the CODEC block
 *   - Set the master clocks mclkadc, mclkdac to use CODEC_MSTR_CLK_FREQ_256xFs
 *   - Set the data format to CODEC_DATA_I2S_justified
 *   - Reset the DAC ADC
 *   - Turn on OVERALL_PWR_DN and MIC_BIAS1_PWR_DN
 *   - Set master volume to REC_MASTER_VOLUME_MAX
 *   - Set recording input to REC_MASTER_VOLUME_MAX
 *   - Set recording path to MIC_DIFF
 *   - Set recording mixer to REC_MODE_NORMAL and REC_ENABLE_HI_PASS_FILTER
 *   - Set ALC mode to ALC_LFT_ENABLE
 *
 * int codec_config_samplerate(uint32_t samplerate);
 *   Set the sampling rate to one of:
 *    - CODEC_SAMPLE_FREQ_8kHz
 *    - CODEC_SAMPLE_FREQ_11_025kHz     (Caution: I2S not accurate below 12 MHz)
 *    - CODEC_SAMPLE_FREQ_12kHz
 *    - CODEC_SAMPLE_FREQ_16kHz
 *    - CODEC_SAMPLE_FREQ_22_05kHz
 *    - CODEC_SAMPLE_FREQ_24kHz
 *    - CODEC_SAMPLE_FREQ_32kHz
 *    - CODEC_SAMPLE_FREQ_44_1kHz
 *    - CODEC_SAMPLE_FREQ_48kHz
 *    - CODEC_SAMPLE_FREQ_88_2kHz
 *    - CODEC_SAMPLE_FREQ_96kHz
 *
 * int codec_set_master_volume(uint8_t vol);
 *    - Set master volume where vol must be between PLAYBACK_MASTER_VOLUME_MIN and PLAYBACK_MASTER_VOLUME_MAX
 *
 * int codec_set_headset_volume(uint8_t vol);
 *    - Set headset volume where vol must be between HEADSET_VOLUME_MIN and HEADSET_VOLUME_MAX
 *
 * uint8_t codec_reg_read_byte(uint8_t reg);
 *    - Read CODEC register (byte)
 *
 * uint16_t codec_reg_read_word(uint8_t reg);
 *    - Read CODEC register (word)
 *
 * void codec_reg_write_byte(uint8_t reg, uint8_t);
 *    - Write CODEC register (byte)
 *
 * void codec_reg_write_word(uint8_t reg, uint16_t);
 *    - Write CODEC register (word)
 *
 ***************************************************************************************************/
#ifndef __ACH118X_CODEC_H
#define __ACH118X_CODEC_H

#include <stdint.h>

/***************************************************************************************************
 *                             CODEC Registers
 *  NAME  (Byte)                                       ADDRESS(  Hex  Dec )  DEFAULT
 *  CODEC_RESET                                                 0x00   0       0x03
 *  CODEC_LATCH                                                 0x01   1       0x00
 *  CODEC_CLK_SEL_0                                             0x02   2       0x00
 *  CODEC_CLK_SEL_1                                             0x03   3       0x00
 *  CODEC_I2S_CFG_0                                             0x0C   12      0x08
 *  CODEC_I2S_CFG_1                                             0x0D   13      0x08
 *  CODEC_I2S_CFG_2                                             0x0E   14      0x01
 *  CODEC_STANDBY_SLEEP_PWR_CTRL                                0x11   17      0xAE
 *  CODEC_PWR_CONSUMPTION_MANAGEMENT                            0x12   18      0x00
 *  CODEC_PWR_CTRL_0                                            0x15   21      0x00
 *  CODEC_PWR_CTRL_1                                            0x16   22      0x00
 *  CODEC_PWR_CTRL_3                                            0x18   24      0x00
 *  CODEC_PWR_CTRL_5                                            0x1A   26      0x00
 *  CODEC_MUTE_CTRL_0                                           0x1D   29      0x00
 *  CODEC_MUTE_CTRL_2                                           0x1F   31      0x00
 *  CODEC_RECORDING_CHANNEL_MASTER_VOLUME_CTRL_LSB              0x24   36      0x14
 *  CODEC_RECORDING_CHANNEL_MASTER_VOLUME_CTRL_MSB              0x25   37      0x14
 *  CODEC_PGA_VOLUME_CTRL_LSB                                   0x26   38      0x12
 *  CODEC_PGA_VOLUME_CTRL_MSB                                   0x27   39      0x12
 *  CODEC_PLAYBACK_CHANNEL_MASTER_VOLUME_CTRL_LSB               0x34   52      0x54
 *  CODEC_PLAYBACK_CHANNEL_MASTER_VOLUME_CTRL_MSB               0x35   53      0x54
 *  CODEC_STEREO_HEADSET_DRIVER_VOLUME_CTRL_LSB                 0x38   56      0x28
 *  CODEC_STEREO_HEADSET_DRIVER_VOLUME_CTRL_MSB                 0x39   57      0x28
 *  CODEC_RECORDING_CHANNEL_INPUT_SELECTOR_LSB                  0x47   71      0x01
 *  CODEC_RECORDING_CHANNEL_INPUT_SELECTOR_MSB                  0x48   72      0x01
 *  CODEC_HEADSET_DRIVER_MIXER_MUX_CTRL_LSB                     0x57   87      0x01
 *  CODEC_HEADSET_DRIVER_MIXER_MUX_CTRL_MSB                     0x58   88      0x01
 *  CODEC_POP_SUPPRESSION_CTRL_REG_1                            0xA2   162     0x03
 *  CODEC_POP_SUPPRESSION_CTRL_REG_2                            0xA3   163     0xF3
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_1_LSB                        0xA4   164     0x00
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_1_MSB                        0xA5   165     0x04
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_2_LSB                        0xA6   166     0x03
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_2_MSB                        0xA7   167     0x08
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_3_LSB                        0xA8   168     0x00
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_3_MSB                        0xA9   169     0x0C
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_4_LSB                        0xAA   170     0x00
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_4_MSB                        0xAB   171     0x10
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_5_LSB                        0xAC   172     0x00
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_5_MSB                        0xAD   173     0x14
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_6_LSB                        0xAE   174     0x00
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_6_MSB                        0xAF   175     0x18
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_7_LSB                        0xB0   176     0x01
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_7_MSB                        0xB1   177     0x1C
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_8_LSB                        0xB2   178     0x0h
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_8_MSB                        0xB3   179     0x20
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_9_LSB                        0xB4   180     0x0h
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_9_MSB                        0xB5   181     0x24
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_10_LSB                       0xB6   182     0x00
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_10_MSB                       0xB7   183     0x28
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_11_LSB                       0xB8   184     0x7F
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_11_MSB                       0xB9   185     0x2C
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_12_LSB                       0xBA   186     0x24
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_12_MSB                       0xBB   187     0x2C
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_13_LSB                       0xBC   188     0x56
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_13_MSB                       0xBD   189     0x30
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_14_LSB                       0xBE   190     0x00
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_14_MSB                       0xBF   191     0x34
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_15_LSB                       0xC0   192     0x37
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_15_MSB                       0xC1   193     0x38
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_16_LSB                       0xC2   194     0x7F
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_16_MSB                       0xC3   195     0x3C
 *  CODEC_ALC_CTRL_REG_0                                        0xC8   200     0x00
 *  CODEC_ALC_CTRL_REG_1                                        0xC9   201     0x06
 *  CODEC_ALC_CTRL_REG_2                                        0xCA   202     0x00
 *  CODEC_ALC_RAMP_DOWN_CTRL                                    0xCB   203     0x06
 *  CODEC_ALC_RAMP_UP_CTRL                                      0xCC   204     0x06
 *  CODEC_ALC_MAXIMUM_AUTOMATICALLY_SET_GAIN                    0xCD   205     0xFF
 *  CODEC_SPECIAL_MODE                                          0xD2   210
 *  CODEC_DAC_MIXER                                             0xD4   212     0x00
 *  CODEC_DIGITAL_ASSISTED_ANALOG_TEST_REG_1                    0xF0   240     0x02
 *  CODEC_DIGITAL_ASSISTED_ANALOG_TEST_REG_2                    0xF1   241     0x02
 *  CODEC_DIGITAL_ASSISTED_ANALOG_TEST_REG_3                    0xF2   242     0x00
 *  CODEC_DIGITAL_ASSISTED_ANALOG_TEST_REG_4                    0xF3   243     0x0A
 *  CODEC_DIGITAL_ASSISTED_ANALOG_TEST_REG_5                    0xF4   244     0x00
 *  CODEC_PURE_ANALOG_TEST_REG_0                                0xF8   248     0x49
 *  CODEC_PURE_ANALOG_TEST_REG_1                                0xF9   249     0xFC
 *  CODEC_PURE_ANALOG_TEST_REG_2                                0xFA   250     0x50
 *  CODEC_PURE_ANALOG_TEST_REG_3                                0xFB   251     0x84
 *
 *  NAME  (Word)                                       ADDRESS(  Hex  Dec )
 *  CODEC_RECORDING_CHANNEL_MASTER_VOLUME_CTRL                  0x24   36
 *  CODEC_PGA_VOLUME_CTRL                                       0x26   38
 *  CODEC_PLAYBACK_CHANNEL_MASTER_VOLUME_CTRL                   0x34   52
 *  CODEC_STEREO_HEADSET_DRIVER_VOLUME_CTRL                     0x38   56
 *  CODEC_RECORDING_CHANNEL_INPUT_SELECTOR                      0x47   71
 *  CODEC_HEADSET_DRIVER_MIXER_MUX_CTRL                         0x57   87
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_1                            0xA4   164
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_2                            0xA6   166
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_3                            0xA8   168
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_4                            0xAA   170
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_5                            0xAC   172
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_6                            0xAE   174
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_7                            0xB0   176
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_8                            0xB2   178
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_9                            0xB4   180
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_10                           0xB6   182
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_11                           0xB8   184
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_12                           0xBA   186
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_13                           0xBC   188
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_14                           0xBE   190
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_15                           0xC0   192
 *  CODEC_POP_SUPPRESSION_CTRL_SEG_16                           0xC2   194
 ***************************************************************************************************/

// R21 - CODEC_PWR_CTRL_0
#define OVERALL_PWR_DN      0x01
#define MIC_BIAS1_PWR_DN    0x04

// R22 - CODEC_PWR_CTRL_1
#define LFT_ADC_PWR         0x01
#define RGT_ADC_PWR         0x02
#define LFT_PGA_PWR         0x04
#define RGT_PGA_PWR         0x08

// R24 - CODEC_PWR_CTRL_3
#define LFT_DAC_PWR         0x01
#define RGT_DAC_PWR         0x02
#define RGT_HEADSET_PWR     0x10
#define LFT_HEADSET_PWR     0x20

// R52, R53 - Playback digital master volume control
#define PLAYBACK_MASTER_VOLUME_MIN      0x00    // -126 dB
#define PLAYBACK_MASTER_VOLUME_MAX      0x54    //    0 dB
#define PLAYBACK_MASTER_VOLUME_DEFAULT  0x20

// R56, R57 - Headset driver volume control
#define HEADSET_VOLUME_MIN              0x00    // -40 dB
#define HEADSET_VOLUME_MAX              0x2E    //   6 dB
#define HEADSET_VOLUME_DEFAULT          0x20    //   0 dB

// R36, R37 - Recording digital master volume control selection
#define REC_MASTER_VOLUME_MIN           0x54    // -96 dB
#define REC_MASTER_VOLUME_MAX           0x00    //  30 dB
#define REC_MASTER_VOLUME_DEFAULT       0x10

// R38, R39 - Recording input PGA volume control
#define INP_PGA_VOLUME_MIN              0x00    // -18 dB
#define INP_PGA_VOLUME_MAX              0x3C    //  42 dB
#define INP_PGA_VOLUME_DEFAULT          0x20

// R71, R72 - CODEC_RECORDING_CHANNEL_INPUT_SELECTOR (lft/rgt)
#define INPUT1_DIFF     0
#define INPUT1_SINGLE   1
#define INPUT2_SINGLE   3
#define MIC_DIFF        4
#define MIC_SINGLE      5

// R211 - Recording Digital Mixer Selection
#define REC_MODE_NORMAL             0x00
#define REC_MODE_MIXED              0x01
#define REC_MODE_SWAPPED            0x02
#define REC_ENABLE_HI_PASS_FILTER   0x04

// R212 - Playback Digital Mixer Selection
#define PLAY_MODE_STEREO            0x00
#define PLAY_MODE_MIXED             0x01
#define PLAY_MODE_SWITCH            0x02

// R200 - ALC mode selection
#define ALC_DISABLE                 0x00
#define ALC_LFT_ENABLE              0x01
#define ALC_RGT_ENABLE              0x02
#define ALC_BOTH_ENABLE             0x03

// R205 - ALC automatic setting of maximum gain (see r38, r39)

// R201 - Output target level
// 00x: 0dBFS target
// 01x: -3dBFS target
// 10x: -6dBFS target
// 11x: -12dBFS target (default)
// xx1: Enable peak limiter
#define ALC_PEAK_LIMIT_0       0x00
#define ALC_PEAK_LIMIT_1       0x01
#define ALC_PEAK_LIMIT_2       0x10
#define ALC_PEAK_LIMIT_3       0x11



// CODEC_RESET
typedef struct {
    uint8_t
        rstadcdpz       : 1,
        rstdacdpz       : 1,
        reserved_2_7    : 6;
} CODEC_RESET_t;


// CODEC_CLK_SEL_0
typedef struct {
    uint8_t
        mclkseladc      : 4,
        reserved_4_7    : 4;
} CODEC_CLK_SEL_0_t;
// CODEC_CLK_SEL_1
typedef struct {
    uint8_t
        mclkseldac      : 4,
        reserved_4_7    : 4;
} CODEC_CLK_SEL_1_t;

#define CODEC_MSTR_CLK_FREQ_256xFs          0
#define CODEC_MSTR_CLK_FREQ_384xFs          1
#define CODEC_MSTR_CLK_FREQ_12MHz_USB       2
#define CODEC_MSTR_CLK_FREQ_24MHz_USB       3
#define CODEC_MSTR_CLK_FREQ_12_288MHz       4
#define CODEC_MSTR_CLK_FREQ_24_576MHz       5
#define CODEC_MSTR_CLK_FREQ_11_2896MHz      6 // Caution: I2S not accurate below 12 MHz
#define CODEC_MSTR_CLK_FREQ_22_5792MHz      7
#define CODEC_MSTR_CLK_FREQ_18_432MHz       8
#define CODEC_MSTR_CLK_FREQ_36_864MHz       9
#define CODEC_MSTR_CLK_FREQ_16_9344MHz      10
#define CODEC_MSTR_CLK_FREQ_33_8688MHz      11

// CODEC_I2S_CFG_0
typedef struct {
    uint8_t
        i2sfsadc        : 4,
        reserved_4_7    : 4;
} CODEC_I2S_CFG_0_t;
// CODEC_I2S_CFG_1
typedef struct {
    uint8_t
        i2sfsdac        : 4,
        reserved_4_7    : 4;
} CODEC_I2S_CFG_1_t;
#define CODEC_SAMPLE_FREQ_8kHz              0
#define CODEC_SAMPLE_FREQ_11_025kHz         1 // Caution: I2S not accurate below 12 MHz
#define CODEC_SAMPLE_FREQ_12kHz             2
#define CODEC_SAMPLE_FREQ_16kHz             3
#define CODEC_SAMPLE_FREQ_22_05kHz          4
#define CODEC_SAMPLE_FREQ_24kHz             5
#define CODEC_SAMPLE_FREQ_32kHz             6
#define CODEC_SAMPLE_FREQ_44_1kHz           7
#define CODEC_SAMPLE_FREQ_48kHz             8
#define CODEC_SAMPLE_FREQ_88_2kHz           9
#define CODEC_SAMPLE_FREQ_96kHz             10
#define CODEC_SAMPLE_FREQ_192kHz            11

// CODEC_I2S_CFG_2
typedef struct {
    uint8_t
        i2smode         : 3,
        i2smsmode       : 1,
        reserved_4_7    : 4;
} CODEC_I2S_CFG_2_t;
#define CODEC_DATA_Right_justified       0
#define CODEC_DATA_I2S_justified         1
#define CODEC_DATA_Left_justified        2
#define CODEC_DATA_Burst_1_format        3
#define CODEC_DATA_Burst_2_format        4
#define CODEC_DATA_Mono_Burst_1_format   5
#define CODEC_DATA_Mono_Burst_2_format   6
#define CODEC_MODE_Master   1
#define CODEC_MODE_Slave    0


// CODEC_PWR_CTRL_0
typedef struct {
    uint8_t
        pdz             : 1,
        reserved_1      : 1,
        pdmbias1z       : 1,
        reserved_3_7    : 5;
} CODEC_PWR_CTRL_0_t;

// CODEC_DIGITAL_ASSISTED_ANALOG_TEST_REG_1
typedef struct {
    uint8_t
        reserved_7      : 1,
        cfgprechvcm     : 2,
        cfgprechvcm2x   : 1,
        reserved_4_7    : 4;
} CODEC_DIGITAL_ASSISTED_ANALOG_TEST_REG_1_t;

// CODEC_DIGITAL_ASSISTED_ANALOG_TEST_REG_2
typedef struct {
    uint8_t
        bypasspwrseq    : 1,
        reserved_1_7    : 6;
} CODEC_DIGITAL_ASSISTED_ANALOG_TEST_REG_2_t;

// CODEC_DIGITAL_ASSISTED_ANALOG_TEST_REG_3
typedef struct {
    uint8_t
        reserved_0_4    : 5,
        bypassmutedly   : 1,
        reserved_6_7    : 2;
} CODEC_DIGITAL_ASSISTED_ANALOG_TEST_REG_3_t;

// CODEC_DIGITAL_ASSISTED_ANALOG_TEST_REG_4
typedef struct {
    uint8_t
        reserved_0      : 1,
        cfgadcdither    : 1,
        demcfg          : 2,
        noiseShapeEn    : 1,
        ditherLvl       : 2,
        disableSR       : 1;
} CODEC_DIGITAL_ASSISTED_ANALOG_TEST_REG_4_t;

// CODEC_DIGITAL_ASSISTED_ANALOG_TEST_REG_5
typedef struct {
    uint8_t
        cfgiref         : 2,
        cfgres          : 2,
        reserved_4_7    : 1;
} CODEC_DIGITAL_ASSISTED_ANALOG_TEST_REG_5_t;

// CODEC_SPECIAL_MODE
#define CODEC_MODE_NORMAL                               0
#define CODEC_MODE_DIGITAL_LOOP_BACK                    1
#define CODEC_MODE_DIGITAL_FILTER_BYPASS                2
#define CODEC_MODE_DIGITAL_AUDIO_INTERFACE_LOOP_BACK    3


//------------------------------------------------------------------
// API prototypes
//------------------------------------------------------------------

/**
  * @brief  Initialize the CODEC
  * @retval None
  */
void  codec_init(void);

/**
  * @brief  Configure the CODEC sample rate
  * @param  samplerate: 8000, 11025, 12000, 16000, 22050, 24000, 32000,
  *                    44100, 48000, 88200, 96000, 192000,
  * @retval -1 if error, 0 on success
  */
int  codec_config_samplerate(uint32_t samplerate);

/**
  * @brief  Set master volume
  * @param  vol: 0 - PLAYBACK_MASTER_VOLUME_MAX (0x54)
  * @retval None
  */
int  codec_set_master_volume(uint8_t vol);

/**
  * @brief  Set headset volume
  * @param  vol: 0 - HEADSET_VOLUME_MAX (0x2E)
  * @retval None
  */
int  codec_set_headset_volume(uint8_t vol);

/**
  * @brief  Read a byte register
  * @param  reg: 0 - 251
  * @retval None
  */
uint8_t  codec_reg_read_byte(uint8_t reg);

/**
  * @brief  Read a word (16 bit) register
  * @param  reg: 36 - 194
  * @retval None
  */
uint16_t  codec_reg_read_word(uint8_t reg);

/**
  * @brief  Write to a byte register
  * @param  reg: 0 - 251
  * @param  data: Byte
  * @retval None
  */
void  codec_reg_write_byte(uint8_t reg, uint8_t data);

/**
  * @brief  Write to a word (16 bit) register
  * @param  reg: one SPI initialisation structure
  * @param  data: Word (16 bit)
  * @retval None
  */
void  codec_reg_write_word(uint8_t reg, uint16_t data);

#endif // __ACH118X_CODEC_H
/************************ (C) COPYRIGHT ART *****END OF FILE****/

