/***************************************************************************************************
 *                              MCU ACH118X driver library
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     ach118x_aes.h
 * @brief    Function prototypes and definitions for the AES firmware library.
 * @version  V1.0.0
 * @date     05-Nov-2020
 ***************************************************************************************************/
#ifndef __ACH118X_AES_H
#define __ACH118X_AES_H

#ifdef __cplusplus
 extern "C" {
#endif

#include "ach118x_type.h"
#include "ach118x_map.h"

#define AES_INT_STATUS_REG  *((vu32*)0x40034810)
#define AES_INT_CLEAR_REG   *((vu32*)0x40034814)

//------------------------------------------------------------------
// Decrypt or Encrypt status register bits
//------------------------------------------------------------------
//AES key schedule completed status out bit ('1' = completed)
#define AES_KEYSCHED_DONE_BIT_SHIFT         (0)
#define AES_KEYSCHED_DONE_BIT_MASK          (1 << AES_KEYSCHED_DONE_BIT_SHIFT)
//AES roundkey select status bit
#define AES_ROUNDKEY_SEL_BIT_SHIFT          (1)
#define AES_ROUNDKEY_SEL_BIT_MASK           (0x3F << AES_ROUNDKEY_SEL_BIT_SHIFT)
//AES roundkey valid status bit
#define AES_ROUNDKEY_VALID_BIT_SHIFT        (7)
#define AES_ROUNDKEY_VALID_BIT_MASK         (1 << AES_ROUNDKEY_VALID_BIT_SHIFT)
//AES key word required out status bit
#define AES_KEYWORD_REQUIREDOUT_BIT_SHIFT   (8)
#define AES_KEYWORD_REQUIREDOUT_BIT_MASK    (1 << AES_KEYWORD_REQUIREDOUT_BIT_SHIFT)
//AES decrypt or encrypt engine busy status out ('1' = busy)
#define AES_BUSY_BIT_SHIFT                  (9)
#define AES_BUSY_BIT_MASK                   (1 << AES_BUSY_BIT_SHIFT)
//AES decrypt or encrypt engine completed status out ('1' = completed), interrupt generated
#define AES_DONE_BIT_SHIFT                  (10)
#define AES_DONE_BIT_MASK                   (1 << AES_DONE_BIT_SHIFT)
//AES status definition
#define AES_KEYSCHED_DONE_STATUS            (AES_KEYSCHED_DONE_BIT_MASK)
#define AES_ROUNDKEY_VALID_STATUS           (AES_ROUNDKEY_VALID_BIT_MASK)
#define AES_KEYWORD_REQUIREDOUT_STATUS      (AES_KEYWORD_REQUIREDOUT_BIT_MASK)
#define AES_BUSY_STATUS                     (AES_BUSY_BIT_MASK)
#define AES_ENCRYPT_DONE_STATUS             (0x00000001)
#define AES_DECRYPT_DONE_STATUS             (0x00000002)

//------------------------------------------------------------------
// Decrypt or Encrypt control register bits
//------------------------------------------------------------------
//AES key length select bit ('00' = 128 bits, '01' = 192 bits, '1X' = 256 bits)
#define AES_KEYLEN_SEL_BIT_SHIFT            (1)
#define AES_KEYLEN_SEL_BIT_MASK             (0x3 << AES_KEYLEN_SEL_BIT_SHIFT)
//AES mode select bit (encrypt/init = '1', decrypt = '0')
#define AES_MODE_SEL_BIT_SHIFT              (3)
#define AES_MODE_SEL_BIT_MASK               (1 << AES_MODE_SEL_BIT_SHIFT)
//AES decrypt or encrypt request control in ('1' = start decryption)
#define AES_REQUEST_CTRL_BIT_SHIFT          (4)
#define AES_REQUEST_CTRL_BIT_MASK           (1 << AES_REQUEST_CTRL_BIT_SHIFT)


//------------------------------------------------------------------
// AES peripheral registers
//------------------------------------------------------------------
typedef struct
{
    vu32 pt_decr0;      //0x54, First Data Read Back From AES DECRYPT region
    vu32 pt_decr1;      //0x58, Second Data Read Back From AES DECRYPT region
    vu32 pt_decr2;      //0x5C, Third Data Read Back From AES DECRYPT region
    vu32 pt_decr3;      //0x60, Last Data Read Back From AES DECRYPT region
    vu32 ct_enc0;       //0x64, First Data Read Back From AES ENCRYPT region
    vu32 ct_enc1;       //0x68, Second Data Read Back From AES ENCRYPT region
    vu32 ct_enc2;       //0x6C, Third Data Read Back From AES ENCRYPT region
    vu32 ct_enc3;       //0x70, Last Data Read Back From AES ENCRYPT region
    vu32 decr_status;   //0x74, Decrypt status register
    vu32 enc_status;    //0x78, Encrypt status register
    vu32 decr_ctrl;     //0x7C, Decrypt control register
    vu32 enc_ctrl;      //0x80, Encrypt control register
    vu32 ct_decr0;      //0x84, Cipher text code 0
    vu32 ct_decr1;      //0x88, Cipher text code 1
    vu32 ct_decr2;      //0x8C, Cipher text code 2
    vu32 ct_decr3;      //0x90, Cipher text code 3
    vu32 pt_enc0;       //0x94, Plain text code0 for encrypt
    vu32 pt_enc1;       //0x98, Plain text code1 for encrypt
    vu32 pt_enc2;       //0x9C, Plain text code2 for encrypt
    vu32 pt_enc3;       //0xA0, Plain text code3 for encrypt
    vu32 decr_key0;     //0xA4, Decrypt key 0
    vu32 decr_key1;     //0xA8, Decrypt key 1
    vu32 decr_key2;     //0xAC, Decrypt key 2
    vu32 decr_key3;     //0xB0, Decrypt key 3
    vu32 decr_key4;     //0xB4, Decrypt key 4
    vu32 decr_key5;     //0xB8, Decrypt key 5
    vu32 decr_key6;     //0xBC, Decrypt key 6
    vu32 decr_key7;     //0xC0, Decrypt key 7
    vu32 enc_key0;      //0xC4, Encrypt key 0
    vu32 enc_key1;      //0xC8, Encrypt key 1
    vu32 enc_key2;      //0xCC, Encrypt key 2
    vu32 enc_key3;      //0xD0, Encrypt key 3
    vu32 enc_key4;      //0xD4, Encrypt key 4
    vu32 enc_key5;      //0xD8, Encrypt key 5
    vu32 enc_key6;      //0xDC, Encrypt key 6
    vu32 enc_key7;      //0xE0, Encrypt key 7
} AES_TypeDef;

typedef enum{
    MODE_Encryption = 0,
    MODE_Decryption
}AES_MODE_T;

typedef enum{
    KEY_Len_128 = 0,
    KEY_Len_192 = 0x01,
    KEY_Len_256 = 0x02,
}AES_KEY_LEN_T;

typedef struct{
    AES_MODE_T mode;
    AES_KEY_LEN_T key_len;
    uint32_t *pKeyBuf;
}AES_InitTypeDef;

typedef enum{
    ENC_Finish_Flag = 0x01,
    DEC_Finish_Flag = 0x02
}AES_IT_FLAG;


//------------------------------------------------------------------
// API prototypes
//------------------------------------------------------------------

/**
  * @brief  xxxxxxxxxxxx
  * @param  AES_InitStruct: xxxxxxxxxxxxxxxxxxxxx
  * @retval none
  */
void AES_Init(AES_InitTypeDef* AES_InitStruct);

/**
  * @brief  xxxxxxxxxxxx
  * @param  mode: xxxxxxxxxxxxxxxxxxxxx
  * @retval none
  */
void AES_ChangeMode(AES_MODE_T mode);

/**
  * @brief  xxxxxxxxxxxx
  * @param  pBuf: xxxxxxxxxxxxxxxxxxxxx
  * @retval none
  */
void AES_WriteCipherToDecrypt(uint32_t *pBuf);

/**
  * @brief  xxxxxxxxxxxx
  * @param  pBuf: xxxxxxxxxxxxxxxxxxxxx
  * @retval none
  */
void AES_ReadPlain(uint32_t *pBuf);

/**
  * @brief  xxxxxxxxxxxx
  * @param  pBuf: xxxxxxxxxxxxxxxxxxxxx
  * @retval none
  */
void AES_WritePlainToEncrypt(uint32_t *pBuf);

/**
  * @brief  xxxxxxxxxxxx
  * @param  pBuf: xxxxxxxxxxxxxxxxxxxxx
  * @retval none
  */
void AES_ReadCipher(uint32_t *pBuf);

/**
  * @brief  xxxxxxxxxxxx
  * @param  pBuf: xxxxxxxxxxxxxxxxxxxxx
  * @retval none
  */
void AES_ModeCmd(AES_MODE_T mode, FunctionalState NewState);

/**
  * @brief  xxxxxxxxxxxx
  * @retval none
  */
uint32_t AES_GetITStatus(void);

/**
  * @brief  xxxxxxxxxxxx
  * @param  yyyyy: xxxxxxxxxxxxxxxxxxxxx
  * @retval none
  */
FlagStatus AES_GetITFlag(AES_IT_FLAG flag);

/**
  * @brief  xxxxxxxxxxxx
  * @param  flag: xxxxxxxxxxxxxxxxxxxxx
  * @retval none
  */
void AES_ClearITFlag(void);


#ifdef __cplusplus
    }//extern "C"
#endif

#endif

/************************ (C) COPYRIGHT ART *****END OF FILE****/

