/***************************************************************************************************
 *                              MCU ACH118X driver library
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     ach118x_adc.h
 * @brief    Function prototypes for the ADC firmware library.
 * @version  V1.0.0
 * @date     05-Nov-2020
 ***************************************************************************************************/
#ifndef __ACH118X_ADC_H
#define __ACH118X_ADC_H

#ifdef __cplusplus
    extern "C" {
#endif

#include "ach118x_type.h"
#include "ach118x_map.h"

//ADC0 interrupt status register
#define ADC0_INTSTATUS_REG          *(uint32_t*)(ADC0_REG_BASE + 0x10)
//ADC0 interrupt clear register
#define ADC0_INTCLR_REG             *(uint32_t*)(ADC0_REG_BASE + 0x14)
//ADC1 interrupt status register
#define ADC1_INTSTATUS_REG          *(uint32_t*)(ADC1_REG_BASE + 0x10)
//ADC1 interrupt clear register
#define ADC1_INTCLR_REG             *(uint32_t*)(ADC1_REG_BASE + 0x14)

//------------------------------------------------------------------
// ADC interrupt status register bits.
//------------------------------------------------------------------
#define ADC_INTSTATUS_BIT_MASK      (0x01)

//------------------------------------------------------------------
// ADC interrupt enable and mask register bits.
//------------------------------------------------------------------
//ADC interrupt control bit.
#define ADC_INTEN_BIT_MASK          (0x01)
//ADC interrupt mask bit.
#define ADC_INTMASK_BIT_MASK        (0x02)

//------------------------------------------------------------------
// ADC end of conversion register bits.
//------------------------------------------------------------------
//ADC conversion done bit. The option can be one of the @FlagStatus enum values.
//#define ADC_CONVDONE_BIT_SHIFT      (0)
#define ADC_CONVDONE_BIT_MASK         (0x01)
//ADC conversion result 10 bits.
#define ADC_CONVRESULT_BIT_MASK       (0x000007FE)
//#define ADC_CONVRESULT_BIT_SHIFT    (1)
//#define ADC_CONVRESULT_BIT_MASK     (0x000003FF << ADC_CONVRESULT_BIT_SHIFT)

//------------------------------------------------------------------
// ADC start of conversion register bits.
// The option of this bit can be one of the @FlagStatus enum values.
//------------------------------------------------------------------
#define ADC_CONVSTA_BIT_MASK        (0x01)

//------------------------------------------------------------------
// ADC power down register bits
//------------------------------------------------------------------
#define ADC_PD_BIT_MASK             (0x01)

//------------------------------------------------------------------
// ADC clock divider register bits
// These bits show the ADC clock divider ratio
//------------------------------------------------------------------
#define ADC_CLKDIV_BIT_MASK         (0x0001FFFF)

//------------------------------------------------------------------
// ADC registers definition
//------------------------------------------------------------------
typedef struct
{
    uint32_t INTEN;      //0xD0, AD conversion interrupt register.
    uint32_t EOC;        //0xD4, AD conversion result and done flag.
    uint32_t SOC;        //0xD8, AD conversion start cmd.
    uint32_t PD;         //0xDC, AD power down.
    uint32_t CLKDIV;     //0xE0, Clock divider ratio.
} ADC_Typedef;

typedef enum
{
    POWER_Off=0,
    POWER_On
}PowerCmd;


//------------------------------------------------------------------
// API prototypes
//------------------------------------------------------------------

/**
  * @brief  Set ADC clock divider.
  * @param  ADCx: Where x can be (0 or 1) to select the ADCx peripheral.
  * @param  div: Divide ratio.
  * @retval none
  */
void        ADC_SetClockDiv(ADC_Typedef* ADCx, uint32_t div);

/**
  * @brief  ADC conersion start.
  * @param  ADCx: Where x can be (0 or 1) to select the ADCx peripheral.
  * @retval none
  */
void        ADC_StartConversion(ADC_Typedef* ADCx);

/**
  * @brief  Check ADC conversion status.
  * @param  ADCx: Where x can be (0 or 1) to select the ADCx peripheral.
  * @retval This value can be one of @FlagStatus enum:
  *           SET:   Specified ADC conversion done.
  *           RESET: Specified ADC conversion isn't done.
  */
FlagStatus  ADC_GetConversionStatus(ADC_Typedef* ADCx);

/**
  * @brief  Read ADC conversion result.
  * @param  ADCx: Where x can be (0 or 1) to select the ADCx peripheral.
  * @retval This value is ADC conversion result, which only low 10bits available.
  */
uint16_t    ADC_GetConversionResult(ADC_Typedef* ADCx);

/**
  * @brief  ADC interrupt control.
  * @param  ADCx: Where x can be (0 or 1) to select the ADCx peripheral.
  * @param  status: ENABLE or DISABLE
  * @retval none
  */
void        ADC_ITCmd(ADC_Typedef* ADCx, FunctionalState status);

/**
  * @brief  ADC interrupt status.
  * @param  ADCx: Where x can be (0 or 1) to select the ADCx peripheral.
  * @retval SET: Interrupt trigged, RESET: No interrupt trigged.
  */
FlagStatus  ADC_GetITFlag(ADC_Typedef* ADCx);

/**
  * @brief  ADC interrupt clear.
  * @param  ADCx: Where x can be (0 or 1) to select the ADCx peripheral.
  * @retval none
  */
void        ADC_ClearITFlag(ADC_Typedef* ADCx);

/**
  * @brief  ADC power down control.
  * @param  ADCx: Where x can be (0 or 1) to select the ADCx peripheral.
  * @param  cmd: ENABLE - ADC power down enable, DISABLE - ADC power down disable.
  * @retval none
  */
void        ADC_PowerCmd(ADC_Typedef* ADCx, PowerCmd cmd);


#ifdef __cplusplus
    }//extern "C"
#endif

#endif // __ACH118X_ADC_H
/************************ (C) COPYRIGHT ART *****END OF FILE****/

