/***************************************************************************************************
 *                               Generic Kernel Interface
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     GKI_uart.c
 * @brief    GKI function interface for UART.
 * @version  V1.0.0
 * @date     05-Nov-2020
 ***************************************************************************************************/

#include "ach118x.h"
#include "GKI_common.h"

#define PRINT_UART  UART0

/******************************************************************************
Function:void GKI_UartInit(UART_Typedef* UARTx,uint32_t baudrate)
Input:UART_Typedef* UARTx               -> select UART port
      uint32_t baudrate                   ->UARTx baudrate
Output:void
Return:void
Description:In GKI interface,initialize UART device with specific baudrate
*****************************************************************************/
void GKI_UartInit(UART_Typedef* UARTx, uint32_t baudrate)
{
    UART_InitTypeDef UART_InitStruct;
    if(UARTx == UART0){
        NVIC_EnableIRQ(UART0_IRQn);
        NVIC_SetPriority(UART0_IRQn, 8);
        SYS_SetITMaskFlag(IRQ_UART0,SET);
    }
    else if(UARTx == UART1){
        SYS_SetPinRemap0(PINMUX_PIN_01, REMAP0_PIN_01_Sel_UART1_CTS);
        SYS_SetPinRemap0(PINMUX_PIN_02, REMAP0_PIN_02_Sel_UART1_RTS);
        SYS_SetPinRemap0(PINMUX_PIN_03, REMAP0_PIN_03_Sel_UART1_TX);
        SYS_SetPinRemap0(PINMUX_PIN_04, REMAP0_PIN_04_Sel_UART1_RX);
        NVIC_EnableIRQ(UART1_IRQn);
        NVIC_SetPriority(UART1_IRQn, 8);
        SYS_SetITMaskFlag(IRQ_UART1,SET);
    }
    else if(UARTx == UART2){
        SYS_SetPinRemap0(PINMUX_PIN_09, REMAP0_PIN_09_Sel_UART2_RX);
        SYS_SetPinRemap0(PINMUX_PIN_10, REMAP0_PIN_10_Sel_UART2_TX);
        SYS_SetPinRemap0(PINMUX_PIN_11, REMAP0_PIN_11_Sel_UART2_CTS);
        SYS_SetPinRemap0(PINMUX_PIN_12, REMAP0_PIN_12_Sel_UART2_RTS);
        NVIC_EnableIRQ(UART2_IRQn);
        SYS_SetITMaskFlag(IRQ_UART2,SET);
        NVIC_SetPriority(UART2_IRQn, 8);

    }
    UART_InitStruct.UART_BaudRate=baudrate;
    UART_InitStruct.UART_WordLength = UART_LEN_8;
    UART_InitStruct.UART_StopBits   = UART_STOP_1;
    UART_InitStruct.UART_Parity     = UART_NONE_PARITY;
    UART_InitStruct.UART_HardwareFlowControl = UART_FLOWCON_DISABLE;
    UART_Init(UARTx, &UART_InitStruct);
    UART_FIFOCmd(UARTx, UART_ALL_CONFIG,DISABLE); // clear old FIFO setting
    UART_ITCmd(UARTx, UART_ALL_CONFIG,DISABLE);   // clear old interrupt setting

    UART_FIFOCmd(UARTx,UART_FIFOE|UART_TET_EMPTY|UART_RCT_HFULL, ENABLE);
    UART_ITCmd(UARTx, UART_RCV_DATA_IT, ENABLE);
}


/******************************************************************************
Function:GKI_UartSendData(UART_Typedef* UARTx,uint8_t data)
Input:UART_Typedef* UARTx      -> select UART port
      data                     -> The byte be send via UART
Output:void
Return:void
Description:In GKI interface,send one byte via uart
*****************************************************************************/
void GKI_UartSendData(UART_Typedef* UARTx, uint8_t data)
{
    UART_SendData(UARTx,data, UART_FOREVER);
}

/******************************************************************************
Function:uint8_t GKI_UartReceiveData(UART_Typedef* UARTx)
Input:UART_Typedef* UARTx     -> select UART port
Output:uint8_t data             -> the byte read from UART
Return:the byte read from UART
Description:in GKI interface,read one byte from uart with 50000 clock timeout
*****************************************************************************/
uint8_t GKI_UartReceiveData(UART_Typedef* UARTx)
{
    return UART_ReceiveData(UARTx, UART_FOREVER);
}

/******************************************************************************
Function:uint8_t GKI_UartRxDataRead(UART_Typedef* UARTx)
Input:UART_Typedef* UARTx     -> select UART port
Output:
Return:1 -> Data ready       0-> not ready
Description:In GKI interface,read one byte from uart with block way
*****************************************************************************/
uint8_t GKI_UartRxDataReady(UART_Typedef* UARTx)
{
    FlagStatus retval;
    retval = UART_GetLineStatus(UARTx, UART_DR_STATUS);
    if(retval != RESET)
    {
      return 1;
    }
    else
    {
      return 0;
    }
}

/******************************************************************************
Function:GKI_UartSendString(char *str)
Input:UART_Typedef* UARTx     -> select UART port
      char *str                 -> The character string is send
Output:void
Return:void
Description:With GKI interface,send character string via UART
*****************************************************************************/
void GKI_UartSendString(UART_Typedef* UARTx,char *str)
{
  char *p = str;
  while(*p != 0)
  {
    GKI_UartSendData(UARTx,*p);
    p++;
  }
}

/*Standard format output*/
void GKI_Printf(char *fmt, ...)
{
    char buf[128] = {0};
    va_list ap;
    va_start(ap, fmt);
    memset(buf,0,sizeof(buf));
    vsprintf(buf, fmt, ap);
    va_end(ap);
    GKI_UartSendString(PRINT_UART, buf);
}

void GKI_UartSendBytes(UART_Typedef* UARTx,uint8_t *p,uint16_t len)
{
    uint16_t i;
    for(i=0;i<len;i++)
    {
        GKI_UartSendData(UARTx, *(p+i));
    }
}

//*********************************************************************************************************
// Description: Initalize the UART interrupt option
// Arguments  : UARTx:select UART port
//              NewState: new state of the uart interrupt.
//                       This parameter can be: ENABLE or DISABLE.
// Returns    : none
//*********************************************************************************************************
void GKI_UartITCmd(UART_Typedef* UARTx, uint16_t UART_IT, FunctionalState NewState)
{
    UART_ITCmd(UARTx, UART_IT,NewState);
}

//*********************************************************************************************************
// Description: set UART RTS output low level
// Arguments  : UARTx:select UART port
//
// Returns    : none
//*********************************************************************************************************
void GKI_UartSetRTSLow(UART_Typedef* UARTx)
{
    UART_ModemCmd(UARTx, UART_RTS_LOW,ENABLE);// RTS output low
}

//*********************************************************************************************************
// Description: set UART RTS output high level
// Arguments  : UARTx:select UART port
//
// Returns    : none
//*********************************************************************************************************
void GKI_UartSetRTSHigh(UART_Typedef* UARTx)
{
    UART_ModemCmd(UARTx, UART_RTS_LOW,DISABLE);// RTS output high
}

