/***************************************************************************************************
 *                               Generic Kernel Interface
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     GKI_spi.c
 * @brief    GKI function interface for SPI.
 * @version  V1.0.0
 * @date     05-Nov-2020
 ***************************************************************************************************/

#include "ach118x.h"
#include "ach118x_spi.h"
#include "ach118x_sys.h"
#include "GKI_spi.h"


// FIFO Threshold
#define SPI_THRESHOLD       (SPI_FIFO_DEPTH/2)


//*******************************************************************************
// Description: Send 16bit data via SPIx directly.
// Arguments  : SPIx    Where x can be (0, 1 or 2) to specified SPI peripheral.
//              Data    The 16bit data.
// Returns    : None.
//*******************************************************************************
void GKI_SpiSendHalfWord(SPI_Typedef* SPIx, uint16_t Data)
{
    SPI_SendData(SPIx, Data);
}

//*******************************************************************************
// Description: Receive 16bit data via SPIx directly.
// Arguments  : SPIx    Where x can be (0, 1 or 2) to specified SPI peripheral.
// Returns    : The received 16bit data.
//*******************************************************************************
uint16_t GKI_SpiGetHalfWord(SPI_Typedef* SPIx)
{
    return (uint16_t)(SPI_GetData(SPIx));
}


//*******************************************************************************
// Description: Get SPIx status.
// Arguments  : SPIx    Where x can be (0, 1 or 2) to specified SPI peripheral.
//              Flag    The status flag option.
// Returns    : None.
//*******************************************************************************
FlagStatus GKI_SpiGetStatus(SPI_Typedef* SPIx, ST_Flag Flag)
{
    return SPI_GetStatusFlag(SPIx,Flag);
}

uint8_t rdDoneSem = 0;
//*******************************************************************************
// Description: Init SPI control information.
// Arguments  : info    The SPI control information object.
//              SPIx    Where x can be (0, 1 or 2) to specified SPI peripheral.
// Returns    : None.
//*******************************************************************************
void GKI_SpiCtrlInfoInit(SpiCtrlInfo_t *info, SPI_Typedef *SPIx)
{
    memset(info, 0, sizeof(SpiCtrlInfo_t));
    info->SPIx = SPIx;
    info->rdDoneSem = xSemaphoreCreateBinary();
    info->wrDoneSem = xSemaphoreCreateBinary();
    info->mutex = xSemaphoreCreateRecursiveMutex();
    rdDoneSem = 0;
}
//extern int getd(void);
//*******************************************************************************
// Description: Read/Write a 16bit data buffer via SPIx with interrupt.
// Arguments  : info    The SPI control information object.
//              ptr     The 16bit data buffer.
//              payload The 16bit data buffer depth.
// Returns    : None.
//*******************************************************************************
//#pragma optimize=none
void GKI_SpiRdWrHalfWordBufWithIntr(SpiCtrlInfo_t *info, uint16_t *ptr, uint32_t payload)
{
    //Clear FIFO. Needed for when last SPI msg did not read its fifo
    //volatile uint32_t tmp = 0;
    //while (!(info->SPIx->SR & ST_TFE_Flag));
    //while (info->SPIx->SR & ST_RFNE_Flag){
    //    tmp = info->SPIx->DR[0];
    //}

    //Mount data.
    info->payload = payload;
    info->wrPtr = ptr + payload;
    info->rcvPtr = ptr;
    info->sndPtr = ptr;
    //Set Rx/Tx thresholds and enable interrupts for Tx/Rx
    info->SPIx->RXFTLR = (payload < SPI0_FIFO_RXF_THD) ? (payload - 1) : (SPI0_FIFO_RXF_THD - 1);
    info->SPIx->TXFTLR = SPI0_FIFO_TX_DEPTH - SPI0_FIFO_TXE_THD;
//    if(getd() != 1)
        info->SPIx->IMR |= (IT_TXE_Flag | IT_RXF_Flag);
//    else
//        info->txeCallBack(info);

    //Wait for all data write done.
//    xSemaphoreTake(info->wrDoneSem, portMAX_DELAY);

    //Wait for all data read done.
#if 0
    xSemaphoreTake(info->rdDoneSem, portMAX_DELAY);
#else
    while(!rdDoneSem)
    {
        vTaskDelay(1);
    }
    rdDoneSem = 0;
#endif
//    if(getd() == 1)
//        printu("%s %d Take info->rdDoneSem\r\n", __func__, __LINE__);
}

//*******************************************************************************
// Description: Read/Write a 16bit data buffer via SPIx with interrupt.
// Arguments  : info    The SPI control information object.
//              ptr     The 16bit data buffer.
//              payload The 16bit data buffer depth.
// Returns    : None.
//*******************************************************************************
//#pragma optimize=none
void GKI_SpiRdWrByteBufWithIntr(SpiCtrlInfo_t *info, uint8_t *ptr, uint32_t payload)
{
    //Clear FIFO. Needed for when last SPI msg did not read its fifo
    //volatile uint32_t tmp = 0;
    //while (!(info->SPIx->SR & ST_TFE_Flag));
    //while (info->SPIx->SR & ST_RFNE_Flag){
    //    tmp = info->SPIx->DR[0];
    //}

    //Mount data.
    info->payload = payload;
    info->wrPtr = ptr + payload;
    info->rcvPtr = ptr;
    info->sndPtr = ptr;
    //Set Rx/Tx thresholds and enable interrupts for Tx/Rx
    info->SPIx->RXFTLR = (payload < SPI0_FIFO_RXF_THD) ? (payload - 1) : (SPI0_FIFO_RXF_THD - 1);
    info->SPIx->TXFTLR = SPI0_FIFO_TX_DEPTH - SPI0_FIFO_TXE_THD;
    info->SPIx->IMR |= (IT_TXE_Flag | IT_RXF_Flag);

    //Wait for all data write done.
    //xSemaphoreTake(info->wrDoneSem, portMAX_DELAY);

    //Wait for all data read done.
#if 0
    xSemaphoreTake(info->rdDoneSem, portMAX_DELAY);
#else
    while(!rdDoneSem)
    {
        vTaskDelay(1);
    }
    rdDoneSem = 0;
#endif
}

//*******************************************************************************
// Description: Read/Write a 16bit data buffer via SPIx with interrupt.
// Arguments  : info    The SPI control information object.
//              ptr     The 16bit data buffer.
//              payload The 16bit data buffer depth.
// Returns    : None.
//*******************************************************************************
void GKI_SpiWrHalfWordBufWithIntr(SpiCtrlInfo_t *info, uint16_t *ptr, uint32_t payload)
{
    //Clear FIFO. Needed for when last SPI msg did not read its fifo
    volatile uint32_t tmp = 0;
    while (!(info->SPIx->SR & ST_TFE_Flag));
    while (info->SPIx->SR & ST_RFNE_Flag){
        tmp = info->SPIx->DR[0];
    }

    //Mount data.
    info->payload = payload;
    info->wrPtr = ptr + payload;
    info->rcvPtr = ptr;
    info->sndPtr = ptr;

    if (payload < SPI_FIFO_DEPTH) {
        // no need to use interrupts here
        __disable_interrupt();
        while (payload--) {
            info->SPIx->DR[0] = *ptr++;
        }
        __enable_interrupt();
    }
    else {
        //Set Tx threshold and enable interrupts for Tx only
        info->SPIx->TXFTLR = SPI_THRESHOLD;
        info->SPIx->IMR |= IT_TXE_Flag;

        //Wait for all data write done.
#if 0
        xSemaphoreTake(info->wrDoneSem, portMAX_DELAY);
#else
        while(!rdDoneSem)
        {
            vTaskDelay(1);
        }
        rdDoneSem = 0;
#endif
    }
}

//*******************************************************************************
// Description: Register SPI TX empty interrupt handler function.
// Arguments  : info    The SPI control information object.
//              func    The TXE handler function.
// Returns    : None.
//*******************************************************************************
void GKI_SpiRegTxeHandler(SpiCtrlInfo_t *info, void (*func)(void *))
{
    info->txeCallBack = func;
}

//*******************************************************************************
// Description: Register SPI RX full interrupt handle function.
// Arguments  : info    The SPI control information object.
//              func    The RXF handler function.
// Returns    : None.
//*******************************************************************************
void GKI_SpiRegRxfHandler(SpiCtrlInfo_t *info, void (*func)(void *))
{
    info->rxfCallBack = func;
}

//*******************************************************************************
// Description: Set SPI TXE threshold.
// Arguments  : SPIx    Where x can be (0, 1 or 2) to specified SPI peripheral.
//              thr_lv  The TXE threshold level.
// Returns    : None.
//*******************************************************************************
void GKI_SpiSetTxFifoThrLevel(SPI_Typedef *SPIx, uint16_t thr_lv)
{
    SPIx->TXFTLR = thr_lv;
}

//*******************************************************************************
// Description: Set SPI RXF threshold.
// Arguments  : SPIx    Where x can be (0, 1 or 2) to specified SPI peripheral.
//              thr_lv  The RXF threshold level.
// Returns    : None.
//*******************************************************************************
void GKI_SpiSetRxFifoThrLevel(SPI_Typedef *SPIx, uint16_t thr_lv)
{
    SPIx->RXFTLR = thr_lv;
}

//*******************************************************************************
// Description: Get SPI divider.
// Arguments  : SPIx    Where x can be (0, 1 or 2) to specified SPI peripheral.
// Returns    : Divide number.
//*******************************************************************************

uint16_t GKI_SpiGetDivider(SPI_Typedef *SPIx)
{
    return SPIx->BAUDR;
}


//*******************************************************************************
// Description: Send then receive 8bit datas via SPIx.
// Arguments  : SPIx      Where x can be (0, 1 or 2) to specified SPI peripheral.
//              pSndBuf   Send buffer.
//              SndLen    Send length.
//              pRcvBuf   Receive buffer.
//              RcvLen    Receive length.
//              ShiftIdx  Unuseful
// Returns    : None.
//*******************************************************************************
void GKI_SpiSendAndGetDatas(SPI_Typedef* SPIx, uint8_t *pSndBuf, uint32_t SndLen, uint8_t *pRcvBuf, uint32_t RcvLen, uint16_t ShiftIdx)
{

    if (taskSCHEDULER_NOT_STARTED == xTaskGetSchedulerState()) {
        portDISABLE_INTERRUPTS();
    } else {
        taskENTER_CRITICAL();
    }
    SPI_SendAndGetDatas(SPIx, pSndBuf, SndLen, pRcvBuf, RcvLen, ShiftIdx);
    if (taskSCHEDULER_NOT_STARTED == xTaskGetSchedulerState()) {
        portENABLE_INTERRUPTS();
    } else {
        taskEXIT_CRITICAL();
    }
}

/******************************************************************************
The following functions us used for ACC1340
******************************************************************************/
void GKI_ACC1340_SpiInit(SPI_Typedef* SPIx)
{
    SPI_InitTypeDef SPI_InitStruct;

    SPI_StructInit(&SPI_InitStruct);
    SPI_Cmd(SPIx, DISABLE);

    if(SPI0 == SPIx){
        SYS_SetClock(CLOCK_APB0_SPI0, ENABLE);
        SYS_SetITMaskFlag(IRQ_SPI0, SET);
        NVIC_EnableIRQ(SPI0_IRQn);
        NVIC_SetPriority(SPI0_IRQn, 1);
        SPI_InitStruct.DFS_32 = DF32_16_bit;
        SPI_InitStruct.FRF = FRF_Motorola;
        SPI_InitStruct.TMOD = Mode_Tx_and_Rx;
        SPI_InitStruct.SPI_FRF = SPI_FRF_Standard;
        SPI_InitStruct.SLV_OE = SLAVE_TXD_Disable;
        SPI_InitStruct.NumOfDataFrame = 22;
        SPI_InitStruct.ClockDivider = 8;//128;//4;
        SPI_InitStruct.RX_FIFO_ThresholdLevel = SPI0_FIFO_RXF_THD - 1;
        SPI_InitStruct.TX_FIFO_ThresholdLevel = SPI0_FIFO_TX_DEPTH - SPI0_FIFO_TXE_THD;
    } else if(SPI1 == SPIx){
        SYS_SetPinRemap0(PINMUX_PIN_13, REMAP0_PIN_13_Sel_SPI1_SSN);
        SYS_SetPinRemap0(PINMUX_PIN_14, REMAP0_PIN_14_Sel_SPI1_SCK);
        SYS_SetPinRemap0(PINMUX_PIN_15, REMAP0_PIN_15_Sel_SPI1_SDI);
        SYS_SetPinRemap0(PINMUX_PIN_16, REMAP0_PIN_16_Sel_SPI1_SDO);
        SYS_SetClock(CLOCK_APB0_SPI1, ENABLE);
        SYS_SetITMaskFlag(IRQ_SPI1, SET);
        NVIC_EnableIRQ(SPI1_IRQn);
        NVIC_SetPriority(SPI1_IRQn, 8);
        SPI_InitStruct.DFS_32 = DF32_16_bit;
        SPI_InitStruct.FRF = FRF_Motorola;
        SPI_InitStruct.TMOD = Mode_Tx_and_Rx;
        SPI_InitStruct.SPI_FRF = SPI_FRF_Standard;
        SPI_InitStruct.SLV_OE = SLAVE_TXD_Disable;
        SPI_InitStruct.NumOfDataFrame = 22;
        SPI_InitStruct.ClockDivider = 4;
        SPI_InitStruct.RX_FIFO_ThresholdLevel = 10;
        SPI_InitStruct.TX_FIFO_ThresholdLevel = 11;
    } else if(SPI2 == SPIx){
        SYS_SetClock(CLOCK_APB0_SPI2, ENABLE);
        SYS_SetITMaskFlag(IRQ_SPI2, SET);
        NVIC_EnableIRQ(SPI2_IRQn);
        NVIC_SetPriority(SPI2_IRQn, 8);
        SPI_InitStruct.DFS_32 = DF32_16_bit;
        SPI_InitStruct.FRF = FRF_Motorola;
        SPI_InitStruct.TMOD = Mode_Tx_and_Rx;
        SPI_InitStruct.SPI_FRF = SPI_FRF_Standard;
        SPI_InitStruct.SLV_OE = SLAVE_TXD_Disable;
        SPI_InitStruct.NumOfDataFrame = 22;
        SPI_InitStruct.ClockDivider = 4;
        SPI_InitStruct.RX_FIFO_ThresholdLevel = 10;
        SPI_InitStruct.TX_FIFO_ThresholdLevel = 11;
    }

    SPI_Init(SPIx, &SPI_InitStruct);

    SPI_ITCmd(SPIx, IT_ALL_Flag, DISABLE);
    SPI_Cmd(SPIx, ENABLE);
}

//*******************************************************************************
// Description: SPI TXE interrupt callback function.
// Arguments  : None.
// Returns    : None.
//*******************************************************************************
void GKI_ACC1340_SpiTx_CallBack(void *vInfo)
{
    //Calculate send information.
    SpiCtrlInfo_t *info = (SpiCtrlInfo_t*)vInfo;
    uint32_t snd_remain = (uint32_t)((uint16_t*)(info->wrPtr) - (uint16_t*)(info->sndPtr));
    uint32_t rcv_remain = (uint32_t)((uint16_t*)(info->wrPtr) - (uint16_t*)(info->rcvPtr));
    uint32_t snd_len = snd_remain < SPI0_FIFO_TXE_THD ? snd_remain : SPI0_FIFO_TXE_THD;
    uint32_t rcv_len = rcv_remain < SPI0_FIFO_RXF_THD ? rcv_remain : SPI0_FIFO_RXF_THD;

    //Set receive threshold.
    if(rcv_len < SPI0_FIFO_RXF_THD) {
        //GKI_SpiSetRxFifoThrLevel(info->SPIx, rcv_len - 1);
        info->SPIx->RXFTLR = rcv_len - 1;
    }
//    if(getd() == 1)
//        printu("%s %d snd_remain = %d snd_len = %d info = %p\r\n", __func__, __LINE__, snd_remain, snd_len, info);
    //Write TX FIFO.
    if (snd_remain < SPI0_FIFO_TXE_THD) {
        int my_snd_len = snd_len;
        uint16_t *mysndPtr = (uint16_t*)(info->sndPtr);
#if 0        
         while(my_snd_len--) {
            printu("%s %d  0x%04x \r\n", __func__, __LINE__, *(uint16_t*)(info->sndPtr));
            mysndPtr = mysndPtr + 1;
        }
#endif
        while(snd_len--) {
//            if(getd() == 1)
//                printu("%s %d  0x%04x \r\n", __func__, __LINE__, *(uint16_t*)(info->sndPtr));
            info->SPIx->DR[0] = *(uint16_t*)(info->sndPtr);
            info->sndPtr = (uint16_t*)(info->sndPtr) + 1;
        }
//        if(getd() == 1)
//            printu("1 %s %d \r\n", __func__, __LINE__);
    }
    else {
        info->SPIx->DR[0] = *(uint16_t*)(info->sndPtr); //1
        info->sndPtr = (uint16_t*)(info->sndPtr) + 1;
        info->SPIx->DR[0] = *(uint16_t*)(info->sndPtr); //2
        info->sndPtr = (uint16_t*)(info->sndPtr) + 1;
        info->SPIx->DR[0] = *(uint16_t*)(info->sndPtr); //3
        info->sndPtr = (uint16_t*)(info->sndPtr) + 1;
        info->SPIx->DR[0] = *(uint16_t*)(info->sndPtr); //4
        info->sndPtr = (uint16_t*)(info->sndPtr) + 1;
        info->SPIx->DR[0] = *(uint16_t*)(info->sndPtr); //5
        info->sndPtr = (uint16_t*)(info->sndPtr) + 1;
        info->SPIx->DR[0] = *(uint16_t*)(info->sndPtr); //6
        info->sndPtr = (uint16_t*)(info->sndPtr) + 1;
//        if(getd() == 1)
//            printu("2 %s %d \r\n", __func__, __LINE__);
    }
//    if(getd() == 1)
//        printu("3 %s %d \r\n", __func__, __LINE__);
    //Give send done semaphore and disable TXE interrupt to stop sending.
    if(info->sndPtr >= info->wrPtr) {
        //SPI_ITCmd(info->SPIx, IT_TXE_Flag, DISABLE);
        info->SPIx->IMR &= ~IT_TXE_Flag;
    }
//    if(getd() == 1)
//        printu("4 %s %d \r\n", __func__, __LINE__);

}
//*******************************************************************************
// Description: SPI RXF interrupt callback function.
// Arguments  : None.
// Returns    : None.
//*******************************************************************************
void GKI_ACC1340_SpiRx_CallBack(void *vInfo)
{
    //Calculate receive information.
    SpiCtrlInfo_t *info = (SpiCtrlInfo_t*)vInfo;
    uint32_t rcv_remain = (uint32_t)((uint16_t*)(info->wrPtr) - (uint16_t*)(info->rcvPtr));
    uint32_t rcv_len = rcv_remain < SPI0_FIFO_RXF_THD ? rcv_remain : SPI0_FIFO_RXF_THD;

    //Read RX FIFO.
    if (rcv_remain < SPI0_FIFO_RXF_THD) {
        while(rcv_len--) {
            *(uint16_t*)(info->rcvPtr) = info->SPIx->DR[0];
            info->rcvPtr = (uint16_t*)(info->rcvPtr) + 1;
        }
    }
    else {
        *(uint16_t*)(info->rcvPtr) = info->SPIx->DR[0]; //1
        info->rcvPtr = (uint16_t*)(info->rcvPtr) + 1;
        *(uint16_t*)(info->rcvPtr) = info->SPIx->DR[0]; //2
        info->rcvPtr = (uint16_t*)(info->rcvPtr) + 1;
        *(uint16_t*)(info->rcvPtr) = info->SPIx->DR[0]; //3
        info->rcvPtr = (uint16_t*)(info->rcvPtr) + 1;
        *(uint16_t*)(info->rcvPtr) = info->SPIx->DR[0]; //4
        info->rcvPtr = (uint16_t*)(info->rcvPtr) + 1;
        *(uint16_t*)(info->rcvPtr) = info->SPIx->DR[0]; //5
        info->rcvPtr = (uint16_t*)(info->rcvPtr) + 1;
        *(uint16_t*)(info->rcvPtr) = info->SPIx->DR[0]; //6
        info->rcvPtr = (uint16_t*)(info->rcvPtr) + 1;
    }

    //Give receive done semaphore and disable RXF interrupt to stop receiving.
    if(info->rcvPtr >= info->wrPtr) {
        //SPI_ITCmd(info->SPIx, IT_RXF_Flag, DISABLE);
        info->SPIx->IMR &= ~IT_RXF_Flag;
#if 0
        BaseType_t xHigherPriTaskWoken;
        xSemaphoreGiveFromISR(info->rdDoneSem, &xHigherPriTaskWoken);
#else
        rdDoneSem = 1;
#endif
//        getd();
//        if(getd() == 1)
//            printu("%s %d Give info->rdDoneSem\r\n", __func__, __LINE__);
    } else {
        //Set receive threshold.
        rcv_remain = (uint32_t)((uint16_t*)(info->wrPtr) - (uint16_t*)(info->rcvPtr));
        uint32_t rcv_len = rcv_remain < SPI0_FIFO_RXF_THD ? rcv_remain : SPI0_FIFO_RXF_THD;
        if(rcv_len < SPI0_FIFO_RXF_THD) {
            //GKI_SpiSetRxFifoThrLevel(info->SPIx, rcv_len - 1);
            info->SPIx->RXFTLR = rcv_len - 1;
        }
    }
}


/******************************************************************************
* The following functions us used for Flash
******************************************************************************/

//*******************************************************************************
// Description: Initializes the SPIx peripheral for master mode.
// Arguments  : SPIx    Where x can be (0, 1 or 2) to specified SPI peripheral.
// Returns    : None.
//*******************************************************************************
void GKI_FLASH_SpiInit(SPI_Typedef* SPIx)
{
    SPI_InitTypeDef SPI_InitStruct;

    if(SPI0 == SPIx){
        SYS_SetClock(CLOCK_APB0_SPI0, ENABLE);
    }
    else if(SPI1 == SPIx){
        SYS_SetClock(CLOCK_APB0_SPI1, ENABLE);
    }
    else if(SPI2 == SPIx){
        SYS_SetClock(CLOCK_APB0_SPI2, ENABLE);
    }

    if(SPIx == SPI1) {
        SYS_SetPinRemap0(PINMUX_PIN_13, REMAP0_PIN_13_Sel_SPI1_SSN);
        SYS_SetPinRemap0(PINMUX_PIN_14, REMAP0_PIN_14_Sel_SPI1_SCK);
        SYS_SetPinRemap0(PINMUX_PIN_15, REMAP0_PIN_15_Sel_SPI1_SDI);
        SYS_SetPinRemap0(PINMUX_PIN_16, REMAP0_PIN_16_Sel_SPI1_SDO);
    }

    SPI_Cmd(SPIx, DISABLE);
    SPI_StructInit(&SPI_InitStruct);
    SPI_InitStruct.DFS = DFS16_8_bit;//DFS_Control_8_bit;
    SPI_InitStruct.FRF = FRF_Motorola;
    SPI_InitStruct.TMOD = Mode_Tx_and_Rx;
    SPI_InitStruct.SPI_FRF = SPI_FRF_Standard;
    SPI_InitStruct.SLV_OE = SLAVE_TXD_Disable;//SLAVE_TXD_Enable;

    SPI_InitStruct.NumOfDataFrame = 0x30;
    if (SPIx == SPI2)
        SPI_InitStruct.ClockDivider = 8;   //Flash OK - (was 12)
    else
        SPI_InitStruct.ClockDivider = 8;//7 failed;//8 ok;//9 ok;//10 is ok;//32; //7 FactoryInit
    SPI_InitStruct.RX_FIFO_ThresholdLevel = 0xff;
    SPI_InitStruct.TX_FIFO_ThresholdLevel = 0xff;
    SPI_Init(SPIx, &SPI_InitStruct);
        SPI_ITCmd(SPIx, IT_ALL_Flag, DISABLE);
    SPI_Cmd(SPIx, ENABLE);
}

void GKI_SpiFlashRx_CallBack(void *vInfo)
{
    //Calculate receive information.
    SpiCtrlInfo_t *info = (SpiCtrlInfo_t *)vInfo;
    uint32_t rcv_remain, rcv_len, temp;
    uint8_t *w = info->rcvPtr;

    //Read RX FIFO.
    rcv_len = info->SPIx->RXFTLR+1;
    while (rcv_len--) {
        if (info->rcvSkp) {
            temp = info->SPIx->DR[0];
            info->rcvSkp--;
        }
        else if (info->rcvLen) {
            *w++ = info->SPIx->DR[0];
            info->rcvLen--;
        }
    }
    info->rcvPtr = w;

    //Give receive done semaphore and disable RXF interrupt to stop receiving.
    rcv_remain = info->rcvLen + info->rcvSkp;
    if(rcv_remain==0) {
        //Clear SPI interrupt mask bit
        info->SPIx->IMR &= ~IT_RXF_Flag;
        BaseType_t xHigherPriTaskWoken;
        xSemaphoreGiveFromISR(info->rdDoneSem, &xHigherPriTaskWoken);
    }
    else {
        //Set receive threshold.
        rcv_len = rcv_remain < SPI_THRESHOLD ? rcv_remain : SPI_THRESHOLD;
        if(rcv_len < SPI_THRESHOLD) {
            info->SPIx->RXFTLR = (rcv_len - 1);
        }
    }
}
//------------------------------------------------------------------------------
void GKI_SpiFlashTx_CallBack(void *vInfo)
{

    //Calculate send information.
    SpiCtrlInfo_t *info = (SpiCtrlInfo_t *)vInfo;
    uint32_t snd_remain, snd_len;
    uint8_t *w = info->sndPtr;

    snd_remain = info->sndLen + info->sndSkp;
    snd_len = MIN(snd_remain, SPI_THRESHOLD);
    while (snd_len--) {
        if (info->sndLen) {
            info->SPIx->DR[0] = *w++;
            info->sndLen--;
        }
        else {
            info->SPIx->DR[0] = 0;
            info->sndSkp--;
        }
    }
    info->sndPtr = w;

    //Clear SPI interrupt mask bit.
    snd_remain = info->sndLen + info->sndSkp;
    if(snd_remain==0) {
    info->SPIx->IMR &= ~IT_TXE_Flag;
        BaseType_t xHigherPriTaskWoken;
        xSemaphoreGiveFromISR(info->wrDoneSem, &xHigherPriTaskWoken);
    }
}

/******************************************************************************
The following functions us used for external spi interface
******************************************************************************/
#define SPI_EXT             (SPI1)

void GKI_SpiInitExtInt(void)
{
    SPI_InitTypeDef SPI_InitStruct;
    SPI_StructInit(&SPI_InitStruct);
    SPI_Cmd(SPI_EXT, DISABLE);

    SYS_SetPinRemap0(PINMUX_PIN_13, REMAP0_PIN_13_Sel_SPI1_SSN);
    SYS_SetPinRemap0(PINMUX_PIN_14, REMAP0_PIN_14_Sel_SPI1_SCK);
    SYS_SetPinRemap0(PINMUX_PIN_15, REMAP0_PIN_15_Sel_SPI1_SDI);
    SYS_SetPinRemap0(PINMUX_PIN_16, REMAP0_PIN_16_Sel_SPI1_SDO);
    SYS_SetClock(CLOCK_APB0_SPI1, ENABLE);
    SYS_SetITMaskFlag(IRQ_SPI1, SET);
    NVIC_EnableIRQ(SPI1_IRQn);
    NVIC_SetPriority(SPI1_IRQn, 1);

    SPI_InitStruct.DFS_32 = DF32_8_bit;//DF32_16_bit;
    SPI_InitStruct.FRF = FRF_Motorola;
    SPI_InitStruct.TMOD = Mode_Tx_and_Rx;
    SPI_InitStruct.SPI_FRF = SPI_FRF_Standard;
    SPI_InitStruct.SLV_OE = SLAVE_TXD_Disable;
    SPI_InitStruct.NumOfDataFrame = 22;
    SPI_InitStruct.ClockDivider = 64;//4;
    SPI_InitStruct.RX_FIFO_ThresholdLevel = 10;
    SPI_InitStruct.TX_FIFO_ThresholdLevel = 11;

    SPI_Init(SPI_EXT, &SPI_InitStruct);
    SPI_ITCmd(SPI_EXT, IT_ALL_Flag, DISABLE);
    SPI_Cmd(SPI_EXT, ENABLE);
}
void GKI_SpiWRByteBufWithIntr(SpiCtrlInfo_t *info, uint8_t *ptr, uint32_t payload)
{
    //Clear FIFO. Needed for when last SPI msg did not read its fifo
    //volatile uint32_t tmp = 0;
    //while (!(info->SPIx->SR & ST_TFE_Flag));
    //while (info->SPIx->SR & ST_RFNE_Flag){
    //    tmp = info->SPIx->DR[0];
    //}

    //Mount data.
    info->payload = payload;
    info->wrPtr = ptr + payload;
    info->rcvPtr = ptr;
    info->sndPtr = ptr;
    //Set Rx/Tx thresholds and enable interrupts for Tx/Rx
    info->SPIx->RXFTLR = (payload < SPI0_FIFO_RXF_THD) ? (payload - 1) : (SPI0_FIFO_RXF_THD - 1);
    info->SPIx->TXFTLR = SPI0_FIFO_TX_DEPTH - SPI0_FIFO_TXE_THD;
    info->SPIx->IMR |= (IT_TXE_Flag | IT_RXF_Flag);

    //Wait for all data write done.
    //xSemaphoreTake(info->wrDoneSem, portMAX_DELAY);

    //Wait for all data read done.
    xSemaphoreTake(info->rdDoneSem, portMAX_DELAY);
}

void GKI_SpiTxeExt_CallBack(void *vInfo)
{
    //Calculate send information.
    SpiCtrlInfo_t *info = (SpiCtrlInfo_t*)vInfo;
    uint32_t snd_remain = (uint32_t)((uint8_t*)(info->wrPtr) - (uint8_t*)(info->sndPtr));
    uint32_t rcv_remain = (uint32_t)((uint8_t*)(info->wrPtr) - (uint8_t*)(info->rcvPtr));
    uint32_t snd_len = snd_remain < SPI0_FIFO_TXE_THD ? snd_remain : SPI0_FIFO_TXE_THD;
    uint32_t rcv_len = rcv_remain < SPI0_FIFO_RXF_THD ? rcv_remain : SPI0_FIFO_RXF_THD;

    //Set receive threshold.
    if(rcv_len < SPI0_FIFO_RXF_THD) {
        //GKI_SpiSetRxFifoThrLevel(info->SPIx, rcv_len - 1);
        info->SPIx->RXFTLR = rcv_len - 1;
    }
    //Write TX FIFO.
    if (snd_remain < SPI0_FIFO_TXE_THD) {
        while(snd_len--) {
            info->SPIx->DR[0] = *(uint8_t*)(info->sndPtr);
            info->sndPtr = (uint8_t*)(info->sndPtr) + 1;
        }
    }
    else {
        info->SPIx->DR[0] = *(uint8_t*)(info->sndPtr); //1
        info->sndPtr = (uint8_t*)(info->sndPtr) + 1;
        info->SPIx->DR[0] = *(uint8_t*)(info->sndPtr); //2
        info->sndPtr = (uint8_t*)(info->sndPtr) + 1;
        info->SPIx->DR[0] = *(uint8_t*)(info->sndPtr); //3
        info->sndPtr = (uint8_t*)(info->sndPtr) + 1;
        info->SPIx->DR[0] = *(uint8_t*)(info->sndPtr); //4
        info->sndPtr = (uint8_t*)(info->sndPtr) + 1;
        info->SPIx->DR[0] = *(uint8_t*)(info->sndPtr); //5
        info->sndPtr = (uint8_t*)(info->sndPtr) + 1;
        info->SPIx->DR[0] = *(uint8_t*)(info->sndPtr); //6
        info->sndPtr = (uint8_t*)(info->sndPtr) + 1;
    }


    //Give send done semaphore and disable TXE interrupt to stop sending.
    if(info->sndPtr >= info->wrPtr) {
        //SPI_ITCmd(info->SPIx, IT_TXE_Flag, DISABLE);
        info->SPIx->IMR &= ~IT_TXE_Flag;
    }
}
void GKI_SpiRxfExt_CallBack(void *vInfo)
{
    //Calculate receive information.
    SpiCtrlInfo_t *info = (SpiCtrlInfo_t*)vInfo;
    uint32_t rcv_remain = (uint32_t)((uint8_t*)(info->wrPtr) - (uint8_t*)(info->rcvPtr));
    uint32_t rcv_len = rcv_remain < SPI0_FIFO_RXF_THD ? rcv_remain : SPI0_FIFO_RXF_THD;

    //Read RX FIFO.
    if (rcv_remain < SPI0_FIFO_RXF_THD) {
        while(rcv_len--) {
            *(uint8_t*)(info->rcvPtr) = info->SPIx->DR[0];
            info->rcvPtr = (uint8_t*)(info->rcvPtr) + 1;
        }
    }
    else {
        *(uint8_t*)(info->rcvPtr) = info->SPIx->DR[0]; //1
        info->rcvPtr = (uint8_t*)(info->rcvPtr) + 1;
        *(uint8_t*)(info->rcvPtr) = info->SPIx->DR[0]; //2
        info->rcvPtr = (uint8_t*)(info->rcvPtr) + 1;
        *(uint8_t*)(info->rcvPtr) = info->SPIx->DR[0]; //3
        info->rcvPtr = (uint8_t*)(info->rcvPtr) + 1;
        *(uint8_t*)(info->rcvPtr) = info->SPIx->DR[0]; //4
        info->rcvPtr = (uint8_t*)(info->rcvPtr) + 1;
        *(uint8_t*)(info->rcvPtr) = info->SPIx->DR[0]; //5
        info->rcvPtr = (uint8_t*)(info->rcvPtr) + 1;
        *(uint8_t*)(info->rcvPtr) = info->SPIx->DR[0]; //6
        info->rcvPtr = (uint8_t*)(info->rcvPtr) + 1;
    }
    //Give receive done semaphore and disable RXF interrupt to stop receiving.
    if(info->rcvPtr >= info->wrPtr) {
        //SPI_ITCmd(info->SPIx, IT_RXF_Flag, DISABLE);
        info->SPIx->IMR &= ~IT_RXF_Flag;
        BaseType_t xHigherPriTaskWoken;
        xSemaphoreGiveFromISR(info->rdDoneSem, &xHigherPriTaskWoken);
    } else {
        //Set receive threshold.
        rcv_remain = (uint32_t)((uint8_t*)(info->wrPtr) - (uint8_t*)(info->rcvPtr));
        uint32_t rcv_len = rcv_remain < SPI0_FIFO_RXF_THD ? rcv_remain : SPI0_FIFO_RXF_THD;
        if(rcv_len < SPI0_FIFO_RXF_THD) {
            //GKI_SpiSetRxFifoThrLevel(info->SPIx, rcv_len - 1);
            info->SPIx->RXFTLR = rcv_len - 1;
        }
    }
}


/*****************************END OF FILE****/

