/***************************************************************************************************
 *                               Generic Kernel Interface
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     GKI_Queue.c
 * @brief    GKI queuing utilities for data.
 * @version  V1.0.0
 * @date     05-Nov-2020
 ***************************************************************************************************/

#include "ach118x.h"
#include "FreeRTOS.h"
#include "task.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "GKI_queue.h"


// Define Functions
#define QFreeSpace(q)    (q->Size - q->Count)
#define QBuf(q,pnt)      ((uint8_t *)q->pBuf + (pnt * q->DataSize))
#define QFull(q)         (q->Count > 0)
#define QReady(q)        (q->Count < q->Size)


#define ENTER_QUEUE_CRITICAL()    vTaskSuspendAll()
#define EXIT_QUEUE_CRITICAL()     xTaskResumeAll()
#ifndef MIN
#define MIN(a,b) ((a) < (b) ? (a) : (b))
#endif



/*******************************************************************************
* Function Name  : GKI_QGet
* Description    : Get next data unit from the queue
*******************************************************************************/
void GKI_QGet(tQUEUE *q, void *data)  // Read a data
{
    ENTER_QUEUE_CRITICAL();
    if (QFull(q))
    {
        memcpy(data, QBuf(q, q->Tail), q->DataSize);
        if (++q->Tail >= q->Size) {
            q->Tail = 0;
        }
        q->Count--;
    }
    EXIT_QUEUE_CRITICAL();
}


/*******************************************************************************
* Function Name  : GKI_QGetBuf
* Description    : Get data buffer from the queue
*
* Params
*   q - queue structure
*   pDestBuf - Destination buffer
*   destLength - size of destination buffer
*
* Return
*   Number of data units transferred to destination buffer
*******************************************************************************/
extern bool bMacrxQ_initialized;
uint32_t GKI_QGetBuf(tQUEUE *q, void *pDestBuf, uint32_t destLength)  // Read destLength data
{
    uint32_t xferCount;
    uint32_t len;

    // Determine data size to transfer
    destLength = MIN(destLength, q->Count);

    // Loop once, or twice if the data wraps in the queue
    for (xferCount=0; xferCount<destLength; xferCount += len)
    {
        len = MIN((q->Size - q->Tail), (destLength - xferCount));
        if (pDestBuf!=NULL) {
            memcpy(pDestBuf, QBuf(q, q->Tail), len*q->DataSize);
            pDestBuf = (uint8 *)pDestBuf + len * q->DataSize;
//            printf("%s %d pDestBuf = %p step = %d\r\n", __func__, __LINE__, pDestBuf, len * q->DataSize);
//            if((uint8 *)pDestBuf > (uint8 *)&bMacrxQ_initialized){
//                printf("GKI_QGetBuf: pDestBuf is out of range\n");
//                while(1);
//            }
        }

        q->Tail  += len;
        if (q->Tail >= q->Size) {
            q->Tail = 0;
        }
        q->Count -= len;
    }

    return destLength;
}


/*******************************************************************************
* Function Name  : GKI_QPut
* Description    : Put data unit into the queue
*******************************************************************************/
void GKI_QPut(tQUEUE *q, void *data) // Store a data
{
    ENTER_QUEUE_CRITICAL();
    if (QReady(q))
    {
        memcpy(QBuf(q, q->Head), data, q->DataSize);
        if (++q->Head >= q->Size) {
            q->Head = 0;
        }
        q->Count++;
    }
    EXIT_QUEUE_CRITICAL();
}


/*******************************************************************************
* Function Name  : GKI_QPut
* Description    : Put data unit into the queue
*******************************************************************************/
void GKI_QPutISR(tQUEUE *q, void *data) // Store a data
{
    if (QReady(q))
    {
        memcpy(QBuf(q, q->Head), data, q->DataSize);
        if (++q->Head >= q->Size) {
            q->Head = 0;
        }
        q->Count++;
    }
}


/*******************************************************************************
* Function Name  : GKI_QPutBuf
* Description    : Put data buffer into the queue
*
* Params
*   q - queue structure
*   pSrcBuf - Source buffer
*   srcLength - size of source buffer
*
* Return
*   Number of data units transferred from source buffer
*******************************************************************************/
uint32_t GKI_QPutBuf(tQUEUE *q, void *pSrcBuf, uint32_t srcLength) // Store srcLength data
{
    uint32_t xferCount;
    uint32_t len;

    // Determine data size to transfer
    srcLength = MIN(srcLength, QFreeSpace(q));

    // Loop once, or twice if the data wraps in the queue
    for (xferCount=0; xferCount<srcLength; xferCount += len) {
        len = MIN((q->Size - q->Head), (srcLength - xferCount));
        memcpy(QBuf(q, q->Head), pSrcBuf, len*q->DataSize);
        pSrcBuf = (uint8 *)pSrcBuf + len*q->DataSize;
        q->Head  += len;
        if (q->Head >= q->Size) {
            q->Head = 0;
        }
        q->Count += len;
    }

    return srcLength;
}


/*******************************************************************************
* Function Name  : GKI_QInit
* Description    : Intialize the queue
*******************************************************************************/
void GKI_QInit(tQUEUE *q, void *pBuf, uint32_t DataSize, uint32_t Size)
{ // DataSize is the data width, and Size is the maximum number of data
    if (pBuf==NULL) {
        if (q->pBuf != NULL)
            free(q->pBuf);
        pBuf = malloc(Size*DataSize);
        if (pBuf==NULL)
            return;
    }
    memset(q, 0, sizeof(tQUEUE));
    q->Size = Size;
    q->DataSize = DataSize;
    q->pBuf = pBuf;
    memset(pBuf, 0xff, Size*DataSize);
}


/*******************************************************************************
* Function Name  : GKI_QPeekBuf
* Description    : Get data buffer without popping the queue
*
* Params
*   q - queue structure
*   pDestBuf - Destination buffer
*   destLength - size of destination buffer
*
* Return
*   Number of data units transferred to destination buffer
*******************************************************************************/
uint32_t  GKI_QPeekBuf(tQUEUE *q, void *pDestBuf, uint32_t destLength)
{
    uint32_t xferCount;
    uint32_t len;
    uint32_t qCount = q->Count;
    uint32_t qTail  = q->Tail;

    // Determine data size to transfer
    destLength = MIN(destLength, qCount);

    // Loop once, or twice if the data wraps in the queue
    for (xferCount=0; xferCount<destLength; xferCount += len) {
        len = MIN((q->Size - qTail), (destLength - xferCount) );
        memcpy(pDestBuf, QBuf(q, q->Tail), len*q->DataSize);
        pDestBuf = (uint8 *)pDestBuf + len*q->DataSize;
        qTail  += len;
        if (qTail >= q->Size) {
            qTail = 0;
        }
        qCount -= len;
    }

    return destLength;
}


/*******************************************************************************
* Function Name  : GKI_QPeek
* Description    : Get next data unit without popping the queue
*******************************************************************************/
void GKI_QPeek(tQUEUE *q, void *data)
{
    if (QFull(q)) {
        memcpy(data, QBuf(q, q->Tail), q->DataSize);
    }
}


/*******************************************************************************
* Function Name  : GKI_QFreeSpace
* Description    : Return the amout of free space available in queue
*******************************************************************************/
uint32_t GKI_QFreeSpace(tQUEUE *q)
{   // The remaining data space in the queue, there is room for data
    return (q->Size - q->Count);
}


/*******************************************************************************
* Function Name  : GKI_QCount
* Description    : Return the count of object in the queue
*******************************************************************************/
uint32_t  GKI_QCount(tQUEUE *q) // The number of data in the queue, if empty can not read the data
{
    return q->Count;
}


/*******************************************************************************
* Function Name  : GKI_QNotReady
* Description    : Get next data unit without popping the queue
*******************************************************************************/
bool GKI_QNotReady(tQUEUE *q) // If the team is full, it can not put data
{
    //return (q->Count >= q->Size);
    if(q->Count >= q->Size){
        return TRUE;
    }else{
        return FALSE;
    }
}



/*******************************************************************************
* Function Name  : GKI_QBuf
* Description    : Get address of next data
*******************************************************************************/
void * GKI_QDataAddr(tQUEUE *q)
{
    return (QBuf(q, q->Tail));
}


//------------------------------------------------------------------------------
// End of GKI_Queue.c
//------------------------------------------------------------------------------

