/***************************************************************************************************
 *                               Generic Kernel Interface
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     GKI_i2c.c
 * @brief    GKI function interface for I2C.
 * @version  V1.0.0
 * @date     18-Nov-2020
 ***************************************************************************************************/

#include <stdint.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <yvals.h>
#include "ach118x_sys.h"
#include "ach118x_i2c.h"

#include "ach118x_sys.h"
#include "GKI_sys.h"
//#include "core_cm4.h"
#define E2PROM_ADDR     (0x34)//(0xAE)
#define OWN_ADDR        (0xA0)

//*********************************************************************
// Description: Initialize i2c device as master
// Arguments  : I2Cx:  select UART port
//              ClockSpeed: i2c SCL clock speed,uint:hz
// Returns    : none
//*********************************************************************
void GKI_I2CMasterInit(I2C_Typedef* I2Cx, uint32_t ClockSpeed)
{

    I2C_InitTypeDef I2C_InitStruct;
    NVIC_EnableIRQ(I2C_IRQn);
    SYS_SetITMaskFlag(IRQ_I2C0,SET);
    I2C_Cmd(I2Cx,DISABLE);

    //I2C_InitStruct.I2C_Clock      = GKI_GetSystemClock()/2;
    I2C_InitStruct.I2C_ClockSpeed = ClockSpeed;
    I2C_InitStruct.I2C_Role       = I2C_MASTER;
    I2C_InitStruct.I2C_AddrType   = I2C_ADDR_7BIT;
    I2C_InitStruct.I2C_AckAddress = E2PROM_ADDR;
    //I2C_InitStruct.I2C_OwnAddress = OWN_ADDR;

    I2C_Init(I2Cx,&I2C_InitStruct);

    I2C_Acknowledge(I2Cx,ENABLE);
    I2C_ITConfig(I2Cx,
                 I2C_TX_OVER | I2C_RX_OVER | I2C_RD_REQ | I2C_RX_DONE | I2C_TX_ABRT |
                 I2C_ACTIVITY| I2C_STOP_DET | I2C_START_DET| I2C_GEN_CALL| I2C_RESTART_DET,
                 DISABLE);
    I2C_ITConfig(I2Cx, I2C_RX_UNDER,DISABLE);
    I2C_ITConfig(I2Cx, I2C_TX_EMPTY,DISABLE);
    I2C_ITConfig(I2Cx, I2C_RX_FULL,DISABLE);

    I2C_Cmd(I2Cx,ENABLE);
}

//*********************************************************************
// Description: Initialize i2c device as slave
// Arguments  : I2Cx:  select UART port
//
// Returns    : none
//*********************************************************************
void GKI_I2CSlaveInit(I2C_Typedef* I2Cx)
{
    I2C_InitTypeDef I2C_InitStruct;

    NVIC_EnableIRQ(I2C_IRQn);
    SYS_SetITMaskFlag(IRQ_I2C0,SET);

    I2C_Cmd(I2Cx,DISABLE);

    I2C_InitStruct.I2C_Role       = I2C_SLAVE;
    I2C_InitStruct.I2C_AddrType   = I2C_ADDR_7BIT;
    //I2C_InitStruct.I2C_AckAddress = E2PROM_ADDR;
    I2C_InitStruct.I2C_OwnAddress = OWN_ADDR;
    I2C_Init(I2Cx,&I2C_InitStruct);
    I2C_Acknowledge(I2Cx,ENABLE);
    I2C_ITConfig(I2Cx,
                 I2C_TX_OVER | I2C_RX_OVER | I2C_RD_REQ | I2C_RX_DONE | I2C_TX_ABRT |
                 I2C_ACTIVITY| I2C_STOP_DET | I2C_START_DET| I2C_GEN_CALL| I2C_RESTART_DET,
                 DISABLE);
    I2C_ITConfig(I2Cx, I2C_RX_UNDER,DISABLE);
    I2C_ITConfig(I2Cx, I2C_TX_EMPTY,DISABLE);
    I2C_ITConfig(I2Cx, I2C_RX_FULL,DISABLE);

    I2C_Cmd(I2Cx,ENABLE);
}

//*********************************************************************
// Description:  send one byte via i2c interface
// Arguments  : I2Cx    select I2C port
//              Data    the data be send via i2c
// Returns    : none
//*********************************************************************
void GKI_I2CSendData(I2C_Typedef* I2Cx, uint8_t Data)
{
    I2C_SendData(I2Cx,Data);
    while(I2C_GetTranFIFOLevel(I2Cx));
}

//*********************************************************************
// Description:  read one byte from i2c interface
// Arguments  : I2Cx    select I2C port
//
// Returns    :the data be read from i2c
//*********************************************************************
uint8_t GKI_I2CReceiveData(I2C_Typedef* I2Cx)
{
    while(!I2C_GetReceFIFOLevel(I2Cx));
    return I2C_ReceiveData(I2Cx);
}

//*********************************************************************
// Description: Enable or Disable I2C interface
// Arguments  : I2Cx    select I2C port
//              NewState new state of the i2c interface.
//                       This parameter can be: ENABLE or DISABLE.
// Returns    :none
//*********************************************************************
void GKI_I2CCmd(I2C_Typedef* I2Cx, FunctionalState NewState)
{
    I2C_Cmd(I2Cx,NewState);
}

//*********************************************************************
// Description: initalize the i2c interrupt
// Arguments  : I2Cx    select I2C port
//              I2C_IT  i2c interrupt option,the option should be:
//                      RX_UNDER    RX_OVER     RX_FULL
//                      TX_OVER     TX_EMPTY    RD_REQ
//                      TX_ABRT     RX_DONE     ACTIVITY
//                      STOP_DET    START_DET   GEN_CALL
//                      RESTART_DET MST_ON_HOLD
//              NewState new state of the i2c interrupt.
//                       This parameter can be: ENABLE or DISABLE.
// Returns    :none
//*********************************************************************
void GKI_I2CITConfig(I2C_Typedef* I2Cx, uint16_t I2C_IT, FunctionalState NewState)
{
    I2C_ITConfig(I2Cx,I2C_IT,NewState);
}

//********************************************************************************************
// Description: controls the level of entries (or below) that trigger the TX_EMPTY interrupt
// Arguments  : I2Cx    select I2C port
//              I2C_Level   transmit fifo trigger level
//
// Returns    :none
//********************************************************************************************
void GKI_I2CTranFIFOLevelConfig(I2C_Typedef* I2Cx, uint8_t I2C_Level)
{
    I2C_TranFIFOLevelConfig(I2Cx,I2C_Level);
}

//********************************************************************************************
// Description: controls the level of entries (or above) that triggers the RX_FULL interrupt
// Arguments  : I2Cx        select I2C port
//              I2C_Level   receive fifo trigger level
//
// Returns    :none
//********************************************************************************************
void GKI_I2CReceFIFOLevelConfig(I2C_Typedef* I2Cx, uint8_t I2C_Level)
{
    I2C_ReceFIFOLevelConfig(I2Cx,I2C_Level);
}

/************************ (C) COPYRIGHT ART *****END OF FILE****/

