/***************************************************************************************************
 *                               Generic Kernel Interface
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     GKI_adc.c
 * @brief    GKI routines for ADC.
 * @version  V1.0.0
 * @date     05-Nov-2020
 ***************************************************************************************************/

#include "ach118x_adc.h"
#include "ach118x_sys.h"
#include "GKI_adc.h"

//*********************************************************************************************************
// Description: Initializes the ADCx peripheral
//*********************************************************************************************************
void GKI_AdcInit(ADC_Typedef* ADCx)
{
    if(ADC0 == ADCx)
    {
        SYS_SetITMaskFlag(IRQ_ADC0, SET);
        NVIC_EnableIRQ(ADC0_IRQn);
    }
    else if(ADC1 == ADCx)
    {
        SYS_SetITMaskFlag(IRQ_ADC1, SET);
        NVIC_EnableIRQ(ADC1_IRQn);
    }
    ADC_PowerCmd(ADCx, POWER_On);
    ADC_SetClockDiv(ADCx, 6);
    ADC_ITCmd(ADCx, ENABLE);
}

//*********************************************************************************************************
// Description: Disable ADC.
// Arguments  : ADCx    Where x can be (0 or 1) to select the ADCx peripheral.
// Returns    : None.
//*********************************************************************************************************
void GKI_AdcDisable(ADC_Typedef* ADCx)
{
    if(ADC0 == ADCx) {
        SYS_SetITMaskFlag(IRQ_ADC0, SET);
        NVIC_DisableIRQ(ADC0_IRQn);
    } else if(ADC1 == ADCx) {
        SYS_SetITMaskFlag(IRQ_ADC1, SET);
        NVIC_DisableIRQ(ADC1_IRQn);
    }
        //ADC_PowerCmd(ADCx,DISABLE);
    ADC_ITCmd(ADCx, DISABLE);
}

//*********************************************************************************************************
// Description: ADC conersion start.
// Arguments  : ADCx     Where x can be (0 or 1) to select the ADCx peripheral.
// Returns    : None.
//*********************************************************************************************************
void GKI_AdcStartConversion(ADC_Typedef* ADCx)
{
    ADC_StartConversion(ADCx);
}

//*********************************************************************************************************
// Description: Check ADC conversion status.
// Arguments  : ADCx     Where x can be (0 or 1) to select the ADCx peripheral.
// Returns    : This value can be one of @FlagStatus enum:
//              @arg SET:   Specified ADC conversion done.
//              @arg RESET: Specified ADC conversion doesn't done.
//*********************************************************************************************************
FlagStatus GKI_AdcGetConversionStatus(ADC_Typedef* ADCx)
{
    return ADC_GetConversionStatus(ADCx);
}

//*********************************************************************************************************
// Description: Read ADC conversion result.
// Arguments  : ADCx     Where x can be (0 or 1) to select the ADCx peripheral.
// Returns    : This value is ADC conversion result, which only low 10bits available.
//*********************************************************************************************************
uint16_t GKI_AdcGetConversionResult(ADC_Typedef* ADCx)
{
    return ADC_GetConversionResult(ADCx);
}

//*********************************************************************************************************
// Description: ADC interrupt control.
// Arguments  : ADCx    Where x can be (0 or 1) to select the ADCx peripheral.
//              status  This parameter can be one of @FunctionalState enum:
//                      @arg ENABLE:  ADC interrupt enable.
//                      @arg DISABLE: ADC interrupt disable.
// Returns    : None
//*********************************************************************************************************
void GKI_AdcITCmd(ADC_Typedef* ADCx, FunctionalState status)
{
    ADC_ITCmd(ADCx, status);
}

//*********************************************************************************************************
// Description: ADC interrupt status.
// Arguments  : ADCx    Where x can be (0 or 1) to select the ADCx peripheral.
// Returns    : This value can be one of @ITStatus enum:
//              @arg SET:   Interrupt trigged.
//              @arg RESET: No interrupt trigged.
//*********************************************************************************************************
FlagStatus GKI_AdcGetITFlag(ADC_Typedef* ADCx)
{
    return ADC_GetITFlag(ADCx);
}

//*********************************************************************************************************
// Description: ADC interrupt clear.
// Arguments  : ADCx    Where x can be (0 or 1) to select the ADCx peripheral.
// Returns    : None.
//*********************************************************************************************************
void GKI_AdcClearITFlag(ADC_Typedef* ADCx)
{
    ADC_ClearITFlag(ADCx);
}

//*********************************************************************************************************
// Description: ADC power down control.
// Arguments  : ADCx    Where x can be (0 or 1) to select the ADCx peripheral.
//              status  This parameter can be one of @FunctionalState enum.
//                      @arg ENABLE:  ADC power down enable.
//                      @arg DISABLE: ADC power down disable.
// Returns    : None.
//*********************************************************************************************************
void GKI_AdcPowerCmd(ADC_Typedef* ADCx, PowerCmd status)
{
    ADC_PowerCmd(ADCx, status);
}

/************************ (C) COPYRIGHT ART *****END OF FILE****/

