/***************************************************************************************************
 *                               Generic Kernel Interface
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     GKI_ExtSpi.c
 * @brief    GKI functions for EXTI.
 * @version  V1.0.0
 * @date     05-Nov-2020
 ***************************************************************************************************/

#include <stdio.h>
#include <string.h>
#include "ach118x_sys.h"
#include "ach118x_spi.h"
#include "ach118x_timer.h"
#include "ach118x_it.h"
#include "GKI_spi.h"
#include "ach118x_map.h"
#include "FreeRTOSConfig.h"
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"
#include "portmacro.h"

#ifndef MIN
#define MIN(a,b) ((a) < (b) ? (a) : (b))
#endif

#define SPI_FIFO_DEPTH      22  //FIFO depth
#define SPI_THRESHOLD       11  //FIFO threshold
#define SPI_EXT             (SPI1)



void GKI_SpiInitExtInt(void)
{
    SPI_InitTypeDef SPI_InitStruct;
    SPI_StructInit(&SPI_InitStruct);
    SPI_Cmd(SPI_EXT, DISABLE);

    SYS_SetPinRemap0(PINMUX_PIN_13, REMAP0_PIN_13_Sel_SPI1_SSN);
    SYS_SetPinRemap0(PINMUX_PIN_14, REMAP0_PIN_14_Sel_SPI1_SCK);
    SYS_SetPinRemap0(PINMUX_PIN_15, REMAP0_PIN_15_Sel_SPI1_SDI);
    SYS_SetPinRemap0(PINMUX_PIN_16, REMAP0_PIN_16_Sel_SPI1_SDO);
    SYS_SetClock(CLOCK_APB0_SPI1, ENABLE);
    SYS_SetITMaskFlag(IRQ_SPI1, SET);
    NVIC_EnableIRQ(SPI1_IRQn);
    NVIC_SetPriority(SPI1_IRQn, 1);

    SPI_InitStruct.DFS_32 = DF32_8_bit;//DF32_16_bit;
    SPI_InitStruct.FRF = FRF_Motorola;
    SPI_InitStruct.TMOD = Mode_Tx_and_Rx;
    SPI_InitStruct.SPI_FRF = SPI_FRF_Standard;
    SPI_InitStruct.SLV_OE = SLAVE_TXD_Disable;
    SPI_InitStruct.NumOfDataFrame = 22;
    SPI_InitStruct.ClockDivider = 64;//4;
    SPI_InitStruct.RX_FIFO_ThresholdLevel = 10;
    SPI_InitStruct.TX_FIFO_ThresholdLevel = 11;

    SPI_Init(SPI_EXT, &SPI_InitStruct);
    SPI_ITCmd(SPI_EXT, IT_ALL_Flag, DISABLE);
    SPI_Cmd(SPI_EXT, ENABLE);
}

/*
void GKI_SpiSndRcv_ExtInt(uint8_t *pTx,uint8_t *pRx,unsigned short len )
{
    unsigned int i;
    for(i=0;i<len;i++){
        while(GKI_SpiGetStatus(SPI_EXT,ST_TFE_Flag) == RESET);
        SPI_SendData(SPI_EXT, *pTx++);
        while(GKI_SpiGetStatus(SPI_EXT,ST_RFNE_Flag) == RESET);
        *pRx++ = SPI_ReciveData(SPI_EXT);
    }
}
*/
void GKI_SpiWRByteBufWithIntr(SpiCtrlInfo_t *info, uint8_t *ptr, uint32_t payload)
{
    //Clear FIFO. Needed for when last SPI msg did not read its fifo
    //volatile uint32_t tmp = 0;
    //while (!(info->SPIx->SR & ST_TFE_Flag));
    //while (info->SPIx->SR & ST_RFNE_Flag){
    //  tmp = info->SPIx->DR[0];
    //}

    //Mount data.
    info->payload = payload;
    info->wrPtr = ptr + payload;
    info->rcvPtr = ptr;
    info->sndPtr = ptr;
    //Set Rx/Tx thresholds and enable interrupts for Tx/Rx
    info->SPIx->RXFTLR = (payload < SPI0_FIFO_RXF_THD) ? (payload - 1) : (SPI0_FIFO_RXF_THD - 1);
    info->SPIx->TXFTLR = SPI0_FIFO_TX_DEPTH - SPI0_FIFO_TXE_THD;
    info->SPIx->IMR |= (IT_TXE_Flag | IT_RXF_Flag);

    //Wait for all data write done.
    //xSemaphoreTake(info->wrDoneSem, portMAX_DELAY);

    //Wait for all data read done.
    xSemaphoreTake(info->rdDoneSem, portMAX_DELAY);
}

void GKI_SpiTxeExt_CallBack(void *vInfo)
{
    //Calculate send information.
    SpiCtrlInfo_t *info = (SpiCtrlInfo_t*)vInfo;
    uint32_t snd_remain = (uint32_t)((uint8_t*)(info->wrPtr) - (uint8_t*)(info->sndPtr));
    uint32_t rcv_remain = (uint32_t)((uint8_t*)(info->wrPtr) - (uint8_t*)(info->rcvPtr));
    uint32_t snd_len = snd_remain < SPI0_FIFO_TXE_THD ? snd_remain : SPI0_FIFO_TXE_THD;
    uint32_t rcv_len = rcv_remain < SPI0_FIFO_RXF_THD ? rcv_remain : SPI0_FIFO_RXF_THD;

    //Set receive threshold.
    if(rcv_len < SPI0_FIFO_RXF_THD) {
        //GKI_SpiSetRxFifoThrLevel(info->SPIx, rcv_len - 1);
        info->SPIx->RXFTLR = rcv_len - 1;
    }
    //Write TX FIFO.
    if (snd_remain < SPI0_FIFO_TXE_THD) {
        while(snd_len--) {
            info->SPIx->DR[0] = *(uint8_t*)(info->sndPtr);
            info->sndPtr = (uint8_t*)(info->sndPtr) + 1;
        }
    }
    else {
        info->SPIx->DR[0] = *(uint8_t*)(info->sndPtr); //1
        info->sndPtr = (uint8_t*)(info->sndPtr) + 1;
        info->SPIx->DR[0] = *(uint8_t*)(info->sndPtr); //2
        info->sndPtr = (uint8_t*)(info->sndPtr) + 1;
        info->SPIx->DR[0] = *(uint8_t*)(info->sndPtr); //3
        info->sndPtr = (uint8_t*)(info->sndPtr) + 1;
        info->SPIx->DR[0] = *(uint8_t*)(info->sndPtr); //4
        info->sndPtr = (uint8_t*)(info->sndPtr) + 1;
        info->SPIx->DR[0] = *(uint8_t*)(info->sndPtr); //5
        info->sndPtr = (uint8_t*)(info->sndPtr) + 1;
        info->SPIx->DR[0] = *(uint8_t*)(info->sndPtr); //6
        info->sndPtr = (uint8_t*)(info->sndPtr) + 1;
    }


    //Give send done semaphore and disable TXE interrupt to stop sending.
    if(info->sndPtr >= info->wrPtr) {
        //SPI_ITCmd(info->SPIx, IT_TXE_Flag, DISABLE);
        info->SPIx->IMR &= ~IT_TXE_Flag;
    }
}
void GKI_SpiRxfExt_CallBack(void *vInfo)
{
    //Calculate receive information.
    SpiCtrlInfo_t *info = (SpiCtrlInfo_t*)vInfo;
    uint32_t rcv_remain = (uint32_t)((uint8_t*)(info->wrPtr) - (uint8_t*)(info->rcvPtr));
    uint32_t rcv_len = rcv_remain < SPI0_FIFO_RXF_THD ? rcv_remain : SPI0_FIFO_RXF_THD;

    //Read RX FIFO.
    if (rcv_remain < SPI0_FIFO_RXF_THD) {
        while(rcv_len--) {
            *(uint8_t*)(info->rcvPtr) = info->SPIx->DR[0];
            info->rcvPtr = (uint8_t*)(info->rcvPtr) + 1;
        }
    }
    else {
        *(uint8_t*)(info->rcvPtr) = info->SPIx->DR[0]; //1
        info->rcvPtr = (uint8_t*)(info->rcvPtr) + 1;
        *(uint8_t*)(info->rcvPtr) = info->SPIx->DR[0]; //2
        info->rcvPtr = (uint8_t*)(info->rcvPtr) + 1;
        *(uint8_t*)(info->rcvPtr) = info->SPIx->DR[0]; //3
        info->rcvPtr = (uint8_t*)(info->rcvPtr) + 1;
        *(uint8_t*)(info->rcvPtr) = info->SPIx->DR[0]; //4
        info->rcvPtr = (uint8_t*)(info->rcvPtr) + 1;
        *(uint8_t*)(info->rcvPtr) = info->SPIx->DR[0]; //5
        info->rcvPtr = (uint8_t*)(info->rcvPtr) + 1;
        *(uint8_t*)(info->rcvPtr) = info->SPIx->DR[0]; //6
        info->rcvPtr = (uint8_t*)(info->rcvPtr) + 1;
    }
    //Give receive done semaphore and disable RXF interrupt to stop receiving.
    if(info->rcvPtr >= info->wrPtr) {
        //SPI_ITCmd(info->SPIx, IT_RXF_Flag, DISABLE);
        info->SPIx->IMR &= ~IT_RXF_Flag;
        BaseType_t xHigherPriTaskWoken;
        xSemaphoreGiveFromISR(info->rdDoneSem, &xHigherPriTaskWoken);
    } else {
        //Set receive threshold.
        rcv_remain = (uint32_t)((uint8_t*)(info->wrPtr) - (uint8_t*)(info->rcvPtr));
        uint32_t rcv_len = rcv_remain < SPI0_FIFO_RXF_THD ? rcv_remain : SPI0_FIFO_RXF_THD;
        if(rcv_len < SPI0_FIFO_RXF_THD) {
            //GKI_SpiSetRxFifoThrLevel(info->SPIx, rcv_len - 1);
            info->SPIx->RXFTLR = rcv_len - 1;
        }
    }
}

