/***************************************************************************************************
 *                               Generic Kernel Interface
 *                      COPYRIGHT(c) 2020 Amped RF Technology (ART)
 *                                 All Rights Reserved
 *
 * @file     GKI_wm8904.h
 * @brief    GKI structures and prototypes for WM8904.
 * @version  V1.0.0
 * @date     05-Nov-2020
 ***************************************************************************************************/
#ifndef _GKI_WM8904_H
#define _GKI_WM8904_H

#include "ach118x_type.h"
#include "ach118x_i2c.h"
#include "wm8904.h"

#ifdef __cplusplus
    extern "C" {
#endif

//PLL1 register0.
typedef union _pll1CtrlReg0_t {
    uint32_t regVal;
    struct {
        uint32_t refDiv1    : 6;
        uint32_t fbDiv1     : 12;
        uint32_t postDiv1_1 : 3;
        uint32_t postDiv2_1 : 3;
        uint32_t divVal_1   : 4;
        uint32_t reserved   : 4;
    } bits;
} pll1CtrlReg0_t;

//PLL1 register1.
typedef union _pll1CtrlReg1_t {
    uint32_t regVal;
    struct {
        uint32_t frac_1          : 24;
        uint32_t pd_1            : 1;
        uint32_t dacPd_1         : 1;
        uint32_t dsmPd_1         : 1;
        uint32_t foutPostDivPd_1 : 1;
        uint32_t fout4PhasePd_1  : 1;
        uint32_t foutVcoPd_1     : 1;
        uint32_t bypass_1        : 1;
        uint32_t reserved        : 1;
    } bits;
} pll1CtrlReg1_t;

//PLL1 spread spectrum register.
typedef union _pll1SpreadSpectrumReg_t {
    uint32_t regVal;
    struct {
        uint32_t spread_1    : 5;
        uint32_t downSpread  : 1;
        uint32_t selExtWave  : 1;
        uint32_t extWaveVal  : 8;
        uint32_t extMaxAddr  : 8;
        uint32_t disableSscg : 1;
        uint32_t reserved0   : 2;
        uint32_t wdtClkEn    : 1;
        uint32_t reserved1   : 5;
    } bits;
} pll1SpreadSpectrumReg_t;

//System clock control register1.
typedef union _sysClkCtrlReg1_t {
    uint32_t regVal;
    struct {
        uint32_t reserved0 : 21;
        uint32_t mclkCodec : 1;
        uint32_t reserved1 : 10;
    } bits;
} sysClkCtrlReg1_t;

//I2C control register.
typedef union _I2cCtrlReg_t{
    uint32_t regVal;
    struct {
        uint32_t masterMode           : 1;
        uint32_t speed                : 2;
        uint32_t slave10bitsAddr      : 1;
        uint32_t master10bitsAddr     : 1;
        uint32_t restartEn            : 1;
        uint32_t slaveDisable         : 1;
        uint32_t stopDetIfSlaveAddred : 1;
        uint32_t txEmptyCtrl          : 1;
        uint32_t rxFifoFullHldCtrl    : 1;
        uint32_t stopDetIfMasterAct   : 1;
        uint32_t reserved             : 21;
    } bits;
} I2cCtrlReg_t;

//I2C target address register
typedef union _I2cTarReg_t {
    uint32_t regVal;
    struct {
        uint32_t tar              : 10;
        uint32_t gcOrStart        : 1;
        uint32_t special          : 1;
        uint32_t master10bitsAddr : 1;
        uint32_t reserved         : 19;
    } bits;
} I2cTarReg_t;

//I2C data cmd register.
typedef union _I2cDataCmdReg_t {
    uint32_t regVal;
    struct {
        uint32_t data     : 8; //Read or Write data.
        uint32_t cmd      : 1; //Read or Write command.
        uint32_t stop     : 1; //Control whether a STOP is issued after the byte is sent or received.
        uint32_t restart  : 1; //Control whether a RESTART is issued before the byte is sent or received.
        uint32_t byte1st  : 1; //Indicate the first byte received after the address phase.
        uint32_t reserved : 20;
    } bits;
} I2cDataCmdReg_t;

//I2C scl high or low count.
typedef union _I2cSclCntReg_t {
    uint32_t regVal;
    struct {
        uint32_t cnt : 16;
        uint32_t reserved : 16;
    } bits;
} I2cSclCntReg_t;

//I2C interrupt status register.
typedef union _I2cIntrReg_t {
    uint32_t regVal;
    struct {
        uint32_t rxUnder    : 1;
        uint32_t rxOver     : 1;
        uint32_t rxFull     : 1;
        uint32_t txOver     : 1;
        uint32_t txEmpty    : 1;
        uint32_t rdReq      : 1;
        uint32_t txAbrt     : 1;
        uint32_t rxDone     : 1;
        uint32_t activity   : 1;
        uint32_t stopDet    : 1;
        uint32_t startDet   : 1;
        uint32_t genCall    : 1;
        uint32_t restartDet : 1;
        uint32_t mstOnHld   : 1;
    } bits;
} I2cIntrReg_t;

//I2C FIFO threshold register.
typedef union _I2cFifoThdReg_t {
    uint32_t regVal;
    struct {
        uint32_t thd      : 8;
        uint32_t reserved : 24;
    } bits;
} I2cFifoThdReg_t;

//I2C interrupt flag clear register.
typedef union _I2cIntrClrReg_t {
    uint32_t regVal;
    struct {
        uint32_t intrFlg  : 1;
        uint32_t reserved : 31;
    } bits;
} I2cIntrClrReg_t;

//I2C enable register.
typedef union _I2cEnableReg_t {
    uint32_t regVal;
    struct {
        uint32_t enable     : 1;
        uint32_t abort      : 1;
        uint32_t txCmdBlock : 1;
        uint32_t reserved   : 29;
    } bits;
} I2cEnableReg_t;

//I2C status register.
typedef union _I2cStatusReg_t {
    uint32_t regVal;
    struct {
        uint32_t activity          : 1;
        uint32_t tfnf              : 1;
        uint32_t tfe               : 1;
        uint32_t rfne              : 1;
        uint32_t rff               : 1;
        uint32_t mstActivity       : 1;
        uint32_t slvActivity       : 1;
        uint32_t mstHldTxFifoEmpty : 1;
        uint32_t mstHldRxFifoFull  : 1;
        uint32_t slvHldTxFifoEmpty : 1;
        uint32_t slvHldRxFifoFull  : 1;
        uint32_t reserved          : 21;
    } bits;
} I2cStatusReg_t;

//I2C FIFO level register.
typedef union _I2cFifoLvReg_t {
    uint32_t regVal;
    struct {
        uint32_t lv       : 8;
        uint32_t reserved : 24;
    } bits;
} I2cFifoLvReg_t;

//I2C Tx abort source register.
typedef union _I2cTxAbrtSrcReg_t {
    uint32_t regVal;
    struct {
        uint32_t noAck7bitsAddr   : 1;
        uint32_t noAck10bitsAddr1 : 1;
        uint32_t noAck10bitsAddr2 : 1;
        uint32_t noAckTxData      : 1;
        uint32_t noAckGcall       : 1;
        uint32_t rdGcall          : 1;
        uint32_t ackDetHs         : 1;
        uint32_t ackDetSbyte      : 1;
        uint32_t noRstrtHs        : 1;
        uint32_t noRstrtSbyte     : 1;
        uint32_t noRstrt10bitsRd  : 1;
        uint32_t mstDis           : 1;
        uint32_t arbLost          : 1;
        uint32_t slvFlushTxFifo   : 1;
        uint32_t slvArbLost       : 1;
        uint32_t slvRdIntx        : 1;
        uint32_t usrAbrt          : 1;
        uint32_t reserved         : 6;
        uint32_t txFlushCnt       : 1;
    } bits;
} I2cTxAbrtSrcReg_t;

//I2C acknowledge general call register.
typedef union _I2cAckGcallReg_t {
    uint32_t regVal;
    struct {
        uint32_t ackGcall : 1;
        uint32_t reserved : 31;
    } bits;
} I2cAckGcallReg_t;


//APIs
void GKI_I2cInitForWm8904(void);
void GKI_I2sInitForWm8904(void);
void GKI_I2cTxData(uint8_t slvAddr, uint8_t addr, uint8_t* data, uint32_t payload);
void GKI_I2cRxData(uint8_t slvAddr, uint8_t addr, uint8_t* data, uint32_t payload);
void GKI_wm8904Init(void);


#ifdef __cplusplus
    }//extern "C"
#endif

#endif // _GKI_WM8904_H

/************************ (C) COPYRIGHT ART *****END OF FILE****/

