/*
 * FreeRTOS Kernel V10.0.1
 * Copyright (C) 2017 Amazon.com, Inc. or its affiliates.  All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * http://www.FreeRTOS.org
 * http://aws.amazon.com/freertos
 *
 * 1 tab == 4 spaces!
 */

/*
 * A sample implementation of pvPortMalloc() and vPortFree() that combines
 * (coalescences) adjacent memory blocks as they are freed, and in so doing
 * limits memory fragmentation.
 *
 * See heap_1.c, heap_2.c and heap_3.c for alternative implementations, and the
 * memory management pages of http://www.FreeRTOS.org for more information.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Defining MPU_WRAPPERS_INCLUDED_FROM_API_FILE prevents task.h from redefining
all the API functions to use the MPU wrappers.  That should only be done when
task.h is included from an application file. */
#define MPU_WRAPPERS_INCLUDED_FROM_API_FILE

#include "FreeRTOS.h"
#include "task.h"
//extern uint32_t __TopOfMemory;
uint32_t __TopOfMemory = 0x200EFFFF;//MEM_TOP//__section_end("HEAP");
#define DEBUG_HEAP
#ifdef DEBUG_HEAP
#define DEB_HEAP_MAX 32
#define DEB_HEAP_MIN 0
static void DebMemMalloc(void *pMem, size_t size);
static void DebMemFree(void *pMem);
#else
#define DebMemMalloc(x,sz)
#define DebMemFree(x)
#endif



#undef MPU_WRAPPERS_INCLUDED_FROM_API_FILE

#if( configSUPPORT_DYNAMIC_ALLOCATION == 0 )
	#error This file must not be used if configSUPPORT_DYNAMIC_ALLOCATION is 0
#endif

/* Block sizes must not get too small. */
#define heapMINIMUM_BLOCK_SIZE	( ( size_t ) ( xHeapStructSize << 1 ) )

/* Assumes 8bit bytes! */
#define heapBITS_PER_BYTE		( ( size_t ) 8 )

/* Allocate the memory for the heap. */
#if( configAPPLICATION_ALLOCATED_HEAP == 1 )
	/* The application writer has already defined the array used for the RTOS
	heap - probably so it can be placed in a special segment or address. */
	extern uint8_t ucHeap[ configTOTAL_HEAP_SIZE ];
#else
	static uint8_t ucHeap[ configTOTAL_HEAP_SIZE ];
#endif /* configAPPLICATION_ALLOCATED_HEAP */

/* Define the linked list structure.  This is used to link free blocks in order
of their memory address. */
typedef struct A_BLOCK_LINK
{
	struct A_BLOCK_LINK *pxNextFreeBlock;	/*<< The next free block in the list. */
	size_t xBlockSize;						/*<< The size of the free block. */
} BlockLink_t;

/*-----------------------------------------------------------*/

/*
 * Inserts a block of memory that is being freed into the correct position in
 * the list of free memory blocks.  The block being freed will be merged with
 * the block in front it and/or the block behind it if the memory blocks are
 * adjacent to each other.
 */
static void prvInsertBlockIntoFreeList( BlockLink_t *pxBlockToInsert );

/*
 * Called automatically to setup the required heap structures the first time
 * pvPortMalloc() is called.
 */
static void prvHeapInit( void );

/* Include first 8k and remaining memory for heap */
static void vInitFirst8k( void );
static void vInitRemainingMemory(void);
static void vAddToHeap(void *BegAddr, void *EndAddr);

/*-----------------------------------------------------------*/

/* The size of the structure placed at the beginning of each allocated memory
block must by correctly byte aligned. */
static const size_t xHeapStructSize	= ( sizeof( BlockLink_t ) + ( ( size_t ) ( portBYTE_ALIGNMENT - 1 ) ) ) & ~( ( size_t ) portBYTE_ALIGNMENT_MASK );

/* Create a couple of list links to mark the start and end of the list. */
static BlockLink_t xStart, *pxEnd = NULL;

/* Keeps track of the number of free bytes remaining, but says nothing about
fragmentation. */
static size_t xFreeBytesRemaining = 0U;
static size_t xMinimumEverFreeBytesRemaining = 0U;

/* Gets set to the top bit of an size_t type.  When this bit in the xBlockSize
member of an BlockLink_t structure is set then the block belongs to the
application.  When the bit is free the block is still part of the free heap
space. */
static size_t xBlockAllocatedBit = 0;

#if 0
typedef struct {
    void *TaskHandler;
    int heapblock;
	size_t heapblock_size;
} Taskinfo;
typedef struct {
    void *addr;
    size_t size;
    Taskinfo *Taskinfo;
	void *caller;
    int line;
} MemAllocEntry;
#define LIST_NUM 500
Taskinfo Task_info_list[20] = {0};
static MemAllocEntry alloc_list[LIST_NUM] = {0}; // 存储分配记录
static uint16_t alloc_list_num = 0;
//#define MAX_TRACE
#if 0
#define GET_FP() __get_R11()  // 若 IAR 使用 R7 作为帧指针
void log_call_chain() {
    void **fp = (void **)GET_FP();
    TaskHandle_t current_task = xTaskGetCurrentTaskHandle();
    const char *task_name = pcTaskGetName(current_task);

    printf("[Task: %s] Call Chain:\n", task_name);
    
    for (int i = 0; i < MAX_CALLER_DEPTH; i++) {
        if (fp == NULL) break;
        void *lr = *(fp + 1);  // 假设 LR 在 FP + 4 的位置
        printf("  Level %d: 0x%p\n", i, lr);
        fp = (void **)(*fp);   // 移动到父帧指针
    }
}
#endif
// 获取内存块实际分配大小
size_t get_allocated_size(void *ptr) {
    BlockLink_t *pxBlock = (BlockLink_t *)((uint8_t *)ptr - sizeof(BlockLink_t));
    size_t n = pxBlock->xBlockSize & 0x7fffffff - sizeof(BlockLink_t);
//    printf("%p n = %d\r\n", ptr, n);
    return n;
}
uint16_t find_alloc_list_index(void *addr, uint8_t is_add)
{
	for(int i = 0; i < 500; i++)
	{
		if(is_add == 1 && alloc_list[i].addr == NULL)
		{
//			printf("%s %d i = %d\r\n", __func__, __LINE__, i);
			return i;
		}
		if(is_add == 0 && alloc_list[i].addr == addr)
		{
//			printf("%s %d i = %d\r\n", __func__, __LINE__, i);
			return i;
		}
	}
	return 0xffff;
}
uint16_t find_Task_list_index(void *vTaskHandler){
	for(int i = 0; i < 20; i++)
	{
		if(Task_info_list[i].TaskHandler == vTaskHandler)
		{
//			printf("%s %d i = %d\r\n", __func__, __LINE__, i);
			return i;
		}
	}
	for(int i = 0; i < 20; i++)
	{
		if(Task_info_list[i].TaskHandler == NULL)
		{
//			printf("%s %d i = %d\r\n", __func__, __LINE__, i);
			Task_info_list[i].TaskHandler = vTaskHandler;
			return i;
		}
	}
	return 0xffff;
}
extern xTaskHandle xcw1200_mac_workqueue_task;
extern xTaskHandle xIdleTaskHndler;
void add_memorylist(void *allc_addr, uint16_t xWantedSize, void *vTaskHandler, void *caller, uint16_t line){
	uint16_t idx = 0, Tid = 0;
	if ((void *)xTaskGetSchedulerState() == xcw1200_mac_workqueue_task
	|| ( void * ) xTaskGetCurrentTaskHandle() == xIdleTaskHndler)
		return;
	if(vTaskHandler == NULL)
		return;
#ifdef MAX_TRACE
	idx = find_alloc_list_index(allc_addr, 1);
	if(idx == 0xffff){
		memset(&alloc_list, 0, sizeof(alloc_list));
		idx = 0;
	}
	Tid = find_Task_list_index(vTaskHandler);
	if(Tid == 0xffff)
		while(1);

	alloc_list[idx].addr = allc_addr;
	alloc_list[idx].size = xWantedSize;
	alloc_list[idx].Taskinfo = &Task_info_list[Tid];
	alloc_list[idx].Taskinfo->heapblock++;
	alloc_list[idx].Taskinfo->heapblock_size += xWantedSize;
	alloc_list[idx].caller = caller;
	alloc_list[idx].line = line;

	alloc_list_num++;
#else
	Tid = find_Task_list_index(vTaskHandler);
	if(Tid == 0xffff)
		while(1);
	Task_info_list[Tid].heapblock++;
//	printf("%s %d heapblock = %d allc_addr = %p\r\n", __func__, __LINE__, Task_info_list[Tid].heapblock, allc_addr);
	Task_info_list[Tid].heapblock_size += xWantedSize;
#endif
}
void remove_memorylist(void *free_addr){
	if (xTaskGetSchedulerState() == taskSCHEDULER_NOT_STARTED)
		return;
	if(( void * ) xTaskGetCurrentTaskHandle() == xcw1200_mac_workqueue_task 
	|| ( void * ) xTaskGetCurrentTaskHandle() == xIdleTaskHndler)
		return;
#ifdef MAX_TRACE
	uint16_t idx = 0;
	idx = find_alloc_list_index(free_addr, 0);
	if(idx == 0xffff){
		goto LEAVE;
		while(1);
	}
	
	alloc_list[idx].Taskinfo->heapblock--;
	alloc_list[idx].Taskinfo->heapblock_size -= get_allocated_size(free_addr);
	memset(&alloc_list[idx], 0, sizeof(alloc_list[idx]));
LEAVE:
	alloc_list_num--;
#else
	uint16_t Tid = 0;
	Tid = find_Task_list_index(( void * ) xTaskGetCurrentTaskHandle());
	if(Tid == 0xffff)
		while(1);
	Task_info_list[Tid].heapblock--;
//	printf("%s %d heapblock = %d allc_addr = %p\r\n", __func__, __LINE__, Task_info_list[Tid].heapblock, free_addr);
	if(Task_info_list[Tid].heapblock < 0)
		printf("%s %d oops\r\n", __func__, __LINE__);
	Task_info_list[Tid].heapblock_size -= get_allocated_size(free_addr);
#endif
}

void update_memorylist(void *allc_addr, void *vTaskHandler, int caller, uint16_t line){
#ifdef MAX_TRACE
	uint16_t idx = 0;
	idx = find_alloc_list_index(allc_addr, 0);
	if(idx == 0xffff)
		while(1);
//	alloc_list[idx].Taskinfo = vTaskHandler;
//	printf("%s %d caller = %d\r\n", __func__, __LINE__, caller);
	alloc_list[idx].caller = (void *)caller;
	alloc_list[idx].line = line;
#endif
}
#else
#ifdef MEM_TRACK
// 全局变量
static MemRecord record_buffer[MAX_RECORDS];    // 循环缓冲区
static UBaseType_t record_index = 0;            // 当前写入位置
static UBaseType_t valid_unfreed = 0;           // 当前有效未释放记录数
static UBaseType_t peak_unfreed = 0;            // 历史未释放记录峰值
static bool buffer_full = 0;                // 缓冲区是否已满
static bool memory_exhausted = 0;           // 内存是否已耗尽

// 哈希表（用于快速查找地址对应的记录索引）
static UBaseType_t hash_table[2048];

// 哈希函数（简单取模）
static inline UBaseType_t hash_func(void *addr) {
    return ((UBaseType_t)addr >> 2) & (2048 - 1);
}

// 插入哈希表
static void hash_insert(void *addr, UBaseType_t index) {
    UBaseType_t slot = hash_func(addr);
    hash_table[slot] = index;
}

// 查找哈希表
static UBaseType_t hash_find(void *addr) {
    UBaseType_t slot = hash_func(addr);
    return hash_table[slot];
}
// 获取当前未释放的内存地址列表
void get_unfreed_addresses(void **array, UBaseType_t max_size) {
    taskENTER_CRITICAL();
    UBaseType_t count = 0;
    for (UBaseType_t i = 0; i < MAX_RECORDS; i++) {
        if (record_buffer[i].is_valid && record_buffer[i].type == MEM_ALLOC) {
            if (count < max_size) {
//                array[count++] = record_buffer[i].address;
				printf("Addr: 0x%4p, Size: %4u, Handler: %4p\r\n",
                   record_buffer[i].address, record_buffer[i].size, record_buffer[i].taskHandle);
            } else {
                break;
            }
        }
    }
	printf("peak_unfreed: %d\r\n", peak_unfreed);
    taskEXIT_CRITICAL();
}
static void log_mem_op(MemOpType type, void *addr, size_t size) {
//    taskENTER_CRITICAL();

    //--- 处理释放操作 ---
    if (type == MEM_FREE) {
        UBaseType_t alloc_idx = hash_find(addr);
        if (alloc_idx < MAX_RECORDS && record_buffer[alloc_idx].is_valid) {
            record_buffer[alloc_idx].is_valid = 0;
            valid_unfreed--;
            hash_table[hash_func(addr)] = MAX_RECORDS; // 标记为无效

//			printf("free: %4d \r\n", alloc_idx);
        }
//        taskEXIT_CRITICAL();
        return;
    }

    //--- 处理分配操作 ---
    // 内存耗尽时不记录新分配（仅保留已有记录）
    if (memory_exhausted) {
//        taskEXIT_CRITICAL();
        return;
    }

    // 查找可覆盖位置
    UBaseType_t target_index = record_index;
    if (buffer_full) {
        // 策略：优先覆盖已释放记录，否则覆盖最旧记录
		UBaseType_t i;
        for (i = 0; i < MAX_RECORDS; i++) {
            UBaseType_t candidate = (record_index + i) % MAX_RECORDS;
            if (!record_buffer[candidate].is_valid) {
                target_index = candidate;
                break;
            }
        }
        // 输出缓冲区满提示（首次触发）
        if (i == 0) {
            printf("[WARN] Record buffer full, enabling overwrite mode.\n");
        }
    }

    // 覆盖旧记录前的清理
    if (record_buffer[target_index].is_valid) {
        if (record_buffer[target_index].type == MEM_ALLOC) {
            valid_unfreed--;
            hash_table[hash_func(record_buffer[target_index].address)] = MAX_RECORDS;
        }
    }

    // 写入新记录
    record_buffer[target_index].type = type;
    record_buffer[target_index].address = addr;
    record_buffer[target_index].size = size;
    record_buffer[target_index].timestamp = xTaskGetTickCount();
    record_buffer[target_index].taskHandle = xTaskGetCurrentTaskHandle();
    record_buffer[target_index].is_valid = 1;
//	printf("mall: %4d \r\n", target_index);
    // 更新哈希表
    hash_insert(addr, target_index);

    // 更新索引与计数器
    valid_unfreed++;
	buffer_full = (valid_unfreed >= MAX_RECORDS); // 直接根据计数器设置状态

    if (valid_unfreed > peak_unfreed) {
        peak_unfreed = valid_unfreed;
        printf("[PEAK] New unfreed record peak: %d\n", peak_unfreed);
    }

    record_index = (target_index + 1) % MAX_RECORDS;
#if 0
    if (!buffer_full && record_index == 0) {
        buffer_full = 1;
    }
#endif
//    taskEXIT_CRITICAL();
}
#else
void get_unfreed_addresses(void **array, UBaseType_t max_size) {
}
#endif
#endif
/*-----------------------------------------------------------*/
void *pvPortMalloc( size_t xWantedSize )
{
BlockLink_t *pxBlock, *pxPreviousBlock, *pxNewBlockLink;
void *pvReturn = NULL;

	vTaskSuspendAll();
	{
		/* If this is the first call to malloc then the heap will require
		initialisation to setup the list of free blocks. */
		if( pxEnd == NULL )
		{
			prvHeapInit();
		}
		else
		{
			mtCOVERAGE_TEST_MARKER();
		}

		/* Check the requested block size is not so large that the top bit is
		set.  The top bit of the block size member of the BlockLink_t structure
		is used to determine who owns the block - the application or the
		kernel, so it must be free. */
		if( ( xWantedSize & xBlockAllocatedBit ) == 0 )
		{
			/* The wanted size is increased so it can contain a BlockLink_t
			structure in addition to the requested amount of bytes. */
			if( xWantedSize > 0 )
			{
				xWantedSize += xHeapStructSize;

				/* Ensure that blocks are always aligned to the required number
				of bytes. */
				if( ( xWantedSize & portBYTE_ALIGNMENT_MASK ) != 0x00 )
				{
					/* Byte alignment required. */
					xWantedSize += ( portBYTE_ALIGNMENT - ( xWantedSize & portBYTE_ALIGNMENT_MASK ) );
					configASSERT( ( xWantedSize & portBYTE_ALIGNMENT_MASK ) == 0 );
				}
				else
				{
					mtCOVERAGE_TEST_MARKER();
				}
			}
			else
			{
				mtCOVERAGE_TEST_MARKER();
			}

			if( ( xWantedSize > 0 ) && ( xWantedSize <= xFreeBytesRemaining ) )
			{
				/* Traverse the list from the start	(lowest address) block until
				one	of adequate size is found. */
				pxPreviousBlock = &xStart;
				pxBlock = xStart.pxNextFreeBlock;
				while( ( pxBlock->xBlockSize < xWantedSize ) && ( pxBlock->pxNextFreeBlock != NULL ) )
				{
					pxPreviousBlock = pxBlock;
					pxBlock = pxBlock->pxNextFreeBlock;
				}

				/* If the end marker was reached then a block of adequate size
				was	not found. */
				if( pxBlock != pxEnd )
				{
					/* Return the memory space pointed to - jumping over the
					BlockLink_t structure at its start. */
					pvReturn = ( void * ) ( ( ( uint8_t * ) pxPreviousBlock->pxNextFreeBlock ) + xHeapStructSize );

					/* This block is being returned for use so must be taken out
					of the list of free blocks. */
					pxPreviousBlock->pxNextFreeBlock = pxBlock->pxNextFreeBlock;

					/* If the block is larger than required it can be split into
					two. */
					if( ( pxBlock->xBlockSize - xWantedSize ) > heapMINIMUM_BLOCK_SIZE )
					{
						/* This block is to be split into two.  Create a new
						block following the number of bytes requested. The void
						cast is used to prevent byte alignment warnings from the
						compiler. */
						pxNewBlockLink = ( void * ) ( ( ( uint8_t * ) pxBlock ) + xWantedSize );
						configASSERT( ( ( ( size_t ) pxNewBlockLink ) & portBYTE_ALIGNMENT_MASK ) == 0 );

						/* Calculate the sizes of two blocks split from the
						single block. */
						pxNewBlockLink->xBlockSize = pxBlock->xBlockSize - xWantedSize;
						pxBlock->xBlockSize = xWantedSize;

						/* Insert the new block into the list of free blocks. */
						prvInsertBlockIntoFreeList( pxNewBlockLink );
					}
					else
					{
						mtCOVERAGE_TEST_MARKER();
					}

					xFreeBytesRemaining -= pxBlock->xBlockSize;

					if( xFreeBytesRemaining < xMinimumEverFreeBytesRemaining )
					{
						xMinimumEverFreeBytesRemaining = xFreeBytesRemaining;
					}
					else
					{
						mtCOVERAGE_TEST_MARKER();
					}

					/* The block is being returned - it is allocated and owned
					by the application and has no "next" block. */
					pxBlock->xBlockSize |= xBlockAllocatedBit;
					pxBlock->pxNextFreeBlock = NULL;
				}
				else
				{
					mtCOVERAGE_TEST_MARKER();
				}
			}
			else
			{
				mtCOVERAGE_TEST_MARKER();
			}
		}
		else
		{
			mtCOVERAGE_TEST_MARKER();
		}

		traceMALLOC( pvReturn, xWantedSize );
	}
//	add_memorylist(pvReturn, xWantedSize, ( void * ) xTaskGetCurrentTaskHandle(), (void *)__get_LR(), __LINE__);
#ifdef MEM_TRACK
	log_mem_op(MEM_ALLOC, pvReturn, xWantedSize);
#endif
	( void ) xTaskResumeAll();

	#if( configUSE_MALLOC_FAILED_HOOK == 1 )
	{
		if( pvReturn == NULL )
		{
			extern void vApplicationMallocFailedHook( void );
			vApplicationMallocFailedHook();
		}
		else
		{
			mtCOVERAGE_TEST_MARKER();
		}
	}
	#endif

    // Debug - log allocated memory
    DebMemMalloc(pvReturn, xWantedSize);

	configASSERT( ( ( ( size_t ) pvReturn ) & ( size_t ) portBYTE_ALIGNMENT_MASK ) == 0 );

//	if(xWantedSize == 2352 || xWantedSize == 3344)//xWantedSize > 500
//		printf("%s %d xWantedSize = %d pvReturn = %p\r\n", __func__, __LINE__, xWantedSize, pvReturn);
//	if(xWantedSize == 2352 || xWantedSize == 3344 || xWantedSize == 6280)// == 6280 || size == 3344)// || size == 2496)
//            printf("%s %d os_malloc(%d) --- ptr = %p\r\n", __func__, __LINE__, xWantedSize, pvReturn);
    
	return pvReturn;
}
/*-----------------------------------------------------------*/
void vPortFree( void *pv )
{
uint8_t *puc = ( uint8_t * ) pv;
BlockLink_t *pxLink;
//	printf("%s %d pv = %p\r\n", __func__, __LINE__, pv);
	if( pv != NULL )
	{
        DebMemFree(pv);

		/* The memory being freed will have an BlockLink_t structure immediately
		before it. */
		puc -= xHeapStructSize;

		/* This casting is to keep the compiler from issuing warnings. */
		pxLink = ( void * ) puc;

		/* Check the block is actually allocated. */
		configASSERT( ( pxLink->xBlockSize & xBlockAllocatedBit ) != 0 );
		configASSERT( pxLink->pxNextFreeBlock == NULL );

		if( ( pxLink->xBlockSize & xBlockAllocatedBit ) != 0 )
		{
			if( pxLink->pxNextFreeBlock == NULL )
			{
				/* The block is being returned to the heap - it is no longer
				allocated. */
				pxLink->xBlockSize &= ~xBlockAllocatedBit;

				vTaskSuspendAll();
#ifdef MEM_TRACK
//				remove_memorylist(pv);
				log_mem_op(MEM_FREE, pv, 0); // 实际大小通过元数据解析
#endif
				{
					/* Add this block to the list of free blocks. */
					xFreeBytesRemaining += pxLink->xBlockSize;
					traceFREE( pv, pxLink->xBlockSize );
					prvInsertBlockIntoFreeList( ( ( BlockLink_t * ) pxLink ) );
				}
				( void ) xTaskResumeAll();
			}
			else
			{
				mtCOVERAGE_TEST_MARKER();
			}
		}
		else
		{
			mtCOVERAGE_TEST_MARKER();
		}
	}
}
/*-----------------------------------------------------------*/

size_t xPortGetFreeHeapSize( void )
{
	return xFreeBytesRemaining;
}
/*-----------------------------------------------------------*/

size_t xPortGetMinimumEverFreeHeapSize( void )
{
	return xMinimumEverFreeBytesRemaining;
}
/*-----------------------------------------------------------*/

void vPortInitialiseBlocks( void )
{
	/* This just exists to keep the linker quiet. */
}
/*-----------------------------------------------------------*/
static size_t xInitTotalHeapSize=0;

static void prvHeapInit( void )
{
BlockLink_t *pxFirstFreeBlock;
uint8_t *pucAlignedHeap;
size_t uxAddress;
size_t xTotalHeapSize = configTOTAL_HEAP_SIZE;

	/* Ensure the heap starts on a correctly aligned boundary. */
	uxAddress = ( size_t ) ucHeap;

	if( ( uxAddress & portBYTE_ALIGNMENT_MASK ) != 0 )
	{
		uxAddress += ( portBYTE_ALIGNMENT - 1 );
		uxAddress &= ~( ( size_t ) portBYTE_ALIGNMENT_MASK );
		xTotalHeapSize -= uxAddress - ( size_t ) ucHeap;
	}

	pucAlignedHeap = ( uint8_t * ) uxAddress;

	/* xStart is used to hold a pointer to the first item in the list of free
	blocks.  The void cast is used to prevent compiler warnings. */
	xStart.pxNextFreeBlock = ( void * ) pucAlignedHeap;
	xStart.xBlockSize = ( size_t ) 0;

	/* pxEnd is used to mark the end of the list of free blocks and is inserted
	at the end of the heap space. */
	uxAddress = ( ( size_t ) pucAlignedHeap ) + xTotalHeapSize;
	uxAddress -= xHeapStructSize;
	uxAddress &= ~( ( size_t ) portBYTE_ALIGNMENT_MASK );
	pxEnd = ( void * ) uxAddress;
	pxEnd->xBlockSize = 0;
	pxEnd->pxNextFreeBlock = NULL;

	/* To start with there is a single free block that is sized to take up the
	entire heap space, minus the space taken by pxEnd. */
	pxFirstFreeBlock = ( void * ) pucAlignedHeap;
	pxFirstFreeBlock->xBlockSize = uxAddress - ( size_t ) pxFirstFreeBlock;
	pxFirstFreeBlock->pxNextFreeBlock = pxEnd;

	/* Only one block exists - and it covers the entire usable heap space. */
	xMinimumEverFreeBytesRemaining = pxFirstFreeBlock->xBlockSize;
	xFreeBytesRemaining = pxFirstFreeBlock->xBlockSize;

	/* Work out the position of the top bit in a size_t variable. */
	xBlockAllocatedBit = ( ( size_t ) 1 ) << ( ( sizeof( size_t ) * heapBITS_PER_BYTE ) - 1 );

    /* Include remaining memory for heap & save grand total value of heap space */
    vInitRemainingMemory();
    xInitTotalHeapSize = xFreeBytesRemaining;
//	printf("%s %d xInitTotalHeapSize = %d\r\n", __func__, __LINE__, xInitTotalHeapSize);
    /* Include first 8k (0x2000000-0x20001F7C in malloc heap */
    vInitFirst8k();
    xFreeBytesRemaining = xInitTotalHeapSize;

#if 0
    BlockLink_t *pxBlock;
    uint32_t total=0;
    printf("\nmalloc memory map\n");
    pxBlock = xStart.pxNextFreeBlock;
    while( pxBlock->pxNextFreeBlock != NULL )
    {
        printf(" 0x%08x %6d\n", pxBlock, pxBlock->xBlockSize);
        total += pxBlock->xBlockSize;
        pxBlock = pxBlock->pxNextFreeBlock;
    }
    printf("   Total:   %6d\n", total);
#endif
}
/*-----------------------------------------------------------*/

static void prvInsertBlockIntoFreeList( BlockLink_t *pxBlockToInsert )
{
BlockLink_t *pxIterator;
uint8_t *puc;

	/* Iterate through the list until a block is found that has a higher address
	than the block being inserted. */
	for( pxIterator = &xStart; pxIterator->pxNextFreeBlock < pxBlockToInsert; pxIterator = pxIterator->pxNextFreeBlock )
	{
		/* Nothing to do here, just iterate to the right position. */
	}

	/* Do the block being inserted, and the block it is being inserted after
	make a contiguous block of memory? */
	puc = ( uint8_t * ) pxIterator;
	if( ( puc + pxIterator->xBlockSize ) == ( uint8_t * ) pxBlockToInsert )
	{
		pxIterator->xBlockSize += pxBlockToInsert->xBlockSize;
		pxBlockToInsert = pxIterator;
	}
	else
	{
		mtCOVERAGE_TEST_MARKER();
	}

	/* Do the block being inserted, and the block it is being inserted before
	make a contiguous block of memory? */
	puc = ( uint8_t * ) pxBlockToInsert;
	if( ( puc + pxBlockToInsert->xBlockSize ) == ( uint8_t * ) pxIterator->pxNextFreeBlock )
	{
		if( pxIterator->pxNextFreeBlock != pxEnd )
		{
			/* Form one big block from the two blocks. */
			pxBlockToInsert->xBlockSize += pxIterator->pxNextFreeBlock->xBlockSize;
			pxBlockToInsert->pxNextFreeBlock = pxIterator->pxNextFreeBlock->pxNextFreeBlock;
		}
		else
		{
			pxBlockToInsert->pxNextFreeBlock = pxEnd;
		}
	}
	else
	{
		pxBlockToInsert->pxNextFreeBlock = pxIterator->pxNextFreeBlock;
	}

	/* If the block being inserted plugged a gab, so was merged with the block
	before and the block after, then it's pxNextFreeBlock pointer will have
	already been set, and should not be set here as that would make it point
	to itself. */
	if( pxIterator != pxBlockToInsert )
	{
		pxIterator->pxNextFreeBlock = pxBlockToInsert;
	}
	else
	{
		mtCOVERAGE_TEST_MARKER();
	}
}

/*-----------------------------------------------------------*/

size_t xGetTotalHeapSize( void )
{
    if (xInitTotalHeapSize==0)
        prvHeapInit();
	return xInitTotalHeapSize;
}

/*-----------------------------------------------------------*/

#define MEM8K_START (void *)0x20000000
#define MEM8K_END   (void *)0x20001F7C
static void vInitFirst8k(void)
{
    // add memory to heap
    vAddToHeap(MEM8K_START, MEM8K_END);
}

/*-----------------------------------------------------------*/

// Assume 8 byte boundary
#pragma section = "HEAP"
static void * pHeapTop = __section_end("HEAP"); // first available byte after heap
#define HEAP_INIT_OFFSET    0
static void vInitRemainingMemory(void)
{
    BlockLink_t *pxLink;
    size_t size;

    // account for linker created init data after HEAP
    pHeapTop = (uint8_t *)pHeapTop + HEAP_INIT_OFFSET;

    // calculate size of remaining memory minus pxEnd
    size = (__TopOfMemory & ~portBYTE_ALIGNMENT_MASK) - (uint32_t)pHeapTop - xHeapStructSize;
//	printf("%s %d (%p & ~%d) - (uint32_t)%p - %d = %d \r\n", __func__, __LINE__, __TopOfMemory, portBYTE_ALIGNMENT_MASK, pHeapTop, xHeapStructSize, size);
    // link previous pxEnd to the start of the remaining memory
    pxEnd->pxNextFreeBlock = pHeapTop;

    // place remaining memory into malloc chain
    pxLink = pHeapTop;                          // Start of memory to add
    memset(pxLink, 0, sizeof(pxLink));          // Zero mem header
    pxLink->xBlockSize = size;                  // Size of remaining memory
    pxLink->pxNextFreeBlock = (void *)((uint32_t)pxLink + size);
//	printf("%s %d %d += %d\r\n", __func__, __LINE__, xFreeBytesRemaining, pxLink->xBlockSize);
    xFreeBytesRemaining += pxLink->xBlockSize;  // Increase total mem count
//	printf("%s %d %d += %d\r\n", __func__, __LINE__, xFreeBytesRemaining, pxLink->xBlockSize);
    // move pxEnd to the end of the remaining memory
    pxEnd = pxLink->pxNextFreeBlock;
	pxEnd->xBlockSize = 0;
	pxEnd->pxNextFreeBlock = NULL;
}

/*-----------------------------------------------------------*/

static void vAddToHeap(void *BegAddr, void *EndAddr)
{
    BlockLink_t *pxLink;
    uint32_t beg = (uint32_t)BegAddr;
    uint32_t end = (uint32_t)EndAddr;
    size_t  size;

    // make sure alignment is ok
    beg += ( portBYTE_ALIGNMENT - 1 );
    beg &= ~( ( size_t ) portBYTE_ALIGNMENT_MASK );
    end &= ~( ( size_t ) portBYTE_ALIGNMENT_MASK );
    size = end - beg - portBYTE_ALIGNMENT;

    // add to free list
    pxLink = (void *)beg;
    pxLink->xBlockSize = size;
    prvInsertBlockIntoFreeList( pxLink );

    // add size to total free
    xInitTotalHeapSize += size;
}


//==============================================================================
#ifdef DEBUG_HEAP

#define TASK_MAX 32
typedef struct {
    TaskHandle_t xHandle;
    TaskStatus_t xTaskDetails;
} DEB_TASK_t;
DEB_TASK_t DebTask[TASK_MAX]={0};


typedef struct {
    void *   pMem;
    uint16_t size;
    uint8_t  ntask;
} DEB_LOG_t;
DEB_LOG_t DebLog[DEB_HEAP_MAX]={0};

static void DebMemMalloc(void *pMem, size_t size)
{
    int i, n;
    TaskHandle_t xHandle;
    TaskStatus_t xTaskDetails;

    // Subtract header structure size
    size -= xHeapStructSize;

    // ignore small mem allocations
    if (size<=DEB_HEAP_MIN)
        return;

    for (i=0; i<DEB_HEAP_MAX; i++) {
        if (DebLog[i].pMem == NULL)
        {
            xHandle = xTaskGetCurrentTaskHandle();
            vTaskGetTaskInfo(xHandle, &xTaskDetails, pdFALSE, eRunning);
            for (n=0; n<TASK_MAX; n++) {
                if (DebTask[n].xHandle == xHandle)
                    break;
                else if (DebTask[n].xHandle == NULL) {
                    DebTask[n].xHandle = xHandle;
                    DebTask[n].xTaskDetails = xTaskDetails;
                    break;
                }
            }
            if (n==TASK_MAX)
                return;
            DebLog[i].pMem = pMem;
            DebLog[i].size = size;
            DebLog[i].ntask = n;

            // Trap for a specific malloc
            //if ((size == 2080) && strstr(DebTask[n].xTaskDetails.pcTaskName, "tcpip")) {
            //    asm("nop");
            //}

            return;
        }
    }
}


static void DebMemFree(void *pMem)
{
    for (int i=0; i<DEB_HEAP_MAX; i++) {
        if (pMem == DebLog[i].pMem) {
            DebLog[i].pMem = NULL;
            break;
        }
    }
}

static char *DebStateText(eTaskState s)
{
    switch (s) {
    case eRunning:     return "Running";
	case eReady:       return "Ready";
	case eBlocked:     return "Blocked";
	case eSuspended:   return "Suspended";
	case eDeleted:     return "Deleted";
	case eInvalid:     return "Invalid";
    default:           return "?";
    }
}

void vDebDumpMalloc(void)
{
    uint32_t total=0;
    for (int i=0; i<DEB_HEAP_MAX; i++) {
        if (DebLog[i].pMem) {
            total += DebLog[i].size;
            printf("%5d %s (%s)\n", DebLog[i].size,
                   DebTask[ DebLog[i].ntask ].xTaskDetails.pcTaskName,
                   DebStateText( eTaskGetState( DebTask[ DebLog[i].ntask ].xHandle ) ));
        }
    }
    printf("%5d total, %d remaining, %d min remaining\n", total, xFreeBytesRemaining, xMinimumEverFreeBytesRemaining);
}

void vDebClearMalloc(void)
{
    memset(&DebTask, 0, sizeof(DebTask));
    memset(&DebLog, 0, sizeof(DebLog));
}

#endif
//==============================================================================

