#include "mqtt_wrapper.h"
#include "wlan_def.h"
#include "Utilities.h"
#include "AT_Config.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

extern void AT_CmdMqttStart(char **ppBuf);
extern void AT_CmdMqttDis(char **ppBuf);
extern void AT_CmdMqttUnSub(char **ppBuf);

extern long MQTT_UnSub(const unsigned char *topic, unsigned short mode);
extern long MQTT_Publish(const unsigned char *topic, const unsigned char *msg, unsigned short len);

#define MAX_IP_DOMAIN_LEN 64
#define MAX_USERNAME_LEN 64
#define MAX_PASSWORD_LEN 64
#define MAX_TOPIC_LEN 20
#define MAX_CERT_FILE_LEN 64

static enum result_type set_config_helper(const char *key, const char *value)
{
    if (key == NULL || value == NULL) return PARA_ERR;
    int ix;
    for (ix = 0; ix < ConfigVarMax; ix++) {
        if (UT_isequal((char *)key, ConfigVar[ix].Name, 0)) break;
    }
    if (ix >= ConfigVarMax) return PARA_ERR;
    if (UT_SetConfig(NULL, ix, (char *)value, false, false) != NULL) return VALUE_ERR;
    return NO_ERR;
}

static enum result_type set_config_int_helper(const char *key, int value)
{
    char val_str[16];
    sprintf(val_str, "%d", value);
    return set_config_helper(key, val_str);
}

enum result_type wlan_mqtt_set_server_ip(const char *ip_or_domain)
{
    if (strnlen(ip_or_domain, MAX_IP_DOMAIN_LEN + 1) > MAX_IP_DOMAIN_LEN) return PARA_ERR;
    return set_config_helper("MQTTServerIP", ip_or_domain);
}

enum result_type wlan_mqtt_set_server_port(int port)
{
    if (port <= 0 || port > 65535) return PARA_ERR;
    return set_config_int_helper("MQTTServerPort", port);
}

enum result_type wlan_mqtt_set_username(const char *username)
{
    if (strnlen(username, MAX_USERNAME_LEN + 1) > MAX_USERNAME_LEN) return PARA_ERR;
    return set_config_helper("MQTTServerUsrName", username);
}

enum result_type wlan_mqtt_set_password(const char *password)
{
    if (strnlen(password, MAX_PASSWORD_LEN + 1) > MAX_PASSWORD_LEN) return PARA_ERR;
    return set_config_helper("MQTTServerPasswd", password);
}

enum result_type wlan_mqtt_set_pub_topic(const char *topic)
{
    if (strnlen(topic, MAX_TOPIC_LEN + 1) > MAX_TOPIC_LEN) return PARA_ERR;
    return set_config_helper("MQTTPubishTopic", topic);
}

enum result_type wlan_mqtt_set_sub_topic(const char *topic)
{
    if (strnlen(topic, MAX_TOPIC_LEN + 1) > MAX_TOPIC_LEN) return PARA_ERR;
    return set_config_helper("MQTTSubscribeTopic", topic);
}

enum result_type wlan_mqtt_set_qos(int qos)
{
    if (qos < 0 || qos > 2) return PARA_ERR;
    return set_config_int_helper("MQTTQoS", qos);
}

enum result_type wlan_mqtt_set_auth_type(int type)
{
    if (type < 0 || type > 4) return PARA_ERR;
    return set_config_int_helper("MQTTAuthType(0;1;2;4)", type);
}

enum result_type wlan_mqtt_set_ca_cert(const char *filename)
{
    if (strnlen(filename, MAX_CERT_FILE_LEN + 1) > MAX_CERT_FILE_LEN) return PARA_ERR;
    return set_config_helper("MQTTCaCrt", filename);
}

enum result_type wlan_mqtt_set_client_cert(const char *filename)
{
    if (strnlen(filename, MAX_CERT_FILE_LEN + 1) > MAX_CERT_FILE_LEN) return PARA_ERR;
    return set_config_helper("MQTTClinetCrt", filename);
}

enum result_type wlan_mqtt_set_client_key(const char *filename)
{
    if (strnlen(filename, MAX_CERT_FILE_LEN + 1) > MAX_CERT_FILE_LEN) return PARA_ERR;
    return set_config_helper("MQTTClinetKey", filename);
}

enum result_type wlan_mqtt_connect(void)
{
    AT_CmdMqttStart(NULL);
    return NO_ERR;
}

enum result_type wlan_mqtt_disconnect(void)
{
    AT_CmdMqttDis(NULL);
    return NO_ERR;
}

enum result_type wlan_mqtt_publish(const char *topic, const char *message, int len)
{
    if (topic == NULL || message == NULL)
        return PARA_ERR;

    if (MQTT_Publish((const unsigned char *)topic, (const unsigned char *)message, (unsigned short)len) == 1)
        return NO_ERR;

    return SEND_ERR;
}

enum result_type wlan_mqtt_subscribe(const char *topic, int rfu)
{
    if (topic == NULL)
        return PARA_ERR;

    if (MQTT_UnSub((const unsigned char *)topic, 1) == 1)
        return NO_ERR;

    return SEND_ERR;
}

enum result_type wlan_mqtt_unsubscribe(const char *topic)
{
    if (topic == NULL)
        return PARA_ERR;

    if (MQTT_UnSub((const unsigned char *)topic, 0) == 1)
        return NO_ERR;

    return SEND_ERR;
}
