#ifdef AMP_SDK_EXAMPLE

/*system includes.*/
#include "ach118x.h"
#include <ctype.h>
#include <stdio.h>

/*GKI header file*/
#include "GKI_uart.h"
#include "GKI_sys.h"

#include <stdio.h>
#include <string.h>

/*freertos includes.*/
#include "FreeRTOS.h"
#include "lwip/ip_addr.h"
#include "task.h"
#include "queue.h"
#include "semphr.h"
#include "timers.h"

#include "GKI_internals.h"

#include "wlan_def.h"
/* wlan API */
#include "wlan_com.h"
#include "wlan_config.h"
#include "wlan_def.h"
#include "wlan_driver.h"
#include "wlan_sys.h"

/* mqtt API */
#include "mqtt_wrapper.h"

#include "sdk_signal_mnt.h" // Include the actual header for the framework

#include "amp_example.h"


// hex print macro
#define AMP_TEST_PRINT_HEX(buff, len)  \
    do {                                \
        for (uint32_t i = 0; i < len; i++) { \
            printf("%02X ", ((uint8_t *)(buff))[i]); \
        }                               \
        printf("\n");                   \
    } while (0)

int split_words(char *str, char **out_parts, int max_parts) {
    int count = 0;
    char *p = str;

    while (*p != '\0') {
        while (*p == ' ') {
            *p = '\0';
            p++;
        }
        if (*p == '\0') {
            break;
        }
        if (count < max_parts) {
            out_parts[count] = p;
        }
        count++;
        while (*p != '\0' && *p != ' ') {
            p++;
        }
    }

    return count;
}

void test_scan_callback(const scan_data_t *pScandata)
{
    scan_data_t *pScan;

    AMP_SYS_LOG("Scan Result:\n");

    for (pScan = (scan_data_t *)pScandata; pScan != NULL; pScan = (scan_data_t *)pScan->next) {
        AMP_SYS_LOG("SSID: %s, Signal: %d, Freq: %d, BSSID: %02X:%02X:%02X:%02X:%02X:%02X",
                    pScan->ssid,
                    pScan->signal,
                    pScan->freq,
                    pScan->bssid[0], pScan->bssid[1], pScan->bssid[2],
                    pScan->bssid[3], pScan->bssid[4], pScan->bssid[5]);
    }
}

/*
 * str words[1]:
 *  "STA"
 *  "AP"
 *  "AP_STA"
 *  "MP"
 *  "AP_MP"
 */
int test_wlan_set_operation_mode(int word_count, char **words)
{
    unsigned char ret = -1;

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }

    if (strcmp(words[1], "STA") == 0) {
        ret = wlan_set_operation_mode(STA);
    } else if (strcmp(words[1], "AP") == 0) {
        ret = wlan_set_operation_mode(AP);
    } else if (strcmp(words[1], "AP_STA") == 0) {
        ret = wlan_set_operation_mode(AP_STA);
    } else if (strcmp(words[1], "MP") == 0) {
        ret = wlan_set_operation_mode(MP);
    } else if (strcmp(words[1], "AP_MP") == 0) {
        ret = wlan_set_operation_mode(AP_MP);
    } else {
        AMP_SYS_LOG("Parameter error");
        return -1;
    }

    AMP_SYS_LOG("This line will not be reached. System will restart automatically.");

    return 0;
}

int test_wlan_get_operation_mode(int word_count, char **words)
{
    operate_mode_t mode;

    mode = wlan_get_operation_mode();
    switch (mode) {
        case STA:
            AMP_SYS_LOG("Operation Mode: STA");
            break;
        case AP:
            AMP_SYS_LOG("Operation Mode: AP");
            break;
        case AP_STA:
            AMP_SYS_LOG("Operation Mode: AP_STA");
            break;
        case MP:
            AMP_SYS_LOG("Operation Mode: MP");
            break;
        case AP_MP:
            AMP_SYS_LOG("Operation Mode: AP_MP");
            break;
        default:
            AMP_SYS_LOG("Operation Mode: Unknown");
            break;
    }

    return 0;
}

int test_wlan_sta_scan(int word_count, char **words)
{
    unsigned char ret = -1;
    scan_callback_t cb = test_scan_callback;

    ret = wlan_sta_scan(&cb);
    AMP_SYS_LOG("wlan_sta_scan ret %02X", ret);

    return 0;
}

int test_wlan_sta_join(int word_count, char **words)
{
    unsigned char ret = -1;
    wlan_ap_t ap_info;

    if (word_count < 3) {
        strcpy((char *)ap_info.ssid, "Xiaomi_7ACA_5G");       // Your WiFi router
        strcpy((char *)ap_info.password, "87654321");          // Your password
    } else {
        strcpy((char *)ap_info.ssid,     words[1]);   // Your WiFi router
        strcpy((char *)ap_info.password, words[2]);   // Your password
    }
    AMP_SYS_LOG("Joining SSID: %s, PASSWORD: %s", ap_info.ssid, ap_info.password);
    ret = wlan_sta_join(&ap_info);
    AMP_SYS_LOG("wlan_sta_join ret %02X", ret);

    return 0;
}

int test_wlan_sta_unjoin(int word_count, char **words)
{
    wlan_sta_unjoin();
    AMP_SYS_LOG("wlan_sta_unjoin ret");

    return 0;
}

int test_wlan_sta_status(int word_count, char **words)
{
    bool ret = 0;

    ret = wlan_sta_status();
    AMP_SYS_LOG("wlan_sta_status ret %02X", ret);

    return 0;
}

int test_wlan_ap_showNetwork(int word_count, char **words)
{
    wlan_ap_showNetwork();
    AMP_SYS_LOG("wlan_show_scan_result ");

    return 0;
}

int test_wlan_show_scan_result(int word_count, char **words)
{
    wlan_show_scan_result();
    AMP_SYS_LOG("wlan_show_scan_result ");

    return 0;
}

int test_wlan_config_info(int word_count, char **words)
{
    if (word_count == 1) {
        //print all config info
        wlan_config_info(0, 0);
    } else if (word_count == 2) {
        //print var_id config info
        int var_id = atoi(words[1]);
        wlan_config_info(var_id, 0);
    } else if (word_count == 3) {
        //set var_id config info
        int var_id = atoi(words[1]);
        wlan_config_info(var_id, words[2]);
    } else {
        AMP_SYS_LOG("Parameter error");
        return -1;
    }

    AMP_SYS_LOG("wlan_config_info ret");

    return 0;
}

int test_wlan_get_config_byID(int word_count, char **words)
{
    char value[128];

    if (word_count < 2) {
        AMP_SYS_LOG("Parameter error");
        return -1;
    }

    int var_id = atoi(words[1]);
    wlan_get_config_byID(value, (short)sizeof(value), (short)var_id);
    AMP_SYS_LOG("Config var %d: %s", var_id, value);

    return 0;
}

int test_wlan_get_all_config(int word_count, char **words)
{
    int i;

    for (i = 1 ; i < AMP_VARID_PASS_PHRASE_ALT; i++) {
        char value[128];
        wlan_get_config_byID(value, (short)sizeof(value), (short)i);
        AMP_SYS_LOG("Config var %d: \t\t%s", i, value);
    }
    return 0;
}

int test_wlan_set_device_name(int word_count, char **words)
{
    unsigned char ret = -1;

    if (word_count < 2) {
        AMP_SYS_LOG("Parameter error");
        return -1;
    }

    ret = wlan_set_device_name(words[1]);
    AMP_SYS_LOG("wlan_set_device_name ret %02X", ret);

    return 0;
}

int test_wlan_get_device_name(int word_count, char **words)
{
    char *name;

    name = wlan_get_device_name();
    AMP_SYS_LOG("Device Name: %s", name);

    return 0;
}

int test_wlan_set_mac_address(int word_count, char **words)
{
    unsigned char ret = -1;
    unsigned char mac[6];

    if (word_count < 2) {
        AMP_SYS_LOG("Parameter error");
        return -1;
    }

//    sscanf(words[1], "%hhx:%hhx:%hhx:%hhx:%hhx:%hhx",
//           &mac[0], &mac[1], &mac[2], &mac[3], &mac[4], &mac[5]);

    wlan_set_mac_address(words[1]);
    AMP_SYS_LOG("wlan_set_mac_address done.(%s)", words[1]);

    return 0;
}

int test_wlan_get_mac_address(int word_count, char **words)
{
    unsigned char mac[6];

    wlan_get_mac_address(mac);
    AMP_SYS_LOG("MAC Address: %02X:%02X:%02X:%02X:%02X:%02X",
           mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

    return 0;
}

int test_wlan_get_dhcp_mode(int word_count, char **words)
{
    bool mode;

    mode = wlan_get_dhcp_mode();
    AMP_SYS_LOG("DHCP Mode: %s", mode ? "Enabled" : "Disabled");

    return 0;
}

int test_wlan_set_dhcp_mode(int word_count, char **words)
{
    unsigned char ret = -1;
    bool mode;

    if (word_count < 2) {
        AMP_SYS_LOG("Parameter error");
        return -1;
    }

    mode = (atoi(words[1]) != 0);
    ret = wlan_set_dhcp_mode(mode);
    AMP_SYS_LOG("wlan_set_dhcp_mode ret %02X", ret);

    return 0;
}

int test_wlan_get_ip_info(int word_count, char **words)
{
    ip_info_int_t ip_info;

    wlan_get_ip_info(&ip_info);

    AMP_SYS_LOG("IP Address: %s", ipaddr_ntoa((ip_addr_t *)&ip_info.ip));
    AMP_SYS_LOG("NetMask:    %s", ipaddr_ntoa((ip_addr_t *)&ip_info.netmask));
    AMP_SYS_LOG("GateWay:    %s", ipaddr_ntoa((ip_addr_t *)&ip_info.gateway));

    return 0;
}

int test_wlan_set_ip_info(int word_count, char **words)
{
    unsigned char ret = -1;
    ip_info_str_t ip_info;

    if (word_count < 4) {
        AMP_SYS_LOG("Parameter error");
        return -1;
    }

    strncpy(ip_info.ip, words[1], sizeof(ip_info.ip) - 1);
    ip_info.ip[sizeof(ip_info.ip) - 1] = '\0';
    strncpy(ip_info.netmask, words[2], sizeof(ip_info.netmask) - 1);
    ip_info.netmask[sizeof(ip_info.netmask) - 1] = '\0';
    strncpy(ip_info.gateway, words[3], sizeof(ip_info.gateway) - 1);
    ip_info.gateway[sizeof(ip_info.gateway) - 1] = '\0';

    ret = wlan_set_ip_info(ip_info);
    AMP_SYS_LOG("wlan_set_ip_info ret %02X", ret);

    return 0;
}

/*
 * str words[1]:
 *   NO_SLEEP,
 *   SHALLOW_SLEEP,
 *   DEEP_SLEEP,
 *   SHALLOW_DEEP_SLEEP
 */
int test_wlan_set_sleep_mode(int word_count, char **words)
{
    unsigned char ret = -1;
    sleep_mode_t mode;

    if (word_count < 2) {
        AMP_SYS_LOG("Parameter error");
        return -1;
    }

    if (strcmp(words[1], "NO_SLEEP") == 0) {
        mode = NO_SLEEP;
    } else if (strcmp(words[1], "SHALLOW_SLEEP") == 0) {
        mode = SHALLOW_SLEEP;
    } else if (strcmp(words[1], "DEEP_SLEEP") == 0) {
        mode = DEEP_SLEEP;
    } else if (strcmp(words[1], "SHALLOW_DEEP_SLEEP") == 0) {
        mode = SHALLOW_DEEP_SLEEP;
    } else {
        AMP_SYS_LOG("Parameter error\n");
        return -1;
    }

    ret = wlan_set_sleep_mode(mode);
    AMP_SYS_LOG("wlan_set_sleep_mode ret %02X\n", ret);

    return 0;
}

int test_wlan_get_sleep_mode(int word_count, char **words)
{
    sleep_mode_t mode;

    mode = wlan_get_sleep_mode();
    AMP_SYS_LOG("Sleep Mode: %d\n", mode);

    return 0;
}

//system API
int test_wlan_restart(int word_count, char **words)
{
    wlan_restart();
    printf("wlan_restart done\n");

    return 0;
}

int test_wlan_printf(int word_count, char **words)
{
    wlan_printf("This is a test printf from wlan_printf\n");
    return 0;
}

//driver API
int test_wlan_set_uart_connect_mode(int word_count, char **words)
{
    printf("test_wlan_set_uart_connect_mode not support.\n");
    return 0;
}

int test_wlan_uart_send(int word_count, char **words)
{
    uart_port_t portnum;
    unsigned char data;

    if (word_count < 3) {
        printf("Parameter error\n");
        return -1;
    }
    portnum = (uart_port_t)atoi(words[1]);
    wlan_uart_send(portnum, (unsigned char)words[2][0]);

    return 0;
}
int test_wlan_set_uart_baudrate(int word_count, char **words)
{
    uart_port_t PortNum;
    unsigned int BaudRate;

    if (word_count < 3) {
        printf("Parameter error\n");
        return -1;
    }
    PortNum = (uart_port_t)atoi(words[1]);
    BaudRate = (unsigned int)atoi(words[2]);
    switch(PortNum) {
        case UART_PORT0:
            GKI_UartInit(UART0, BaudRate);
            break;
        case UART_PORT1:
            GKI_UartInit(UART1, BaudRate);
            break;
        case UART_PORT2:
            GKI_UartInit(UART2, BaudRate);
            break;
        default:
            printf("Parameter error! %s %d \r\n", __FILE__, __LINE__);
            return -1;
    }
    printf("wlan_set_uart_baudrate done.(PortNum=%d, BaudRate=%d)\n", PortNum, BaudRate);
    return 0;
}

int test_wlan_spi_init(int word_count, char **words)
{
    spi_port_t portnum;

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }
    portnum = (spi_port_t)atoi(words[1]);
//    wlan_spi_init(portnum);
    printf("wlan_spi_init done.(portnum=%d)\n", portnum);

    return 0;
}
int test_wlan_spi_send_bytes(int word_count, char **words)
{
    return 0;
}
int test_wlan_spi_sndrcv_bytes(int word_count, char **words)
{
    printf("test_wlan_spi_sndrcv_bytes not support.\n");
    return 0;
}
int test_wlan_spi_rcv_bytes(int word_count, char **words)
{
    printf("test_wlan_spi_rcv_bytes not support.\n");
    return 0;
}
int test_wlan_gpio_config(int word_count, char **words)
{
    wlan_gpio_t gpio;
    gpio_direction_t dir;

    if (word_count < 3) {
        printf("Parameter error\n");
        return -1;
    }

    gpio = (wlan_gpio_t)atoi(words[1]);
    dir = (gpio_direction_t)atoi(words[2]);
    wlan_gpio_config(gpio, dir);
    printf("wlan_gpio_config done. gpio=%d, dir=%d\n", gpio, dir);

    return 0;
}

int test_wlan_gpio_set(int word_count, char **words)
{
    wlan_gpio_t gpio;
    unsigned char value;

    if (word_count < 3) {
        printf("Parameter error\n");
        return -1;
    }

    gpio = (wlan_gpio_t)atoi(words[1]);
    value = (unsigned char)atoi(words[2]);
    wlan_gpio_set(gpio, value);
    printf("wlan_gpio_set done. gpio=%d, value=%d\n", gpio, value);

    return 0;
}

int test_wlan_gpio_get(int word_count, char **words)
{
    wlan_gpio_t gpio;
    unsigned char value;

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }

    gpio = (wlan_gpio_t)atoi(words[1]);
    value = wlan_gpio_get(gpio);
    printf("wlan_gpio_get done. gpio=%d, value=%d\n", gpio, value);

    return 0;
}

typedef struct {
    int pin_num;
    int interval_ms;
} GpioConfig_t;

void vGpioTask(void *arg)
{
    GpioConfig_t *gpio_config = (GpioConfig_t *)arg;
    wlan_gpio_t gpio;
    wlan_printf("GPIO_EXAMPLE\r\n");

    gpio = (wlan_gpio_t)(gpio_config->pin_num);

    wlan_gpio_config(gpio, GPIO_OUTPUT);
    while (1) {
        if (ulTaskNotifyTake(pdTRUE, 0) > 0) {
            break;
        }
        vTaskDelay(gpio_config->interval_ms);
        wlan_gpio_set(gpio, 1);
        printf("GPIO %d HIGH\n", gpio);
        vTaskDelay(gpio_config->interval_ms);
        wlan_gpio_set(gpio, 0);
        printf("GPIO %d LOW\n", gpio);
    }

    vTaskDelete(NULL);
}

xTaskHandle gpio_task_handle;

int test_wlan_gpio_start(int word_count, char **words)
{
    static GpioConfig_t gpio_config;

    if (gpio_task_handle != NULL) {
        printf("wlan_gpio_task already running.\n");
        return -1;
    }

    if (word_count < 3) {
        printf("Parameter error. Usage: wlan_gpio_start <pin> <ms>\n");
        return -1;
    }

    gpio_config.pin_num = atoi(words[1]);
    gpio_config.interval_ms = atoi(words[2]);

    xTaskCreate(vGpioTask, "wlan_gpio_task", 512, &gpio_config, 6, &gpio_task_handle);

    printf("wlan_gpio_task created. Pin: %d, Interval: %dms\n", 
           gpio_config.pin_num, gpio_config.interval_ms);

    return 0;
}

int test_wlan_gpio_stop(int word_count, char **words)
{
    if (gpio_task_handle != NULL) {
        xTaskNotifyGive(gpio_task_handle);
        gpio_task_handle = NULL;
    }

    return 0;
}

typedef struct {
    int pin_num;
    int interval_ms;
} SpiConfig_t;

void vSpiTask(void *arg)
{
    SpiConfig_t *spi_config = (SpiConfig_t *)arg;
    wlan_gpio_t gpio;
    unsigned int ret;

    wlan_printf("SPI_EXAMPLE\r\n");

    gpio = (wlan_gpio_t)(spi_config->pin_num);

//    wlan_spi_init(SPI_PORT1);

    while (1) {
        if (ulTaskNotifyTake(pdTRUE, 0) > 0) {
            break;
        }
//        ret = wlan_spi_send(SPI_PORT1,0xA5);
        printf("SPI Send 0xA5, ret=%d\n", ret);
        vTaskDelay(spi_config->interval_ms);
    }

    vTaskDelete(NULL);
}

xTaskHandle spi_task_handle = NULL;

int test_wlan_spi_start(int word_count, char **words)
{
    SpiConfig_t spi_config;

    if (spi_task_handle != NULL) {
        printf("wlan_spi_task already running.\n");
        return -1;
    }

    if (word_count < 3) {
        printf("Parameter error\n");
        return -1;
    }
    spi_config.pin_num = atoi(words[1]);
    spi_config.interval_ms = atoi(words[2]);
    xTaskCreate(vSpiTask, "wlan_spi_task", 512, &spi_config, 6, &spi_task_handle);
    printf("wlan_spi_task created.\n");

    return 0;
}

int test_wlan_spi_stop(int word_count, char **words)
{
    if (spi_task_handle != NULL) {
        xTaskNotifyGive(spi_task_handle);
        spi_task_handle = NULL;
    }

    return 0;
}


void timer_function_example(TimerHandle_t xTimer)
{
    printf("Timer callback function executed.\n");
}

int test_wlan_timer_config(int word_count, char **words)
{
    unsigned long TimerPeriodInTicks;
    unsigned long uxAutoReload;
    int timer_id;
    unsigned char timer_index;

    if (word_count < 5) {
        printf("Parameter error\n");
        return -1;
    }

    TimerPeriodInTicks = pdMS_TO_TICKS((TickType_t)atoi(words[2]));
    uxAutoReload = (unsigned long)atoi(words[3]);
    timer_id = atoi(words[4]);

    timer_index = wlan_timer_config(words[1], TimerPeriodInTicks, uxAutoReload, (void *)(long)timer_id, timer_function_example);
    printf("wlan_timer_config done. name=%s, period=%lu, autoreload=%lu, id=%d. timer_index=%02d\n",
           words[1], TimerPeriodInTicks, uxAutoReload, timer_id, timer_index);
    return 0;
}

int test_wlan_timer_start(int word_count, char **words)
{
    unsigned char timer_index;

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }
    timer_index = (unsigned char)atoi(words[1]);
    wlan_timer_start(timer_index);
    printf("wlan_timer_start done. timer_index=%02d\n", timer_index);
    return 0;
}

int test_wlan_timer_stop(int word_count, char **words)
{
    unsigned char timer_index;

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }
    timer_index = (unsigned char)atoi(words[1]);
    wlan_timer_stop(timer_index);
    //wlan_timer_destroy(timer_index);
    printf("wlan_timer_stop done. timer_index=%02d\n", timer_index);

    return 0;
}

/* mqtt API test */
int test_wlan_mqtt_set_server_ip(int word_count, char **words)
{
    enum result_type ret = PARA_ERR;

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }

    ret = wlan_mqtt_set_server_ip(words[1]);
    printf("wlan_mqtt_set_server_ip ret %02X\n", ret);

    return 0;
}

int test_wlan_mqtt_set_server_port(int word_count, char **words)
{
    enum result_type ret = PARA_ERR;
    int port;

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }

    port = atoi(words[1]);
    ret = wlan_mqtt_set_server_port(port);
    printf("wlan_mqtt_set_server_port ret %02X\n", ret);

    return 0;
}

int test_wlan_mqtt_set_username(int word_count, char **words)
{
    enum result_type ret = PARA_ERR;

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }

    ret = wlan_mqtt_set_username(words[1]);
    printf("wlan_mqtt_set_username ret %02X\n", ret);

    return 0;
}

int test_wlan_mqtt_set_password(int word_count, char **words)
{
    enum result_type ret = PARA_ERR;

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }

    ret = wlan_mqtt_set_password(words[1]);
    printf("wlan_mqtt_set_password ret %02X\n", ret);

    return 0;
}

int test_wlan_mqtt_set_pub_topic(int word_count, char **words)
{
    enum result_type ret = PARA_ERR;

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }

    ret = wlan_mqtt_set_pub_topic(words[1]);
    printf("wlan_mqtt_set_pub_topic ret %02X\n", ret);

    return 0;
}

int test_wlan_mqtt_set_sub_topic(int word_count, char **words)
{
    enum result_type ret = PARA_ERR;

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }

    ret = wlan_mqtt_set_sub_topic(words[1]);
    printf("wlan_mqtt_set_sub_topic ret %02X\n", ret);

    return 0;
}

int test_wlan_mqtt_set_qos(int word_count, char **words)
{
    enum result_type ret = PARA_ERR;
    int qos;

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }

    qos = atoi(words[1]);
    ret = wlan_mqtt_set_qos(qos);
    printf("wlan_mqtt_set_qos ret %02X\n", ret);

    return 0;
}

int test_wlan_mqtt_set_auth_type(int word_count, char **words)
{
    enum result_type ret = PARA_ERR;
    int type;

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }

    type = atoi(words[1]);
    ret = wlan_mqtt_set_auth_type(type);
    printf("wlan_mqtt_set_auth_type ret %02X\n", ret);

    return 0;
}

int test_wlan_mqtt_connect(int word_count, char **words)
{
    enum result_type ret = PARA_ERR;

    ret = wlan_mqtt_connect();
    printf("wlan_mqtt_connect ret %02X\n", ret);

    return 0;
}

int test_wlan_mqtt_disconnect(int word_count, char **words)
{
    enum result_type ret = PARA_ERR;

    ret = wlan_mqtt_disconnect();
    printf("wlan_mqtt_disconnect ret %02X\n", ret);

    return 0;
}

int test_wlan_mqtt_publish(int word_count, char **words)
{
    enum result_type ret = PARA_ERR;
    char *topic;
    char *message;
    int message_len;


    if (word_count < 3) {
        printf("Parameter error\n");
        return -1;
    }

    topic = words[1];
    message = words[2];
    message_len = strlen(message);

    ret = wlan_mqtt_publish(topic, message, message_len);

    printf("wlan_mqtt_publish ret %02X\n", ret);

    return 0;
}

int test_wlan_mqtt_subscribe(int word_count, char **words)
{
    enum result_type ret = PARA_ERR;
    char *topic;
    int rfu = 0;

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }

    topic = words[1];

    ret = wlan_mqtt_subscribe(topic, rfu);

    printf("wlan_mqtt_subscribe ret %02X\n", ret);

    return 0;
}

int test_wlan_mqtt_unsubscribe(int word_count, char **words)
{
    enum result_type ret = PARA_ERR;
    char *topic;

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }

    topic = words[1];

    ret = wlan_mqtt_unsubscribe(topic);

    printf("wlan_mqtt_unsubscribe ret %02X\n", ret);

    return 0;
}

/*
 * This is an example function to setup a mesh network.
 * It configures the device as a Mesh Point (MP), sets the channel,
 * mesh ID, static IP address, and gateway, then restarts the system
 * to apply the changes.
 */
int test_mesh_setup_example(int word_count, char **words)
{
    wlan_printf("Starting Mesh Configuration...\n");

    // 1. Configure Device Mode to "MP"
    wlan_config_info(AMP_VARID_DEVICE_MODE, "MP");
    printf("Set Device Mode(var %d) to MP\n", AMP_VARID_DEVICE_MODE);

    // 2. Configure Static IP Address
    wlan_config_info(AMP_VARID_IP_ADDRESS, "192.168.0.90");
    printf("Set Static IP Address to 192.168.0.90\n");

    // 3. Configure NetMask
    wlan_config_info(AMP_VARID_NET_MASK, "255.255.255.0");
    printf("Set NetMask to 255.255.255.0\n");

    // 4. Configure Gateway IP
    wlan_config_info(AMP_VARID_GATE_WAY, "192.168.0.1");
    printf("Set Gateway IP to 192.168.0.1\n");

    // 5. Configure Channel (e.g., Channel 161)
    wlan_config_info(AMP_VARID_CHANNEL, "161");
    printf("Set Channel to 161\n");

    // 6. Configure Mesh ID
    wlan_config_info(AMP_VARID_MESH_ID, "mymesh123456789");
    printf("Set Mesh ID to mymesh123456789\n");

    // 7. Configure Mesh Auth Type (e.g., 2 for Auth)
    wlan_config_info(AMP_VARID_MESH_AUTH_TYPE, "2");
    printf("Set Mesh Auth Type to 2\n");

    // 8. Configure Mesh Passphrase
    wlan_config_info(AMP_VARID_MESH_PASS_PHRASE, "12345678");
    printf("Set Mesh Passphrase to 12345678\n");

    printf("Waiting 1 second before restart...\n");
    vTaskDelay(1000);

    // 9. Restart System to Apply Changes
    wlan_printf("Configuration complete. Restarting system...\n");
    wlan_restart();
}

// Task handle for the connection monitor
static TaskHandle_t xWifiConnectTask = NULL;

// API to start the STA connection process
void wifi_sta_start(char *ssid, char *password)
{
    static wlan_ap_t ap_config;
    operate_mode_t current_mode;
    bool dhcp_mode;
    bool need_restart = 0;

    // 1. Check DeviceMode (Must be STA)
    current_mode = wlan_get_operation_mode();
    if (current_mode != STA) {
        wlan_printf("Current mode is not STA. Setting to STA...\n");
        wlan_set_operation_mode(STA);
        need_restart = 1;
    }

    // 2. Check DHCPMode (Must be true)
    dhcp_mode = wlan_get_dhcp_mode();
    if (!dhcp_mode) {
        wlan_printf("DHCP is disabled. Enabling...\n");
        wlan_set_dhcp_mode(1);
        need_restart = 1;
    }

    if (need_restart) {
        wlan_printf("Configuration changed. Restarting system...\n");
        wlan_restart();
        return;
    }

    // Prepare AP configuration
    memset(&ap_config, 0, sizeof(wlan_ap_t));
    if (ssid) strncpy((char*)ap_config.ssid, ssid, 32);
    if (password) strncpy((char*)ap_config.password, password, 64);

    wlan_sta_join(&ap_config);
}

int test_wlan_sta_example(int word_count, char **words)
{
    char *ssid = NULL;
    char *password = NULL;

    // Default credentials if none provided
    if (word_count < 2) {
        /* ezviz 12345678 */
        ssid = "ezviz";
        password = "12345678";
    }

    if (word_count >= 2) {
        ssid = words[1];
    }
    if (word_count >= 3) {
        password = words[2];
    }

    wifi_sta_start(ssid, password);

    return 0;
}

#define MQTT_EXAMPLE_BROKER        "192.168.1.76"
#define MQTT_EXAMPLE_PORT          1883
#define MQTT_EXAMPLE_PORT_TLS      1885
#define MQTT_EXAMPLE_USER          "admin"
#define MQTT_EXAMPLE_PASS          "password"
#define MQTT_EXAMPLE_PUB_TOPIC     "CToS"
#define MQTT_EXAMPLE_SUB_TOPIC     "SToC"

void mqtt_example_task(void *pvParameters)
{
    int mode = (int)(long)pvParameters;

    printf("MQTT Example Task Started. Mode(%d): %s\n", mode, mode == 0 ? "Non-TLS" : "TLS");

    wlan_config_info(AMP_VARID_LINK_TYPE, "1"); // Enable MQTT debug info
    wlan_mqtt_set_server_ip(MQTT_EXAMPLE_BROKER);
    wlan_mqtt_set_username(MQTT_EXAMPLE_USER);
    wlan_mqtt_set_password(MQTT_EXAMPLE_PASS);
    wlan_mqtt_set_qos(1);
    wlan_mqtt_set_pub_topic(MQTT_EXAMPLE_PUB_TOPIC);
    wlan_mqtt_set_sub_topic(MQTT_EXAMPLE_SUB_TOPIC);

    if (mode) {
        wlan_mqtt_set_auth_type(mode); // TLS
        // Set CA and client certs if needed
        wlan_mqtt_set_ca_cert("ca.crt");
        wlan_mqtt_set_client_cert("client.crt");
        wlan_mqtt_set_client_key("client.key");
        wlan_mqtt_set_server_port(MQTT_EXAMPLE_PORT_TLS);
    } else {
        wlan_mqtt_set_auth_type(mode); // No Authentication / Anonymous.
        wlan_mqtt_set_server_port(MQTT_EXAMPLE_PORT);
    }

    /* Note: wlan_mqtt_connect() is async and returns 0 even if connection fails. */
    wlan_mqtt_connect();
}

void mqtt_example_start(int mode)
{
    xTaskCreate(mqtt_example_task, "MQTT_Example_TLS", 3024, (void *)mode, 3, NULL);
}

int test_mqtt_example(int word_count, char **words)
{
    int mode;   //authtype: 0=User/Pass, 1=Cert, 2=Mutual,

    if (word_count < 2) {
        printf("Parameter error\n");
        return -1;
    }

    mode = atoi(words[1]);
    mqtt_example_task((void *)mode);

    return 0;
}

int test_os_malloc_free(int word_count, char **words)
{
    int size = 128;
    void *ptr;

    if (word_count >= 2) {
        size = atoi(words[1]);
    }
    if (size <= 0) {
        printf("Parameter error\n");
        return -1;
    }

    ptr = os_malloc((size_t)size);
    printf("os_malloc(%d) = %p\n", size, ptr);
    if (ptr == NULL) {
        return -1;
    }

    memset(ptr, 0xA5, (size_t)size);
    os_free(ptr);
    printf("os_free(%p)\n", ptr);

    return 0;
}

/*********************************************************************
* This is an example AT command callback.
* To evoke this enter: at+wf test
*
* Use the AddAtCmd() function to add your own AT command
**********************************************************************/
typedef int (*cmd_handler_t)(int argc, char **argv);

struct cmd_entry {
    const char *name;
    cmd_handler_t handler;
};

void AT_Test(char **ppBuf)
{
#define WORD_COUNT_MAX 10
    int word_count;
    char *words[WORD_COUNT_MAX];
    static const struct cmd_entry cmds[] = {
        {"wlan_show_scan_result", test_wlan_show_scan_result},
        {"wlan_sta_scan", test_wlan_sta_scan},
        {"wlan_sta_join", test_wlan_sta_join},
        {"wlan_sta_unjoin", test_wlan_sta_unjoin},
//        {"wlan_sta_connect", test_wlan_sta_connect},
//        {"wlan_sta_disconnect", test_wlan_sta_disconnect},
        {"wlan_sta_status", test_wlan_sta_status},
        {"wlan_ap_showNetwork", test_wlan_ap_showNetwork},
//        {"wlan_sta_send", test_wlan_sta_send},
        //config
        {"wlan_get_device_name", test_wlan_get_device_name},
        {"wlan_set_device_name", test_wlan_set_device_name},
        {"wlan_get_mac_address", test_wlan_get_mac_address},
        {"wlan_set_mac_address", test_wlan_set_mac_address},
        {"wlan_get_dhcp_mode", test_wlan_get_dhcp_mode},
        {"wlan_set_dhcp_mode", test_wlan_set_dhcp_mode},
        {"wlan_get_ip_info", test_wlan_get_ip_info},
        {"wlan_set_ip_info", test_wlan_set_ip_info},
        {"wlan_get_sleep_mode", test_wlan_get_sleep_mode},
        {"wlan_set_sleep_mode", test_wlan_set_sleep_mode},
        {"wlan_get_operation_mode", test_wlan_get_operation_mode},
        {"wlan_set_operation_mode", test_wlan_set_operation_mode},
        {"wlan_config_info", test_wlan_config_info},
        {"wlan_get_config_byID", test_wlan_get_config_byID},
        {"wlan_get_all_config", test_wlan_get_all_config},
        //system API
        {"wlan_restart", test_wlan_restart},
        {"wlan_printf", test_wlan_printf},
        {"os_malloc_free", test_os_malloc_free},
        //driver API list
        {"wlan_set_uart_connect_mode", test_wlan_set_uart_connect_mode},
        {"wlan_uart_send", test_wlan_uart_send},
        {"wlan_set_uart_baudrate", test_wlan_set_uart_baudrate},
        {"wlan_spi_init", test_wlan_spi_init},
        {"wlan_spi_send_bytes", test_wlan_spi_send_bytes},
        {"wlan_spi_sndrcv_bytes", test_wlan_spi_sndrcv_bytes},
        {"wlan_spi_rcv_bytes", test_wlan_spi_rcv_bytes},
        {"wlan_spi_start", test_wlan_spi_start},
        {"wlan_spi_stop", test_wlan_spi_stop},

        {"wlan_gpio_config", test_wlan_gpio_config},
        {"wlan_gpio_set", test_wlan_gpio_set},
        {"wlan_gpio_get", test_wlan_gpio_get},

        {"wlan_gpio_start", test_wlan_gpio_start},
        {"wlan_gpio_stop", test_wlan_gpio_stop},
        {"wlan_timer_config", test_wlan_timer_config},
        {"wlan_timer_start", test_wlan_timer_start},
        {"wlan_timer_stop", test_wlan_timer_stop},
        //mqtt API list
        {"wlan_mqtt_set_server_ip", test_wlan_mqtt_set_server_ip},
        {"wlan_mqtt_set_server_port", test_wlan_mqtt_set_server_port},
        {"wlan_mqtt_set_username", test_wlan_mqtt_set_username},
        {"wlan_mqtt_set_password", test_wlan_mqtt_set_password},
        {"wlan_mqtt_set_pub_topic", test_wlan_mqtt_set_pub_topic},
        {"wlan_mqtt_set_sub_topic", test_wlan_mqtt_set_sub_topic},
        {"wlan_mqtt_set_qos", test_wlan_mqtt_set_qos},
        {"wlan_mqtt_set_auth_type", test_wlan_mqtt_set_auth_type},
//        {"wlan_mqtt_set_ca_cert", test_mqtt_api_set_ca_cert},
//        {"wlan_mqtt_set_client_cert", test_mqtt_api_set_client_cert},
//        {"wlan_mqtt_set_client_key", test_mqtt_api_set_client_key},
        {"wlan_mqtt_connect", test_wlan_mqtt_connect},
        {"wlan_mqtt_disconnect", test_wlan_mqtt_disconnect},
        {"wlan_mqtt_publish", test_wlan_mqtt_publish},
        {"wlan_mqtt_subscribe", test_wlan_mqtt_subscribe},
        {"wlan_mqtt_unsubscribe", test_wlan_mqtt_unsubscribe},
        //for complex example
        {"mesh_setup_example", test_mesh_setup_example},
        {"wlan_sta_example", test_wlan_sta_example},
        {"mqtt_example", test_mqtt_example},
    };
    const int cmd_count = (int)(sizeof(cmds) / sizeof(cmds[0]));

    AMP_SYS_LOG("AT test command -- \n");
    if (**ppBuf == 0) {
        AMP_SYS_LOG("No parameter\n");
        return;
    }

    word_count = split_words(*ppBuf, words, WORD_COUNT_MAX);
    for (int i = 0; i < word_count; i++) {
        AMP_SYS_LOG("Word %d: %s\n", i, words[i]);
    }

    if (word_count == 0) {
        AMP_SYS_LOG("No command\n");
        return;
    }

    for (int i = 0; i < cmd_count; i++) {
        if (strcmp(words[0], cmds[i].name) == 0) {
            cmds[i].handler(word_count, words);
            return;
        }
    }

    AMP_SYS_LOG("Parameter error\n");
}


/**
 * @brief Check if the given WiFi channel is valid for 2.4GHz or 5GHz bands.
 * @param channel The WiFi channel number to validate.
 * @return 0 if valid, -1 if invalid.
 */
int is_valid_wifi_channel(int channel) {
    // 1. 2.4GHz channel: 1 ~~ 14
    if (channel >= 1 && channel <= 14) {
        return 0;
    }

    // 2. 5GHz channel: 36 ~~ 165
    if (channel >= 36 && channel <= 165) {
        return 0;
    }

    return -1;
}

/*
 * @brief Check if the mesh network parameters are valid.
 * This includes checking the Mesh ID, Pass Phrase, WiFi Channel,
 * IP Address, NetMask, and Gateway.
 * @return 0 if all parameters are valid, -1 if any parameter is invalid.
 */
int check_mesh_param_valid(void)
{
    int channel = 0;
    char value[32] = {0};

    wlan_get_config_byID(value, (short)sizeof(value), (short)AMP_VARID_MESH_ID);
    if (strlen(value) < 1) {
        AMP_SYS_LOG("Mesh ID is empty");
        return -1;
    }
    AMP_SYS_LOG("Mesh ID: %s", value);

    wlan_get_config_byID(value, (short)sizeof(value), (short)AMP_VARID_MESH_PASS_PHRASE);
    if (strlen(value) < 1) {
        AMP_SYS_LOG("Mesh Pass Phrase is empty");
        return -1;
    }
    AMP_SYS_LOG("Mesh Pass Phrase: %s", value);

    wlan_get_config_byID(value, (short)sizeof(value), (short)AMP_VARID_CHANNEL);
    channel = atoi(value);
    if (is_valid_wifi_channel(channel) != 0) {
        AMP_SYS_LOG("Invalid WiFi Channel: %d", channel);
        return -1;
    }
    AMP_SYS_LOG("WiFi Channel: %d", channel);

    //check IP address is valid
    wlan_get_config_byID(value, (short)sizeof(value), (short)AMP_VARID_IP_ADDRESS);
    if (strlen(value) < 7) {
        AMP_SYS_LOG("IP Address is invalid: %s", value);
        return -1;
    }
    AMP_SYS_LOG("IP Address: %s", value);

    //check NetMask is valid
    wlan_get_config_byID(value, (short)sizeof(value), (short)AMP_VARID_NET_MASK);
    if (strlen(value) < 7) {
        AMP_SYS_LOG("NetMask is invalid: %s", value);
        return -1;
    }
    AMP_SYS_LOG("NetMask: %s", value);

    //check Gateway is valid
    wlan_get_config_byID(value, (short)sizeof(value), (short)AMP_VARID_GATE_WAY);
    if (strlen(value) < 7) {
        AMP_SYS_LOG("Gateway is invalid: %s", value);
        return -1;
    }
    AMP_SYS_LOG("Gateway: %s", value);

    return 0;
}

void start_wifi_example(void)
{
    int ret;
    operate_mode_t mode;

    mode = wlan_get_operation_mode();

    switch (mode) {
        case STA:
            AMP_SYS_LOG("Device in STA mode. Starting STA example...");

            //Attempt to rejoin the AP
            AMP_SYS_LOG("cleanup status. unjoin WiFi network...");
            wlan_sta_unjoin();

            test_wlan_sta_example(0, NULL);
            break;
        case AP:
            AMP_SYS_LOG("Device in AP mode. Starting STA example...");
            break;
        case MP:
            if ((ret = check_mesh_param_valid()) == 0) {
                AMP_SYS_LOG("Mesh parameters valid. System will restart to apply MESH settings.");
            } else {
                AMP_SYS_LOG("Mesh parameters invalid. Please configure correctly.");
            }
            break;
    }
}

void on_init_down(amp_network_status_t *net_status)
{
    start_wifi_example();

    return;
}

void on_join_success(amp_network_status_t *net_status)
{
    AMP_SYS_LOG("Join status updated to JOIN_SUCCESS");
    return;
}

void on_join_fail(amp_network_status_t *net_status)
{
    AMP_SYS_LOG("Join status updated to JOIN_FAILED");

    // Attempt to rejoin the AP
    AMP_SYS_LOG("Re-attempting WiFi connection...");
    start_wifi_example();

    return;
}



void on_wlan_connected(void)
{
    operate_mode_t mode;

    // Determine current operation mode
    mode = wlan_get_operation_mode();

    // Act based on mode
    if (mode == STA) {
        AMP_SYS_LOG("Connected in STA mode. Wait for IP assignment...");
        return;
    } else if (mode == MP) {
        AMP_SYS_LOG("Connected in MESH mode. Attempting MQTT connection...");
        mqtt_example_task((void *)0); // Trigger MQTT reconnect or related action
        return;
    }
}

void on_wlan_disconnected(void)
{
//    wlan_mqtt_disconnect();

    //sleep 10s
    vTaskDelay(pdMS_TO_TICKS(10000));
    AMP_SYS_LOG("Re-attempting WiFi connection...");
    start_wifi_example();
    return;
}

void on_wlan_ip_changed(void)
{
    ip_info_int_t ip_info;

    wlan_get_ip_info(&ip_info);
    if (ip_info.ip.addr != 0 && ip_info.netmask.addr != 0) {
        AMP_SYS_LOG(" New IP: %08X", ip_info.ip.addr);

        //for STA mode, start MQTT connection
        //argment 0: no TLS
        mqtt_example_task((void *)0); // Trigger MQTT reconnect or related action
    } else {
        AMP_SYS_LOG(" IP Address is 0. Possibly disconnected.");
    }
}

void on_wlan_mqtt_connected(void)
{
    const char *test_payload = "Hello from ACH118x MQTT Test!";
    int ret;

    // Publish a test message upon successful MQTT connection
    ret = wlan_mqtt_publish("CToS", test_payload, strlen(test_payload));
    if (ret == 0) {
        AMP_SYS_LOG("Published test MQTT message successfully.");
    } else {
        AMP_SYS_LOG("Failed to publish test MQTT message. Error code: %d", ret);
    }
}

void on_mqtt_disconnected(void)
{
    return;
}

void on_mqtt_data_down(sdk_signal_t *msg)
{
    if (msg->data) {
        AMP_TEST_PRINT_HEX(msg->data, msg->data_len);
    }
}

void on_uart_data_rx(sdk_signal_t *msg)
{
    app_uart_rx_event_t *uart_evt;

    if (msg->data) {
        uart_evt = (app_uart_rx_event_t *)msg->data;
        AMP_SYS_LOG("[Event] UART Data Received. len = %d", uart_evt->len);
        AMP_TEST_PRINT_HEX(uart_evt->data, uart_evt->len);
    }
}

void msg_processed_for_test(sdk_signal_t *msg, amp_network_status_t *net_status)
{
    AMP_SYS_LOG("msg->source: %d, msg->id: %d", msg->source, msg->id);

    switch (msg->id) {
    case SIG_SDK_SYS_INIT_DONE:
    {
        AMP_SYS_LOG("[Event] System Initialization Completed!");
        on_init_down(net_status);
        break;
    }

    case SIG_SDK_JOIN_SUCESS:
    {
        AMP_SYS_LOG("[Event] Successfully Joined WiFi Network!");
        on_join_success(net_status);
        break;
    }
    case SIG_SDK_JOIN_FAILED:
    {
        AMP_SYS_LOG("[Event] Failed to Join WiFi Network!");
        on_join_fail(net_status);
        break;
    }
    case SIG_SDK_WLAN_CONNECTED:
    {
        AMP_SYS_LOG("[Event] WiFi Connected!");
        on_wlan_connected();
        break;
    }
    case SIG_SDK_WLAN_DISCONNECTED:
        {
            AMP_SYS_LOG("[Event] WiFi Disconnected!");
            on_wlan_disconnected();
            break;
        }
    case SIG_SDK_WLAN_IP_CHANGED:
        {
            AMP_SYS_LOG("[Event] IP Address Changed!");
//            on_wlan_ip_changed();
            break;
        }

    case SIG_SDK_MQTT_CONNECT_OK:
        {
            AMP_SYS_LOG("[Event] MQTT Connected Successfully!");
            on_wlan_mqtt_connected();
            break;
        }

    case SIG_SDK_MQTT_DISCONNECT:
        {
            AMP_SYS_LOG("[Event] MQTT Disconnected!");
            on_mqtt_disconnected();
            break;
        }

    case SIG_SDK_MQTT_DATA_DOWN:
        {
            AMP_SYS_LOG("[Event] MQTT Data Received - Processing Payload. len = %d", msg->data_len);
            on_mqtt_data_down(msg);
            break;
        }

    case SIG_SDK_UART_DATA_RX:
        {
            AMP_SYS_LOG("[Event] UART Data RX Signal Received.");
            on_uart_data_rx(msg);
            break;
        }

    default:
        AMP_SYS_LOG("Received unknown signal ID: %d", msg->id);
        break;
    }
}

// ==========================================
// 3. Application Main Task
// ==========================================

void vAppMainTask(void *pvParameters)
{
    sdk_signal_t msg;
    amp_network_status_t net_status = {0};
    
    // 1. Create the Queue to receive signals
    // The depth (20) allows buffering multiple bursts of events.
    QueueHandle_t xAppQueue = xQueueCreate(20, sizeof(sdk_signal_t));
    if (xAppQueue == NULL) {
        AMP_SYS_LOG("Error: Failed to create App Queue.");
        return;
    }

    AMP_SYS_LOG("App Queue initialized. Subscribing to SDK signals...");

    // 2. Subscribe to signals using the SDK API
    // SDK_SUB_MODE_REPEAT: We receive the signal, but system default handlers also get it.

    // -> System Init Done
    wlan_sdk_subscribe(SIG_SDK_SYS_INIT_DONE, xAppQueue, SDK_SUB_MODE_REPEAT);

    // -> WiFi Join Success
    wlan_sdk_subscribe(SIG_SDK_JOIN_SUCESS, xAppQueue, SDK_SUB_MODE_REPEAT);

    // -> WiFi Join Fail
    wlan_sdk_subscribe(SIG_SDK_JOIN_FAILED, xAppQueue, SDK_SUB_MODE_REPEAT);
    
    // -> UART RX
    //wlan_sdk_subscribe(SIG_SDK_UART_DATA_RX, xAppQueue, SDK_SUB_MODE_REPEAT);

    // -> WiFi Connect
    wlan_sdk_subscribe(SIG_SDK_WLAN_CONNECTED, xAppQueue, SDK_SUB_MODE_REPEAT);
    
    // -> WiFi Disconnect
    wlan_sdk_subscribe(SIG_SDK_WLAN_DISCONNECTED, xAppQueue, SDK_SUB_MODE_REPEAT);

    // -> IP Changed
    wlan_sdk_subscribe(SIG_SDK_WLAN_IP_CHANGED, xAppQueue, SDK_SUB_MODE_REPEAT);

    // -> MQTT Connect OK
    wlan_sdk_subscribe(SIG_SDK_MQTT_CONNECT_OK, xAppQueue, SDK_SUB_MODE_REPEAT);
    
    // -> MQTT Disconnect
    wlan_sdk_subscribe(SIG_SDK_MQTT_DISCONNECT, xAppQueue, SDK_SUB_MODE_REPEAT);

    // -> MQTT Data Down
    wlan_sdk_subscribe(SIG_SDK_MQTT_DATA_DOWN, xAppQueue, SDK_SUB_MODE_REPEAT);

    AMP_SYS_LOG("Subscriptions active. Entering Event Loop.");

    // 3. Event Loop
    for (;;) {
        // Block indefinitely until a signal arrives
        if (xQueueReceive(xAppQueue, &msg, portMAX_DELAY) == pdPASS) {

            // Process the received signal
            msg_processed_for_test(&msg, &net_status);

            /*
             * warnning:
             * repeat mode, DO NOT free msg.data here
             */
            //if (msg.data) {
            //    // Free the data buffer if allocated
            //    os_free(msg.data);
            //}
        }
    }
}

#endif // AMP_SDK_EXAMPLE
