#ifndef __MQTT_WRAPPER_H
#define __MQTT_WRAPPER_H

#include "ach118x.h"
#include "wlan_def.h"

/**
 * @brief Configures the MQTT Broker IP address or Domain name.
 * @param ip_or_domain: Pointer to null-terminated string (Max 128 chars).
 * @return NO_ERR on success, PARA_ERR or VALUE_ERR on failure.
 */
enum result_type wlan_mqtt_set_server_ip(const char *ip_or_domain);

/**
 * @brief Sets the MQTT Broker port.
 * @param port: Port number (1-65535). Default is 1883.
 * @return NO_ERR on success, PARA_ERR on invalid port.
 */
enum result_type wlan_mqtt_set_server_port(int port);

/**
 * @brief Sets the username for MQTT authentication.
 * @param username: Pointer to username string (Max 64 chars).
 * @return NO_ERR on success, error code otherwise.
 */
enum result_type wlan_mqtt_set_username(const char *username);

/**
 * @brief Sets the password for MQTT authentication.
 * @param password: Pointer to password string (Max 64 chars).
 * @return NO_ERR on success, error code otherwise.
 */
enum result_type wlan_mqtt_set_password(const char *password);

/**
 * @brief Sets the default topic for publishing messages.
 * @param topic: Pointer to topic string (Max 20 chars).
 * @return NO_ERR on success, PARA_ERR if too long.
 */
enum result_type wlan_mqtt_set_pub_topic(const char *topic);

/**
 * @brief Sets the topic to subscribe to upon connection.
 * @note Current SDK supports only one active subscription topic.
 * @param topic: Pointer to topic string (Max 20 chars).
 * @return NO_ERR on success, PARA_ERR if too long.
 */
enum result_type wlan_mqtt_set_sub_topic(const char *topic);

/**
 * @brief Configures the Quality of Service (QoS) level.
 * @param qos: QoS Level: 0 (At most once), 1 (At least once), 2 (Exactly once).
 * @return NO_ERR on success, PARA_ERR if invalid.
 */
enum result_type wlan_mqtt_set_qos(int qos);

/**
 * @brief Selects the MQTT authentication method.
 * @param type: 0: None, 1: User/Pass, 2: Server Auth, 4: Mutual Auth.
 * @return NO_ERR on success, PARA_ERR if invalid.
 */
enum result_type wlan_mqtt_set_auth_type(int type);

/**
 * @brief Sets the filename of the CA certificate for SSL/TLS.
 * @note File must be pre-uploaded to the module filesystem.
 * @param filename: Name of the file in the filesystem (Max 64 chars).
 * @return NO_ERR on success.
 */
enum result_type wlan_mqtt_set_ca_cert(const char *filename);

/**
 * @brief Sets the filename of the Client certificate.
 * @note File must be pre-uploaded to the module filesystem.
 * @param filename: Name of the file in the filesystem (Max 64 chars).
 * @return NO_ERR on success.
 */
enum result_type wlan_mqtt_set_client_cert(const char *filename);

/**
 * @brief Sets the filename of the Client private key.
 * @note File must be pre-uploaded to the module filesystem.
 * @param filename: Name of the file in the filesystem (Max 64 chars).
 * @return NO_ERR on success.
 */
enum result_type wlan_mqtt_set_client_key(const char *filename);

/**
 * @brief Initiates an asynchronous connection to the MQTT broker.
 * @note Precondition: Wi-Fi must be connected and MQTT params set.
 * @return NO_ERR (Always returns 0). Check logs for connection result.
 */
enum result_type wlan_mqtt_connect(void);

/**
 * @brief Terminates the MQTT session.
 * @return NO_ERR (Always returns 0).
 */
enum result_type wlan_mqtt_disconnect(void);

/**
 * @brief Publishes a message to a specific topic.
 * @param topic: Target topic (Max 20 chars).
 * @param message: Payload data.
 * @param len: Payload length in bytes.
 * @return NO_ERR on success, SEND_ERR on failure.
 */
enum result_type wlan_mqtt_publish(const char *topic, const char *message, int len);

/**
 * @brief Subscribes to a specific topic.
 * @param topic: Target topic (Max 20 chars).
 * @param rfu: Reserved for Future Use (Pass 0).
 * @return NO_ERR on success, SEND_ERR on failure.
 */
enum result_type wlan_mqtt_subscribe(const char *topic, int rfu);

/**
 * @brief Unsubscribes from a specific topic.
 * @param topic: Target topic.
 * @return NO_ERR on success, SEND_ERR on failure.
 */
enum result_type wlan_mqtt_unsubscribe(const char *topic);

#endif // __MQTT_WRAPPER_H
